/*
 * Copyright (c) 2014 Daniel Kang
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <omp.h>
#include <boost/program_options.hpp>

#include <string>

#include "../consts.h"
#include "../util/util.h"

using boost::program_options::variables_map;

#ifndef CLASSES_LOADER_H_
#define CLASSES_LOADER_H_

class Loader {
 protected:
  const int NUM_THREADS;
  const int64_t start, end;
  const gtype *genome;
  const ftype *counts;

  // Don't do anything
  Loader(const int64_t s, const int64_t e) :
      NUM_THREADS(omp_get_max_threads()),
      start(s), end(e), genome(NULL), counts(NULL) {}

  // Only load genome
  Loader(const int64_t N, const int64_t offset, const std::string& genome_fname) :
      NUM_THREADS(omp_get_max_threads()),
      start(0), end(N), genome(load_genome(end - start, KBIG, offset, genome_fname.c_str())),
      counts(NULL) {}

  // Normal constructor
  Loader(const int64_t N, const variables_map& vm) :
      NUM_THREADS(omp_get_max_threads()),
      start(0), end(N),
      genome(load_genome(end - start, KBIG, 0, vm["genome"].as<std::string>().c_str())),
      counts(load_counts(end - start, vm["read_max"].as<int64_t>(),
                         vm["smooth_window_size"].as<int64_t>(),
                         vm["reads"].as<std::string>().c_str())) {}

  // Heldout constructor
  explicit Loader(const variables_map& vm) :
      NUM_THREADS(omp_get_max_threads()),
      start(0), end(vm["heldout_size"].as<int64_t>()),
      genome(load_genome(end - start, KBIG, vm["heldout_start"].as<int64_t>(),
                         vm["genome"].as<std::string>().c_str())),
      counts(load_counts(end - start, vm["read_max"].as<int64_t>(),
                         vm["smooth_window_size"].as<int64_t>(),
                         vm["heldout_reads"].as<std::string>().c_str())) {}

 public:
  static gtype* load_genome(const int64_t SIZE, const int KS, const int64_t offset,
                            const char *fname);

  static ftype* load_counts(const int64_t SIZE, const int max_reads, const int window_size,
                            const char *fname);

  virtual ~Loader() {
    if (genome != NULL) free((gtype*) genome);
    if (counts != NULL) free((ftype*) counts);
  }

  Loader(const Loader&) = delete;
  Loader& operator=(const Loader&) = delete;
};

#endif  // CLASSES_LOADER_H_
