# contactBias.py
# Jonathan M. Galazka, Andrew D. Klocko
# 2015
# Usage: python ./contactBias.py
#
# Calculates and plots the 'bias' that a given locus has for interactions with upstream regions
# (towards the left telomere) or downstream regions (toward the right telomere).

import numpy as np
import matplotlib.pyplot as plt

datasets = ['NMF39','N3944','N5580']
resolution = 10000

lg = 1 # select chromosome to plot

# get chromosomes starts and ends
res_string = str(resolution)
chr_starts_path = '/Volumes/HD/HiC2/NMF39/NMF39_ic-' + res_string + '-diag-txt/ChromosomeStarts'
chr_starts_array = np.loadtxt(chr_starts_path, delimiter=' ')
chr_starts = np.transpose(chr_starts_array.astype(int)[0:7])
chr_ends = np.transpose(chr_starts_array.astype(int)[1:8])

# loop through datasets
for dataset in datasets:

	dataset_path = '/Volumes/HD/HiC2/' + dataset + '/' + dataset + '_ic-' + res_string + '-diag-txt/heatmap'	
	array = np.loadtxt(dataset_path, delimiter=' ')

	lg_array = array[chr_starts[lg]:chr_ends[lg],chr_starts[lg]:chr_ends[lg]] # get chromosome array
	lg_shape, lg_shape = lg_array.shape # get dimensions of chrom array
	
	mb_bins = 300000 / resolution # calculate the number of bins to move 0.3 mb up or downstream

	# plot chromosome hic heatmap
	f2, ax2 = plt.subplots(1)
	heatmap = ax2.imshow(np.log2(lg_array), interpolation='nearest', cmap='coolwarm')
	heatmap.set_clim(0,10)
	f2.colorbar(heatmap)
	f2.savefig('/Users/galazkaj/Desktop/direction_index_heatmap_' + dataset + '.pdf')
	
	# make a list called indexes to hold directional index values
	# initialize the first 300000/resolution with zeros
	indexes = list()
	for k in range(0, mb_bins):
		indexes.append(0)

	# calculate directional index values
	for i in range(mb_bins, lg_shape - mb_bins):

		# calculate total interactions 300 kb up
		mb_up_total = 0
		for j in range(1, mb_bins):
			mb_up = lg_array[i, i+j]
			mb_up_total = mb_up_total + mb_up
	
		# calculate total interactions 300 kb down
		mb_down_total = 0
		for j in range(1, mb_bins):
			mb_down = lg_array[i, i-j]
			mb_down_total = mb_down_total + mb_down
		
		# calculate directional index
		mb_total =  mb_up_total + mb_down_total
		up_frac = mb_up_total / mb_total
		down_frac = mb_down_total / mb_total
		direction_index = np.log2(up_frac / down_frac)
	
		indexes.append(direction_index)
	
	# initialize the last 300000/resolution with zeros
	for k in range(0, mb_bins):
		indexes.append(0)
	
	# plot directional indexes
	f1, ax1 = plt.subplots(1)
	indexes_array = np.array(indexes)	
	x = range(0, len(indexes))
	ax1.plot(x, indexes_array, color='black')
	ax1.autoscale(enable=True, axis='x', tight=True)
	ax1.set_ylim(-2.5,2.5)
	ax1.fill_between(x, 0, indexes_array, where=indexes_array>0, facecolor='blue', interpolate=True)
	ax1.fill_between(x, 0, indexes_array, where=indexes_array<0, facecolor='red', interpolate=True)
	
	
	f1.savefig('/Users/galazkaj/Desktop/direction_index_' + dataset + '.pdf')

# plot matching K9me27 and K9me3 tracks	
k9_path = '/Volumes/HD/HiC2/ChIP_tracks/h3k9me3_3/h3k9me3_3_chip_lg_' + str(lg) + '_20k.txt'
k27_path = '/Volumes/HD/HiC2/ChIP_tracks/h3k27me3/h3k27me3_chip_lg_' + str(lg) + '_20k.txt'

k9_array = np.transpose(np.loadtxt(k9_path))
k27_array = np.transpose(np.loadtxt(k27_path))

f3, ax3 = plt.subplots(1, figsize=(10,2))
x = range(0, k9_array.size)
ax3.bar(x, k9_array, color='black', edgecolor='black')
ax3.autoscale(enable=True, axis='x', tight=True)
f3.savefig('/Users/galazkaj/Desktop/k9_track_' + dataset + '.pdf')

f4, ax4 = plt.subplots(1)
x = range(0, k27_array.size)
ax4.bar(x, k27_array, color='black', edgecolor='black')
ax4.autoscale(enable=True, axis='x', tight=True)
f4.savefig('/Users/galazkaj/Desktop/k27_track_' + dataset + '.pdf')

plt.show()











