function [coupling, fva, beta] = Coupling(name, varargin)
% Calculate blocked reactions, full, partial, and directional coupling for
% the given network with 'name', stoichiometric matrix 'S', 'Reactions',
% and 'Metabolites'.
debug = false;

    try 
		
    if (debug)
        fprintf(strcat(datestr(now), '. PID: %d.\n'), sscanf(evalc('feature getpid'), '%*s%*s%d'));
    end
    
    epsilon = 1; % minimum flux of at least one exchange reaction
    doEssential = 0; % 1 = calculate essential reactions
	doAnti = 0; % 1 = calculate anti-coupling
	doInhibitors = 0; % 1 = calculate inhibitive coupling
    
    lpmethod = 1;
	scaind = 1;
    optimize = 0;
	
	% parse the arguments
	arg = 1;
    while (arg+1 <= nargin)
        if (strcmp(inputname(arg+1), 'lpmethod'))
            lpmethod = varargin{arg};
        elseif (strcmp(inputname(arg+1), 'S'))
            S = varargin{arg};
        elseif (strcmp(inputname(arg+1), 'rev'))
            rev = varargin{arg};
        elseif (strcmp(inputname(arg+1), 'fctable'))
            fctable = varargin{arg};
        elseif (strcmp(inputname(arg+1), 'blocked'))
            blocked = varargin{arg};
        elseif (strcmp(inputname(arg+1), 'solver'))
            solver = varargin{arg};            
        elseif (ischar(varargin{arg}))
            if (strcmp(inputname(arg+1), 'fva'))
				fva = varargin{arg};
			elseif (strcmp(inputname(arg+1), ''))
                if (strcmp(varargin{arg}, 'opt'))
                    optimize = 1;
                elseif (strcmp(varargin{arg}, 'essential'))
                    doEssential = 1;
                elseif (strcmp(varargin{arg}, 'anti'))
                    doAnti = 1;
                elseif (strcmp(varargin{arg}, 'inhibitor'))
                    doInhibitors = 1;                    
                else
                    from_to = sscanf(varargin{arg}', '%d-%d');
                    if (size(from_to) == [2 1])
                        from = from_to(1);
                        to = from_to(2);
                    end
                end
            end
        end
        arg = arg+1;
    end
    
    % load the data not provided as arguments
    % NOTE: if S or rev are passed as arguments then they must not contain
    % any blocked reactions!
    if (~exist('blocked', 'var'))
        % only if blocked is not passed we modify S and rev
        blocked = load(strcat(name, '.blocked'));
    end
    if (~exist('S', 'var'))
        S = load(strcat(name, '.S'));
        S = S(:,~blocked);
    end
    if (~exist('rev', 'var'))
        rev = load(strcat(name, '.rev'));
        rev = rev(~blocked,:);
    end
    if (~exist('fctable', 'var'))
        % load fctable/coupling
        if (doEssential || doAnti)
            fctable = load(strcat(name, '.fctable'));
        else
            fctable = load(strcat(name, '.anticoupling'));
        end
    end
	if (~exist('solver', 'var'))
        solver = 'cplex';
	end
    
    % check for dicrepancy between S and blocked vector
    if (size(S, 2)+length(find(blocked)) ~= length(blocked))
        error('Length of blocked (%d) must match columns in S (%d) + blocked reactions (%d).', length(blocked), size(S, 2), length(find(blocked)));
    end
    % check for discrepancy between S and coupling table
	if (size(S, 2) ~= size(fctable))
        error('Columns in S (%d) must match dimension of coupling table (%dx%d).', size(S, 2), size(fctable, 1), size(fctable, 2));
	end

	% determine the dimensions
    m = size(S, 1);
	n = size(S, 2);
    uncoupled = zeros(n, n);
    if ((exist('from', 'var') || exist('to', 'var')) && ~doInhibitors)
        error('From-to range implemented only for inhibitor coupling.');
    end
    if (~exist('from', 'var'))
        from = 1;
    end
    if (~exist('to', 'var'))
        to = n;
    end
    
    % copy full, partial, and directional coupling from fctable to coupling
    coupling = fctable;
    
	evalc('run initTomlab');
	
	% specify lower and upper bounds according to reversibility
	% here we may use the upper and lower bounds (B) provided by the model
	lowerBound = zeros(n,1);
	lowerBound(find(rev)) = -1000;
	upperBound = repmat(1000, n, 1);
    maxBound = max(upperBound);
	
	options = struct;
	options.name = name;
	options.solver = solver;
	
	% vector pointing at the exchange reactions, 
	% i.e., columns having either only positive or only negative coefficients
	% ideally, one would take mass unbalanced reactions as exchange reactions
    import = (sum(abs(S), 1) == sum(S, 1));
    export = (sum(abs(S), 1) == -sum(S, 1));
    numReversibleImport = length(find(import.*rev'));
    numReversibleExport = length(find(export.*rev'));
    % check if there is at least one import
    if (length(find(import)) + numReversibleExport < 1)
        fprintf('ERROR: %s has %d unblocked import/reversible export reactions.\n', name, length(find(import)) + numReversibleExport);
        coupling = -1;
        return
    end
    % check if there is at least one export
    if (length(find(export)) + numReversibleImport < 1)
        fprintf('ERROR: %s has %d unblocked export/reversible import reactions.\n', name, length(find(export)) + numReversibleImport);
        coupling = -1;
        return
    end
    exchange = import | export;
    numExchange = size(find(exchange), 2);
	if (numExchange ~= 0)
        fprintf('%s has %d unblocked exchange reactions.\n', name, numExchange);
	else
        fprintf('%s has %d unblocked exchange reactions. Using all reactions for non-zero constraint.\n', name, numExchange);
        exchange = 1:n;
        numExchange = size(find(exchange), 2);
	end
	
    if (doAnti || doInhibitors)
        % pre-calculate the weakly connected components (speedup for most
        % large networks)
        t3 = tic;
        A = stoich2Adjacency(S, rev);
        [num, C] = graphconncomp(sparse(A), 'Directed', 1, 'Weak', 1);
        t3 = toc(t3);
        fprintf('%d weakly connected components, largest has %g%% vertices. Time: %g s.\n', num, length(find(C==mode(C)))*100/size(A, 1), t3);
    end
    
	essential = 0;
	anticoupled = 0;
	inhibitors = 0;
	inhibitor_conflicts = 0;
    t1 = tic;
    
    if (doEssential)
        % ESSENTIAL REACTIONS: v_i==0 => v==0
        t2 = tic;
		% decision variables for essential reactions
		v = tom('v', n, 1);
        x = tom('x', numExchange, 1, 'integer');
        v_forward_ex = tom('v_forward_ex', numExchange, 1);
        v_reversed_ex = tom('v_reversed_ex', numExchange, 1);
        x_forward_ex = tom('x_forward_ex', numExchange, 1, 'integer');
        x_reversed_ex = tom('x_reversed_ex', numExchange, 1, 'integer');
        % initial setup of basic constraints
		constraints = { ...
			S*v == 0;
			lowerBound <= v <= upperBound;
            v(exchange) == v_forward_ex - v_reversed_ex;
            0 <= v_forward_ex <= x_forward_ex*maxBound;
            0 <= v_reversed_ex <= x_reversed_ex*maxBound;
            x_forward_ex + x_reversed_ex <= 1;
            epsilon*x <= v_forward_ex + v_reversed_ex <= maxBound*x;
            sum(x) >= 1;
        };
		Prob = sym2prob('mip', [], constraints, [], options);
		Prob.optParam.IterPrint = 0; % Set to 1 to see iterations.
		Prob.MIP.cpxControl.LPMETHOD = lpmethod;
        Prob.MIP.cpxControl.SCAIND = scaind;
        
        for i=1:n
			% set v_i to zero
			x_L = lowerBound;
			x_U = upperBound;
			x_L(i) = 0;
			x_U(i) = 0;
			Prob = modify_x_L(Prob, x_L, 1:n);
			Prob = modify_x_U(Prob, x_U, 1:n);
            
            tomrun = tomRun(solver, Prob, 0);
			if (tomrun.ExitFlag == 4)
                % possible numerical error. try default solver algorithm
                % and normal scaling.
                Prob.MIP.cpxControl.LPMETHOD = 0;
                Prob.MIP.cpxControl.SCAIND = 0;
                tomrun = tomRun(solver, Prob, 0);
                if (tomrun.ExitFlag == 4)
                    % set the diagonal of the essential reaction to 7
                    coupling(i,i) = 7;
                    essential = essential + 1;
                else
                    fprintf('Fixed numerical error while calculating essentiality of reaction %d using default algorithm and scaling.\n', i);
                end
                Prob.MIP.cpxControl.LPMETHOD = lpmethod;
                Prob.MIP.cpxControl.SCAIND = scaind;
			elseif (tomrun.ExitFlag ~= 0)
				save(strcat('Error015_', datestr(now, 'yyyy-mm-dd_HHMM'), '.mat'));
				error('tomRun error while calculating essentiality of reaction %d.\nExitFlag: %d. %s\n', i, tomrun.ExitFlag, tomrun.ExitText);
			end
        end
        t2 = toc(t2);
        fprintf('Network has %d essential reactions. Time: %g s.\n', essential, t2);
    end
    
    if (doAnti)
		% decision variables for anti coupling
		v = tom('v', n, 1);
        x = tom('x', numExchange, 1, 'integer');
        v_forward_ex = tom('v_forward_ex', numExchange, 1);
        v_reversed_ex = tom('v_reversed_ex', numExchange, 1);
        x_forward_ex = tom('x_forward_ex', numExchange, 1, 'integer');
        x_reversed_ex = tom('x_reversed_ex', numExchange, 1, 'integer');
        % initial setup of basic constraints
		constraints = { ...
			S*v == 0;
			lowerBound <= v <= upperBound;
            v(exchange) == v_forward_ex - v_reversed_ex;
            0 <= v_forward_ex <= x_forward_ex*maxBound;
            0 <= v_reversed_ex <= x_reversed_ex*maxBound;
            x_forward_ex + x_reversed_ex <= 1;
            epsilon*x <= v_forward_ex + v_reversed_ex <= maxBound*x;
            sum(x) >= 1;
        };
		Prob = sym2prob('mip', [], constraints, [], options);
		Prob.optParam.IterPrint = 0; % Set to 1 to see iterations.
        Prob.MIP.cpxControl.LPMETHOD = lpmethod;
        Prob.MIP.cpxControl.SCAIND = scaind;
        
        charCount = 0;
        for i=1:(n-1)
            % print i as progress output
            if (charCount > 0)
                for k=1:charCount
                    fprintf('\b');
                end
                pause(.02);
                charCount = 0;
            end
            message = sprintf('Finding anti-coupled pairs for reaction %d of %d.', i, n);
            fprintf(message);
            pause(.02);
            charCount = charCount+length(message);
            
            for j=(i+1):to
                % skip reaction pairs with existing coupling, essential
                % reactions, and from different weakly connected components
                if (uncoupled(i,j) == 0 && uncoupled(j,i) == 0 && coupling(i, i) ~= 7 && coupling(j, j) ~= 7 && C(i+m) == C(j+m))
                    % calculate anti coupling: v(i)==0 => v(j)~=0, which prohibits v(i)==0 &&
                    % v(j)==0
					x_L = lowerBound;
					x_U = upperBound;
					x_L(i) = 0;
					x_U(i) = 0;
					x_L(j) = 0;
					x_U(j) = 0;
					Prob = modify_x_L(Prob, x_L, 1:n);
					Prob = modify_x_U(Prob, x_U, 1:n);
                    tomrun = tomRun(solver, Prob, 0);
                    if (tomrun.ExitFlag == 4)
                        % possible numerical error. try default solver algorithm
                        % and normal scaling.
                        Prob.MIP.cpxControl.LPMETHOD = 0;
                        Prob.MIP.cpxControl.SCAIND = 0;
                        tomrun = tomRun(solver, Prob, 0);
                        if (tomrun.ExitFlag == 4)
                             % anti coupling is symmetric
                            coupling(i,j) = 5;
                            coupling(j,i) = 5;
                            anticoupled = anticoupled+2;
                        else
                            fprintf(' Fixed numerical error while calculating anti-coupling of reactions (%d,%d) using default algorithm and scaling.\n', i, j);
                            charCount = 0;
                        end
                        Prob.MIP.cpxControl.LPMETHOD = lpmethod;
                        Prob.MIP.cpxControl.SCAIND = scaind;
                    end
                    if (tomrun.ExitFlag == 0)
                        % store the indices of reaction pairs with zero flux
                        solution = getSolution(tomrun);
                        zeroV = find(solution.v == 0);
                        % get all combinations of zero flux indices
						zeroIndices = zeros((length(zeroV)*(length(zeroV)-1))/2, 2);
                        a = 1;
						for k=1:length(zeroV)-1
                            zeroIndices(a:a+length(zeroV)-(k+1), 1) = zeroV(k);
                            % could try replacing the inner for-loop by the
                            % remaining sub-vector from notInhibitor
							for l=(k+1):length(zeroV)
								zeroIndices(a, 2) = zeroV(l);
                                a = a + 1;
							end
						end
                        matrixIndices = sub2ind([n n], zeroIndices(:,1)', zeroIndices(:,2)');
                        matrixIndices2 = sub2ind([n n], zeroIndices(:,2)', zeroIndices(:,1)');						
                        % set uncoupled=1 for all combinations of zero flux indices
                        uncoupled(matrixIndices) = 1;
						uncoupled(matrixIndices2) = 1;
                    elseif (tomrun.ExitFlag ~= 4)
						save(strcat('Error014_', datestr(now, 'yyyy-mm-dd_HHMM'), '.mat'));
                        error('tomRun error while calculating reactions (%d,%d).\nExitFlag: %d. %s\n', i, j, tomrun.ExitFlag, tomrun.ExitText);
                    end
                end
            end
			% write the temporary results every 100 reactions
%             if (mod(num, 100) == 0)
%                 dlmwrite(strcat(name, '.', int2str(from), '-', int2str(to), '.anticoupling'), coupling, '\t');
%             end
        end
        
        fprintf('\nAnti-coupled pairs: %d.\n', anticoupled);
        dlmwrite(strcat(name, '.anticoupling'), coupling, '\t');
    end
    
    if (doInhibitors)
        
        clear FBAObject;
        % split reversible reactions
        rev = (lowerBound<0);
        [S_split, rev_split] = splitRev(S, rev, -1);
        n_split = n+length(find(rev));
        % determine from:to in the split matrix
        from_split = splitIndex(from, rev);
        to_split = splitIndex(to, rev);
        if (rev_split(to_split))
            to_split = to_split + 1;
        end
%         reversible = rev_split + [0; rev_split(1:n_split-1)];
        B = repmat([0; 1000], 1, n_split);
        FBAO = FBAObject(name, S_split, B);
        % we need the maxima of all fluxes, not only from:to
        if (~exist('fva', 'var'))
            fva = FVA(FBAO, 'max', rev, rev_split, solver);
        end
        uncoupled = false(n_split, n_split);
        charCount = 0;
        upperBound = 1000;
        num = 0;
        
        % define the basic problem once, specify it later
        % the exchange constraint is not necessary here, as futile cycles
        % also lead to larger maxima when they are added to a flux
        % distribution involving exchange reactions.
        % this makes perfect biological sense: metabolites are frequently
        % converted back to their substrates, even at equillibrium, when
        % only the net flux is zero, but both directions run.
        v = tom('v', n_split, 1);
        objective = [];
        if (optimize)
            beta = ones(n_split, n_split);
            c = zeros(n_split, 1);
            objective = c'*v;
        end
		constraints = { ...
			S_split*v == 0; % v(1) >= 1;
			0 <= v <= upperBound; %sum(v(exchange)) >= epsilon;
		};
		Prob = sym2prob('lp', objective, constraints, [], options);
        Prob.MIP.cpxControl.LPMETHOD = 1;
        Prob.MIP.cpxControl.SCAIND=1;
        
        % speed-up: pre-calculate the unsplit indices
        unsplitIndices = zeros(n_split, 1);
        for i=1:n_split
            unsplitIndices(i) = unsplitIndex(i, rev);
        end
        
        for i=from_split:to_split
			% determine i in the unsplit matrix for accessing coupling
			i_unsplit = unsplitIndices(i);
            % print i as progress output
            num = num + 1;
            if (charCount > 0)
                for k=1:charCount
                    fprintf('\b');
                end
                pause(.02);
                charCount = 0;
            end
            message = sprintf('Finding inhibitor targets for flux %d (%d of %d).', i, num, (to_split-from_split+1));
            fprintf(message);
            pause(.02);
            charCount = charCount+length(message);
%             previousUncoupled = length(find(uncoupled));
            
            for j=1:n_split
                % determine j in the unsplit matrix for accessing coupling
                j_unsplit = unsplitIndices(j);
                
                % skip reactions with v_j_max = 0, fully coupled pairs, and
                % reactions from different weakly connected components
                if (fva(i) <= 0 || fva(j) <= 0 || coupling(i_unsplit, j_unsplit) == 1 || C(i_unsplit+m) ~= C(j_unsplit+m))
                    % set beta to 1 and leave coupling unchanged
                    if (optimize)
                        beta(i,j) = 1;
                    end
                % skip indices corresponding to the same reaction, essential reactions as target (j), and known uncoupled pairs
                elseif (i_unsplit ~= j_unsplit && coupling(j_unsplit, j_unsplit) ~= 7 && (optimize || (uncoupled(i,j) == 0)))
                    if (optimize)
                        % maximize v_j with v_i = v_i_max
                        c = zeros(n_split, 1);
                        c(j) = -1;
                        Prob = modify_c(Prob, c);
                    end
                    % set v_i to v_i_max
                    % replace bounds x_L and x_U to specify v(i) == fva(i)
                    x_L = zeros(n_split, 1);
                    x_U = repmat(upperBound, n_split, 1);
                    % reset the lower bounds first to avoid conflict with upper bounds
                    Prob = modify_x_L(Prob, x_L);
                    if (n_split >= 50)
                        % large networks require a numerical tolerance
                        i_tol = 0.99;
                        j_tol = 99.99;
                    else
                        i_tol = 1;
                        j_tol = 100;
                    end
                    x_L(i) = fva(i) * i_tol; % 1% tolerance
                    x_U(i) = upperBound;
                    if (~optimize)
                        % set lower bound of v(j) to v_j_max
                        x_L(j) = fva(j)/j_tol;
                    end
                    % set reversed of i and j to 0 to avoid reversible futile cycle
                    if (rev_split(i))
                        x_U(i+1) = 0;
                    elseif (i > 1 && rev_split(i-1))
                        x_U(i-1) = 0;
                    end
                    if (rev_split(j))
                        x_U(j+1) = 0;
                    elseif (j > 1 && rev_split(j-1))
                        x_U(j-1) = 0;
                    end
					if (find(x_U < x_L))
						save(strcat('Error006_', datestr(now, 'yyyy-mm-dd_HHMM'), '.mat'));
						crossover = find(x_U < x_L);
						error('x_L and x_U have crossover values: x_L(%d)=%d, x_U(%d)=%d.', crossover(1), x_L(crossover(1)), crossover(1), x_U(crossover(1)));
					end
                    Prob = modify_x_U(Prob, x_U);
                    Prob = modify_x_L(Prob, x_L);
                    tomrun = tomRun(solver, Prob, 0);
                    
                    % success means we have a solution for optimization
                    success = (optimize && tomrun.ExitFlag == 0);
                    
                    if (optimize && tomrun.ExitFlag == 4)
                        % possible numerical error, try automatic solver algorithm
						Prob.MIP.cpxControl.LPMETHOD = 0;
                        tomrun = tomRun(solver, Prob, 0);
						if (tomrun.ExitFlag == 0)
                            success = 1;
						else
							% try normal scaling
							Prob.MIP.cpxControl.SCAIND = 0;
							tomrun = tomRun(solver, Prob, 0);
							if (tomrun.ExitFlag == 0)
								success = 1;
							else
								% try no scaling
								Prob.MIP.cpxControl.SCAIND = -1;
								tomrun = tomRun(solver, Prob, 0);
								if (tomrun.ExitFlag == 0)
									success = 1;
								end
							end
						end
						Prob.MIP.cpxControl.LPMETHOD = lpmethod;
						Prob.MIP.cpxControl.SCAIND = scaind;
						
                        if (~success)
							% full inhibition: may happen if the other direction of
                            % a reversible reaction, which was set to 0, is essential
							beta(i, j) = 0;
                        end
                    end
                    
					if (optimize && success)
                        % store the minimal beta: maximal v_j / v_j_max
                        solution = getSolution(tomrun);
                        % fix numerical imprecisions
                        % for the cases v_j<=-0.0001, v_j>fva(j)+0.0001, and v_j>upperBound+0.0001
                        % the calculation could be repeated with different scaling/algorithm
                        if (solution.v(j) <= -0.0001)
                            fprintf(' Maximum flux of %d (reaction %d) is %d with flux %d=%d (reaction %d).\n', j, j_unsplit, solution.v(j), i, fva(i), i_unsplit);
                            charCount = 0;
                            beta(i, j) = 1;
                        elseif (solution.v(j) < 0)
                                beta(i, j) = 0;
                        elseif (solution.v(j) > fva(j))
                            if (solution.v(j) > fva(j) + 0.0001)
                                fprintf(' FVA maximum of %d (reaction %d) is %d, maximum with flux %d=%d (reaction %d) is %d.\n', j, j_unsplit, fva(j), i, fva(i), i_unsplit, solution.v(j));
                                charCount = 0;
                            end
                            beta(i, j) = 1;
                        elseif (solution.v(j) > upperBound)
                            if (solution.v(j) > upperBound + 0.0001)
                                fprintf(' Maximum larger than upper bound for flux %d (reaction %d): %d, ub=%d.\n', j, rev_split(j), solution.v(j), upperBound);
                                charCount = 0;
                            end
                            beta(i, j) = 1;
                        else
                            beta(i, j) = solution.v(j)/fva(j);
						end
					end
                    
                    % set the coupling value with cut-off 1%
                    if ((optimize && beta(i,j) <= 0.01) || (~optimize && tomrun.ExitFlag == 4))
                        % inhibitor coupling is NOT symmetric!
                        % use a 7-bit representation to store all
                        % possible combinations of coupling:
                        % dec   bits   description
                        %  0   0000000 no coupling
                        %  1   0000001 full (unused)
                        %  2   0000010 partial
                        %  3   0000011 reversed directional
                        %  4   0000100 directional
                        %  5   0000101 anti
                        %  8   0001000 forward-forward
                        % 16   0010000 forward-reversed
                        % 32   0100000 reversed-forward
                        % 64   1000000 reversed-reversed
                        % ...and all combinations thereof involving
                        % inhibitor coupling. The last 2 bits represent the
                        % value of other coupling types. The first 5 bits
                        % represent inhibitor coupling bit-wise.
                        % convert dec-to-bit: bitget(<dec>, 7:-1:1)
                        % convert bit-to-dec: bin2dec(num2str(<bits>))

                        % determine the direction of v_i and v_j and
                        % corresponding bit representation
                        if (i > 1 && rev_split(i-1))
                            if (j > 1 && rev_split(j-1))
                                % i and j are reversed
                                summand = 64;
                            else
                                % i reversed, j forward
                                summand = 32;
                            end
                        else
                            if (j > 1 && rev_split(j-1))
                                % i forward, j reversed
                                summand = 16;
                            else
                                % i and j are forward
                                summand = 8;
                            end
                        end

                        value = coupling(i_unsplit, j_unsplit);
                        if (value > 0 && value < 6)
                            % conflict with other coupling type (2-5)
%                             fprintf(' Inhibitor conflict at [%d,%d]=%d. New value: %d = %d+%d (7-bit).\n', i_unsplit, j_unsplit, value, value+summand, value, summand);
%                             charCount = 0;
                            inhibitor_conflicts = inhibitor_conflicts+1;
                        end
                        coupling(i_unsplit,j_unsplit) = value+summand;
                        inhibitors = inhibitors+1;
                    end
                    
                    if (~optimize && tomrun.ExitFlag == 0)
                        % store the indices of reaction pairs with maximum flux
                        solution = getSolution(tomrun);
                        
                        % remove reversible futile cycles by subtracting the smaller
                        v_net = solution.v;
                        for k=1:n_split-1
                            if (rev_split(k))
                                if (v_net(k) > v_net(k+1))
                                    v_net(k) = v_net(k) - v_net(k+1);
                                    v_net(k+1) = 0;
                                else
                                    v_net(k+1) = v_net(k+1) - v_net(k);
                                    v_net(k) = 0;
                                end
                            end
                        end
                        
                        % determine reactions with maximum flux (1% tolerance)
                        notInhibitor = find(v_net >= fva'); %-(fva'./100));
                        % get all ordered combinations of indices with maximum flux
                        % for-loops are about two times faster than combnk
                        notInhibitorIndices = zeros((length(notInhibitor)*(length(notInhibitor)-1))/2, 2);
                        a = 1;
                        for k=1:length(notInhibitor)-1
                            notInhibitorIndices(a:a+length(notInhibitor)-(k+1), 1) = notInhibitor(k);
                            % could try replacing the inner for-loop by the
                            % remaining sub-vector from notInhibitor
                            for l=(k+1):length(notInhibitor)
                                notInhibitorIndices(a, 2) = notInhibitor(l);
                                a = a + 1;
                            end
                        end
                        matrixIndices = sub2ind([n_split n_split], notInhibitorIndices(:,1)', notInhibitorIndices(:,2)');
                        matrixIndices2 = sub2ind([n_split n_split], notInhibitorIndices(:,2)', notInhibitorIndices(:,1)');
                        % set uncoupled(k,l)=1 for all maximum flux indices k and l
                        uncoupled(matrixIndices) = 1;
                        uncoupled(matrixIndices2) = 1;
                        
                        % find pairs (v_i,v_j) with v_i==v_i_max and v_j>=v_j_max
                        notInhibited = setdiff(find(v_net >= fva'/2), notInhibitor);
                        % get all combinations between v_i and v_j
                        % for-loop with mod-assignment of individual elements is not faster
                        notInhibitedIndices = [repmat(notInhibitor(:),length(notInhibited), 1), repmat(notInhibited(:),length(notInhibitor), 1)];
                        % set uncoupled(i,j)=1 for all indices at least half the max
                        matrixIndices3 = sub2ind([n_split n_split], notInhibitedIndices(:,1)', notInhibitedIndices(:,2)');
                        uncoupled(matrixIndices3) = 1;
                        % free some memory
                        clear solution; clear notInhibitorIndices; clear matrixIndices; clear matrixIndices2;
                        clear notInhibitedIndices; clear matrixIndices3;
                    end
					
                    if (tomrun.ExitFlag ~= 0 && tomrun.ExitFlag ~= 4)
                        save(strcat('Error004_', datestr(now, 'yyyy-mm-dd_HHMM'), '.mat'));
                        error('tomRun error while calculating fluxes (%d,%d), reactions (%d,%d).\nExitFlag: %d. %s\n', i, j, i_unsplit, j_unsplit, tomrun.ExitFlag, tomrun.ExitText);
                    end
                end
            end
%             newUncoupled = length(find(uncoupled));
%             if (newUncoupled > previousUncoupled)
%                 fprintf(' Solutions of flux %d yielded %d additional uncoupled pairs.\n', i, newUncoupled - previousUncoupled);
%                 charCount = 0;
%             end
            % write the temporary results every 100-101 fluxes (finish reversible reactions)
%             if ((mod(num, 100) == 0 && ~rev_split(i)) || (mod(num-1, 100) == 0 && i > 1 && rev_split(i-1)))
%                 dlmwrite(strcat(name, '.', int2str(from), '-', int2str(to), '.coupling'), coupling, '\t');
%                 if (optimize)
%                     dlmwrite(strcat(name, '.', int2str(from), '-', int2str(to), '.beta'), beta, '\t');
%                 end
%             end
        end
        
        fprintf('\nInhibitor coupled flux pairs: %d. Pairs with multiple coupling types: %d.\n', inhibitors, inhibitor_conflicts);
        if (from ~= 1 || to ~= n)
            dlmwrite(strcat(name, '.', int2str(from), '-', int2str(to), '.coupling'), coupling, '\t');
            if (optimize)
                dlmwrite(strcat(name, '.', int2str(from), '-', int2str(to), '.beta'), beta, '\t');
            end
        else
            dlmwrite(strcat(name, '.coupling'), coupling, '\t');
            if (optimize)
                dlmwrite(strcat(name, '.beta'), beta, '\t');
            end
        end        
    end
    
    % print elapsed time
    t1 = toc(t1);
    d = floor(t1/86400);
    h = floor(t1/3600);
    minutes = floor(t1/60);
    s = rem(t1, 60);
    if (d > 0)
        fprintf('Time: %d days, %d hours.\n', d, h-(d*24));
    elseif (h > 0)
        fprintf('Time: %d hours, %d minutes.\n', h, minutes-(h*60));
    else
        fprintf('Time: %d minutes, %g seconds.\n', minutes, s);
    end
    
    catch err
        save(strcat('Error009_', datestr(now, 'yyyy-mm-dd_HHMM'), '.mat'));
        rethrow(err);
    end
end
