#!/usr/bin/env ruby

require 'fileutils'

require 'tempfile'
require 'gene_info'
require 'optparse'
require 'set'
require 'nokogiri'
require 'returning'

def change_extension(filename, newext)
	base = File.basename(filename).partition('.').first
	File.join(File.dirname(filename), base + '.' + newext)
end

def edge_name(from, to)
	(from < to) ? "#{from}_#{to}" : "#{to}_#{from}"
end

def add_edge(edges, from, to, confidence)
	returning edge_name(from, to) do |edge_str|
		edges[edge_str]['source'] = from
		edges[edge_str]['target'] = to
		edges[edge_str]['confidence'] = confidence
	end
end

def add_node(nodes, terminals, anchors, gi, from)
	nodes[from]["type"] = 'terminal' if terminals.include?(from)
	nodes[from]["type"] = 'anchor' if anchors.include?(from)
	if gi.include? from
		nodes[from]["name"] = gi[from].default_id 
	else
		nodes[from]["name"] = from
	end
end

species = nil
alpha = 0.25
OptionParser.new do |opts|
	opts.banner = "Usage: #{File.basename($PROGRAM_NAME)} [OPTIONS] geneset"
	
	opts.separator ''
	opts.separator 'Options:' 

	opts.on( '-s', '--species NAME', String,
	         'Select species (default: human)') do |opt|
		species = opt
	end

	opts.on( '-a', '--alpha alpha', Float,
	         'Set the alpha parameter for ANAT (default: 0.25)') do |opt|
		alpha = opt
	 end

	opts.on( '-h', '--help',
	         'Show this help.') do |opt|
		puts opts
		exit
	end
end.parse!

raise 'Missing gene set filename' if ARGV.empty?

species ||= 'human'
anchors_filename = ARGV.shift
terminals_filename = ARGV.shift
out_filename = ARGV.shift unless ARGV.empty?

SpeciesMap = {
	'human' => GeneInfo::Human
}

NetworkMap = {
	'human' => File.join(File.dirname(__FILE__), '../data_nov_1/H_sapiens_with_y2h.net' )
}

species.downcase!
gi = GeneInfo.for_species(SpeciesMap[species])

# read gene set
def read_gene_set(filename, gi)
	returning Set.new do |geneset|
		lines = 0
		IO.foreach(filename) do |line| 
			line.chomp.split(/\s/).each do |g| 
				lines +=1
				geneset << gi[g].entrez_id if gi.include? g
			end
		end

		$stderr.puts "Mapped #{geneset.size}/#{lines} in #{filename}"
	end
end

terminals = read_gene_set(terminals_filename, gi)
anchors = read_gene_set(anchors_filename, gi)

raise "Empty terminals set" if terminals.empty?
raise "Empty anchors set" if anchors.empty?

# run anat
Steiner = '/home/bnet/atiasnir/anat/steiner/steinprt'

anat_input = Tempfile.new('set')
anat_output = Tempfile.new('net')

nodes = Hash.new { |hash, key| hash[key] = {} }
edges = Hash.new { |hash, key| hash[key] = {} }
begin
	if anchors.size > 1 
		anchor_name = 'ANCHOR'
		network_file= Tempfile.new('ppi')
		anchors.each do |g|
			network_file.puts [anchor_name, g, 1, 0].join("\t")
		end	
		IO.foreach(NetworkMap[species]) do |line| 
			network_file.write line
		end
		network_file.flush
		network_filename = network_file.path
	else
		anchor_name = anchors.first
		network_filename = NetworkMap[species]
	end

	terminals.each {|g| anat_input.puts [anchor_name, g].join("\t") }
	anat_input.flush

	FileUtils.cp anat_input.path, "/vol/scratch/atiasnir/set.txt"

	
	cmdline = "#{Steiner} -f . -n #{network_filename} -c 0 -b #{alpha} -s #{anat_input.path} -r #{anat_output.path} -l 0.75"
	$stderr.puts "RUNNING: #{cmdline}"
	IO.popen(cmdline) do |p|
		p.each {|l| $stderr.puts l}
	end

	FileUtils.cp anat_output.path, "/vol/scratch/atiasnir/result.txt"

	IO.foreach(anat_output.path) do |line| 
		source, target, confidence = line.chomp.split(/\s/).first(3)

		from = source.to_i
		to = target.to_i

		add_node(nodes, terminals, anchors, gi, from) # unless source == 'ANCHOR'
		add_node(nodes, terminals, anchors, gi, to) # unless target == 'ANCHOR'

		#unless source == 'ANCHOR' || target == 'ANCHOR'
			edge_id = add_edge(edges, from, to, confidence)
			edges[edge_id]['anat'] = true
		#end
	end
ensure
	anat_input.close(true)
	anat_output.close(true) 
end

IO.foreach(network_filename) do |line| 
	from, to, confidence = line.chomp.split(/\t/).first(3)
	from = from.to_i
	to = to.to_i

	next unless nodes.include?(from) && nodes.include?(to)

	add_edge(edges, from, to, confidence)
end


doc = Nokogiri::XML::Builder.new do |xml|
	label = defined?(out_filename) ? File.basename(out_filename) : 'mock'
	xml.graph('xmlns' => 'http://www.cs.rpi.edu/XGMML', 'label' => label, 'directed' => '0') {
		xml.att :name => 'alpha', :type => 'real', :value => alpha
		nodes.each do |id, attr|
			xml.node(:id => id, :label => id) {
				xml.att :name => 'type', :type => 'string', :value => attr['type']
				xml.att(:name => 'name', :type => 'string', :value => attr['name']) if attr.include? 'name'
				case attr['type']
				when 'anchor'
					xml.graphics :type => "ELLIPSE", :h => "40.0", :w => "40.0", :fill => "#9999FF", :width => "1", :outline => "#666666"
				when 'terminal'
					xml.graphics :type => "ELLIPSE", :h => "40.0", :w => "40.0", :fill => "#00CCFF", :width => "1", :outline => "#666666"
				else
					xml.graphics :type => "ELLIPSE", :h => "40.0", :w => "40.0", :fill => "#CCCCCC", :width => "1", :outline => "#666666"
				end
			}
		end	
	
		edges.each do |id, attr|
			source = attr.delete("source")
			target = attr.delete("target")
			xml.edge(:source => source,
				:target => target,
				:label => "#{source} (pp) #{target}",
				:id => "#{source} (pp) #{target}") {
				xml.att(:name => 'interaction',
						:type => 'string', 
						:value => 'pp') 
				xml.att(:name => 'anat',
						:type => 'real', 
						:value => 1) if attr['anat']
				xml.att(:name => 'confidence',
						:type => 'real', 
						:value => attr['confidence'])
				xml.graphics :fill => "#000000", :width => (attr['confidence'].to_f * 10).to_i
			}
		end
	}
end.doc

#doc.write_xml_to($stderr)
#$stderr.puts ""

# run MCL
Mcl = '/home/bnet/atiasnir/app/bin/mcl'
NS =  { 'g' => 'http://www.cs.rpi.edu/XGMML' }

clusters = []
IO.popen("#{Mcl} - --abc -o - -I 1.8", "r+") do |p|
	doc.xpath('g:graph/g:edge', NS).each do |e|
		str = [e['source'], e['target'], e.at_xpath('g:att[@name="confidence"]/@value', NS).value].join("\t")
		p.puts str
	end	
	p.close_write
	p.each {|l| clusters << l.strip }
end

clusters.each_with_index do |cluster,idx|
	elements = cluster.split(/\s+/).map {|g| g.to_i}
	elements.each do |id|
		Nokogiri::XML::Builder.with(doc.at_xpath("g:graph/g:node[@id='#{id}']", NS)) do |xml|
			xml.att :name => 'cluster', :type => 'real', :value=> idx
		end
	end	
end

if defined? out_filename
	File.open(out_filename, 'w') do |f| 
		doc.write_xml_to(f)
	end
else	
	doc.write_xml_to($stdout)
end
