#!/bin/sh 

#    This program is based on SeqTrimMap version 1.0, 01-06-2012
#    Copyright 2012, Antonio Marco


mkdir tmp_folder
LOCAL=`pwd`
TEMP_PATH=$LOCAL/tmp_folder

# READ ARGUMENTS
args=`getopt zbiskScp:m:v:o:f:l:x:u:t: $*`

# PRINT HELP
if [ $# -eq 0 ] ; then
	clear
	echo 'USAGE: ./SeqTrimMap [OPTIONS] <INPUT_FASTA> <GENOME_FASTA_Bowtie_format> <genome_indexed_file> <mir_gff>'
	echo
	echo 'Options:'
	echo '  -m INT	Save appart reads mapping to more than INT positions'
	echo '  -v INT	Mismatches allowed, default is 0'
	echo '  -o <FILENAME>	Base name for output files'
	echo '  -f <FILENAME>	Filter reads mapping to the specified file (FASTA)'
	echo '  -l INT	Minimum read length after trimming to be mapped, default is 17'
	echo '  -x INT	Maximum read length before trimming'
	echo
	exit 1 
fi

# DEFAULT VALUES
mflag=-1 # Save appart reads mapping to more than 'mflag' positions (-1 means 'option disabled')
vflag=0 # Mismatches allowed
oflag="mapped_reads" # Base name for output files
fflag=0 # Filter reads mapping to the specified file (FASTA)
lflag=17 # Minimum read length after trimming to be mapped, default 17
xflag="no" # Maximum read length before trimming


# SET PARAMETERS
set -- $args
for i
do
	case "$i" in
		-m) shift;mflag="$1";shift;;
		-v) shift;vflag="$1";shift;;
		-o) shift;oflag="$1";shift;;
		-f) shift;fflag="$1";shift;;
		-l) shift;lflag="$1";shift;;
		-x) shift;xflag="$1";shift;;
	esac
done

clear

echo '* Preparing INPUT files'
echo -e "@name\nNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN\n+\nNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN" >  $TEMP_PATH/$$_temp_unmapped.-1
cat ${2} | sed '/^$/d' >> $TEMP_PATH/$$_temp_unmapped.-1

# Set TRIM limit
read_length=`head -2 $TEMP_PATH/$$_temp_unmapped.-1 | tail -1 | awk '{print length}'`
let trim_limit=$read_length-$lflag

if [[ $xflag == "no" ]]; then
	first_trim=$read_length #Default: read length
else
	first_trim=$xflag #User defined
	# MODIFY FIRST UNMAPPED FILE NAME
	let unmapped_file_number=$read_length-$first_trim-1
	mv $TEMP_PATH/$$_temp_unmapped.-1 $TEMP_PATH/$$_temp_unmapped.${unmapped_file_number}
fi

# Remove any previous STATs files
ls -1 ${oflag}.bowtie_stats > /dev/null 2>&1; if [ "$?" = "0" ]; then rm ${oflag}.bowtie_stats; fi

let trim_init=$read_length-$first_trim

echo "MAPPING (using bowtie)"

for ((i=$trim_init; i<=$trim_limit; i++))
do
	let actual_length=$read_length-$i
	let "i_ant = $i - 1"

	# FILTERING
	if [[ $fflag != 0 ]]; then
		echo >>${oflag}.bowtie_stats
		echo "Filtering reads at  length $actual_length" >>${oflag}.bowtie_stats
		echo -en "\r   - filtering reads at length $actual_length"

		bowtie -a --best --strata -v $vflag -3 $i --un $TEMP_PATH/$$_filtered.in $fflag $TEMP_PATH/$$_temp_unmapped.$i_ant > $TEMP_PATH/$$_filtered.out.$i 2>>${oflag}.bowtie_stats
		mv $TEMP_PATH/$$_filtered.in $TEMP_PATH/$$_temp_unmapped.$i_ant
	fi

	# MAPPING
	echo -en "\r   -   mapping reads at length $actual_length"
	if [[ $mflag == -1 ]]; then
		multiple_mapping_flag=''
	else
		multiple_mapping_flag=" -m $mflag --max $TEMP_PATH/$$_many_maps.$i "
	fi

	echo >>${oflag}.bowtie_stats
	echo "Mapping reads at  length $actual_length" >>${oflag}.bowtie_stats
	bowtie -a --best --strata -v $vflag -3 $i --un $TEMP_PATH/$$_temp_unmapped.$i $multiple_mapping_flag ${3} $TEMP_PATH/$$_temp_unmapped.$i_ant > $TEMP_PATH/$$_reads_mapped.$i 2>>${oflag}.bowtie_stats
	rm $TEMP_PATH/$$_temp_unmapped.$i_ant

done
echo -en "\r   - Bowtie mapping statistics saved in file: ${oflag}.bowtie_stats"

echo
echo '* Formatting OUTPUT files'


rm $TEMP_PATH/$$_temp_unmapped.*
ls -1 $TEMP_PATH/$$_filtered.out.* > /dev/null 2>&1; if [ "$?" = "0" ]; then cat $TEMP_PATH/$$_filtered.out.* > ${oflag}.filteredout; rm $TEMP_PATH/$$_filtered.out.*; fi
ls -1 $TEMP_PATH/$$_${fflag}.* > /dev/null 2>&1; if [ "$?" = "0" ]; then rm $TEMP_PATH/$$_${fflag}.*; fi
ls -1 $TEMP_PATH/$$_many_maps.* > /dev/null 2>&1; if [ "$?" = "0" ]; then cat $TEMP_PATH/$$_many_maps.* > ${oflag}.manymaps; rm $TEMP_PATH/$$_many_maps.*; fi

cat $TEMP_PATH/$$_reads_mapped.* | awk '{if(length($6)>16){print}}' > ${oflag}.mod.bowtie
rm $TEMP_PATH/$$_reads_mapped.*


perl allbowtie2sam.pl ${oflag}.mod.bowtie > $oflag.sam
samtools view -Sbu -t ${4} -o ${oflag}.bam $oflag.sam
samtools sort $oflag.bam $oflag.mod.sorted
rm $oflag.sam
rm $oflag.bam


perl mod_bam_to_mirs.pl $oflag.mod.sorted.bam ${2} ${5} $oflag.mod.bowtie > $oflag.mod.to_mirs.txt


exit 0
