import os
import sys
import stat
import shutil
import logging

from ..common import util
from ..common import job
from ..common.job import JobInfo

logger = logging.getLogger(__name__)

#-------------------------------------------------------------------------
# setup
#-------------------------------------------------------------------------
root_path = os.path.dirname(os.path.abspath(__file__))

jobTmpl_path = os.path.realpath(os.path.join(
  root_path,
  '..',
  'common',
  'job.sh'
))
worker_path = os.path.realpath(os.path.join(
  root_path,
  '..',
  '..',
  'rfa-worker.py',
))
jobTmplTxt = None
with open(jobTmpl_path) as f:
  jobTmplTxt = f.read()

#-------------------------------------------------------------------------
# batch
#-------------------------------------------------------------------------
class Batch(object):

  @property
  def scratch_path(self):
    return self._scratch_path
  @property
  def numJobs(self):
    return len(self.configs)

  def __init__(
    self,
    scratch_path,
    configs,
    uid,
    jobopts,
  ):
    self.uid = uid
    self.configs = configs
    self.jobopts = jobopts
    self._scratch_path = os.path.join(
      scratch_path,
      'batch.{0}'.format(self.uid),
    )
    util.mkdir_p(self.scratch_path)
    
  def run(
    self,
    checkfirst=False,
  ):
    def checkOutput(config):
      return (
        config.outputs != None and 
        config.outputs != [] and
        False not in map(
          lambda(f): os.path.exists(f),
          config.outputs,
        ) 
      )

    for config in self.configs:
      if checkfirst and checkOutput(config):
        logger.info('batch {0} worker {1} output generated'.format(
          self.uid,
          config.uid,
        ))
        continue

      # create slave batch script + config
      config_path = os.path.join(
        self.scratch_path,
        '{0}-{1}.config.json'.format(
          self.uid,
          config.uid,
        )
      )
      script_path = os.path.join(
        self.scratch_path,
        '{0}-{1}.launch.sh'.format(
          self.uid,
          config.uid,
        )
      )
      config.dump(config_path)
      txt = jobTmplTxt
      txt = txt.replace('<worker_path>', worker_path)
      txt = txt.replace('<args>', '-c {0}'.format(config_path))
      with open(script_path, 'w') as f:
        f.write(txt)
      permission = os.stat(script_path)
      os.chmod(script_path, permission.st_mode | stat.S_IXUSR)
      #die
      job.submit(
        JobInfo(
          script=script_path,
          **self.jobopts
        )
      )

    job.waitAll() 

#-------------------------------------------------------------------------
# worker abstract class
#-------------------------------------------------------------------------
class Worker(object):

  def __init__(
    self,
    config,
  ):
    self.config = config
    self.deliver_list = []

  def run(self):
    raise NotImplementedError()

  def deliver(self):
    if len(self.deliver_list) > 0:
      logger.info('copying deliverables back to: {0}'.format(
        self.config.outputDir_path))
      for src in self.deliver_list:
        logger.info('  - src: {0}'.format(src))
        if os.path.isdir(src):
          shutil.move(src, self.config.outputDir_path)
        else:
          shutil.copy(src, self.config.outputDir_path)

