import os
import sys
import json
import re
import logging

logger = logging.getLogger(__name__)

from .common import util
from .worker.well import WellStats
from .worker.wellsim import WellSim
from .worker.estimate import Estimator
from .worker.aligner import Aligner
from .bam.cloud_iter import CloudFinderTenX, CloudFinderMoleculo

#==========================================================================
# config base
#==========================================================================
class Config(object):

  @property
  def technology(self): return self.config_map['technology']
  @property
  def sampleInfo(self): return self.config_map['sampleInfo_map']
  @property
  def results_path(self): return self.config_map['resultsDir_path']
  @property
  def wellStats_path(self): return os.path.join(self.results_path, 'well-stats')
  @property
  def simWells_path(self): return os.path.join(self.results_path, 'sim-wells-fq')
  @property
  def simAlignments_path(self): return os.path.join(self.results_path, 'sim-wells-bam')
  @property
  def sampleAlignments_path(self): return os.path.join(self.results_path, 'sample-wells-bam')
  @property
  def params_path(self): return os.path.join(self.wellStats_path, 'params.p')
  @property
  def simSampleInfo(self): 
    if 'simSampleInfo_map'in self.config_map: 
      return self.config_map['simSampleInfo_map']
    else:
      return None
  @property
  def simParams(self): 
    assert 'simParams_map' in self.config_map
    assert 'numWells' in self.config_map['simParams_map']
    assert 'numBarcodesPerWell' in self.config_map['simParams_map']
    #assert 'genomeCov' in self.config_map['simParams_map']
    return self.config_map['simParams_map']
  @property
  def cloudFinder_cls(self):
    if self.technology == '10x':
      return CloudFinderTenX
    elif self.technology == 'moleculo':
      return CloudFinderMoleculo
    else:
      raise NotImplementedError()

  @classmethod
  def fromJson(cls, path):
    config_map = util.jsonLoadASCII(path)
    return cls(config_map)
    
  def __init__(
    self, 
    config_map,
  ):

    self.config_map = config_map
    self.__setup__()

  def __setup__(self):
    # check required
    requiredKeys = [
      'technology',
      'referenceFasta_path',
      'referenceBowtieIndex_path',
      'sampleInfo_map',
      'resultsDir_path',
    ]
    requiredAbs = [
      'resultsDir_path',
      'referenceFasta_path',
      'referenceBowtieIndex_path',
    ]
    requiredFiles = [
      self.config_map['referenceFasta_path'],
      self.config_map['referenceBowtieIndex_path'] + '.1.bt2',
    ]
    for key in requiredKeys:
      if key not in self.config_map:
        logger.error('required config key "{0}" not specified'.format(key))
        sys.exit(1)

    for key in requiredAbs:
      if not os.path.isabs(self.config_map[key]):
        logger.error('config key "{0}" required to be absolute path'.format(key))
        sys.exit(2)

    if not os.path.isfile(self.config_map['referenceFasta_path']):
      logger.error('configured fasta "{0}" does not exist'.format(
        self.config_map['referenceFasta_path']))
      sys.exit(3)
    if not os.path.isfile(self.config_map['referenceBowtieIndex_path'] + '.1.bt2'):
      logger.error('configured bowtie2 index prefix not valid')
      logger.error('  - {0} not found'.format(
        self.config_map['referenceBowtieIndex_path'] + '.1.bt2'))
      sys.exit(3)

    if self.technology not in [
      'moleculo',
      '10x',
    ]:
      logger.error('technology "{0}" not supported'.format(self.technology))
      sys.exit(4)

    # mkdirs
    util.mkdir_p(self.results_path)
    util.mkdir_p(self.wellStats_path)
    util.mkdir_p(self.simWells_path)
    util.mkdir_p(self.simAlignments_path)
    util.mkdir_p(self.sampleAlignments_path)

  # dereference path in config
  def get(self, *argv):
    proc = []
    root = self.config_map
    for arg in argv:
      if arg in root:
        root = root[arg]
        proc.append(arg)
      else:
        logger.error('property key {0} not present in config {1}'.format(
          arg,
          str(proc),
        ))
        sys.exit(1)
        break

    if type(root) == type({}):
      return dict(root)
    else:
      return root

  # dereference path (making path if does not exist)
  # - last element is value
  # - second to last value is key in leaf dict
  def set(self, *argv):
    assert len(argv) >= 2
    el = argv[-1]
    skey = argv[-2]
    path = argv[:-2]
    root = self.config_map
    for key in path:
      if key not in root:
        root[key] = {}
      root = root[key]
    root[skey] = el

  def dump(self, path):
    with open(path, 'w') as f:
      f.write(json.dumps(self.config_map, indent=2))

  def getSimWellFqPaths(self, limit=None):
    # override to use already generated simulated wells
    if self.simSampleInfo:
      return self.__getWellFqPaths__(
        self.simSampleInfo,
        limit=limit,
      )
    # use simulated wells generated from this sample
    else:
      return self.__getWellFqPaths__(
        {
          '1': {
            "fqDir_path" : self.simWells_path,
            "fqFnameFilter_str" :  "._(...)",
          }
        },
        limit=limit,
      )

  def getWellFqPaths(self, limit=None):
    return self.__getWellFqPaths__(
      self.sampleInfo,
      limit=limit,
    )

  def __getWellFqPaths__(self, sampleInfo_map, limit=None):
    pairs = []
    for (laneID_str, info_map) in sampleInfo_map.items():
      laneID = int(laneID_str)
      path = info_map['fqDir_path'] 

      fqFname_list = filter(
        lambda(s): s.endswith('.fastq') or s.endswith('.fq'),
        os.listdir(path),
      )   
      fq1_list = sorted(filter(
        lambda(s): s.endswith('1.fastq') or s.endswith('1.fq'),
        fqFname_list,
      ))  
      fq2_list = sorted(filter(
        lambda(s): s.endswith('2.fastq') or s.endswith('2.fq'),
        fqFname_list,
      ))  

      fqPair_list = zip(fq1_list, fq2_list);

      fqFnameFilter_str = info_map['fqFnameFilter_str']
      fqFnameFilter_re = re.compile(fqFnameFilter_str)

      for (fq1, fq2) in fqPair_list:
        match = re.match(fqFnameFilter_re, fq1)
        wellID_1 = int(match.group(1))
        match = re.match(fqFnameFilter_re, fq2)
        wellID_2 = int(match.group(1))
        assert wellID_1 == wellID_2
        wellID = wellID_1
        fq1_path = os.path.join(path, fq1)
        fq2_path = os.path.join(path, fq2)
        pairs.append(
          ((laneID, wellID), fq1_path, fq2_path)
        )
        if limit and len(pairs) >= limit:
          return pairs
        
    return pairs

    
#==========================================================================
# worker config
#==========================================================================
class WorkerConfig(Config):
  @property
  def uid(self):
    return 'l{0}_w{1:03d}'.format(
      self.config_map['inputs']['laneID'],
      self.config_map['inputs']['wellID'],
    )
  @property
  def worker_cls(self):
    cmd = self.config_map['inputs']['cmd']
    if cmd == 'stats':
      return WellStats
    elif cmd == 'sim':
      return WellSim
    elif cmd == 'estimate':
      return Estimator
    elif cmd == 'align':
      return Aligner
    else:
      raise NotImplementedError()

  @property
  def inputs(self): return self.config_map['inputs']
  @property
  def outputs(self): return self.config_map['outputs']
  @property
  def outputDir_path(self):
    if 'outputDir_path' not in self.inputs:
      return None
    else:
      return self.inputs['outputDir_path']
  @property
  def debug(self):
    if 'debug' in self.config_map['inputs']:
      return self.config_map['inputs']['debug']
    else:
      return False

  @classmethod
  def spawn(
    cls,
    gconfig,
    inputs_map,
    outputs_list,
  ):
    config_map = dict(gconfig.config_map)
    config_map['inputs'] = inputs_map
    config_map['outputs'] = outputs_list
    return cls(config_map)

  def __init__(
    self, 
    config_map,
  ):

    super(WorkerConfig, self).__init__(config_map)

  def __setup__(self):

    super(WorkerConfig, self).__setup__()

    # check required
    assert 'inputs' in self.config_map
    assert 'cmd' in self.config_map['inputs']
    assert 'outputs' in self.config_map

#==========================================================================
# config factory for workers
#==========================================================================
class ConfigFactory(object):

  def __init__(self, gconfig):
    self.gconfig = gconfig

  def getStatsConfigs(self, limit=None):

    configs = []
    for ((laneID, wellID), fq1, fq2) in \
      self.gconfig.getWellFqPaths(limit):
      input_map = {
        'cmd' : 'stats',
        'laneID' : laneID,
        'wellID' : wellID,
        'fq1_path' : fq1,
        'fq2_path' : fq2,
        'outputDir_path' : self.gconfig.wellStats_path,
      }
      stats_path = os.path.join(
        self.gconfig.wellStats_path,
        '{0}_{1}.stats.p'.format(laneID, wellID),
      )
      output_list = [
        stats_path,
      ]
      wconfig = WorkerConfig.spawn(
        self.gconfig,
        input_map,
        output_list,
      )
      configs.append(wconfig)
    return configs

  def getSimConfigs(self):
    configs = []
    laneID = 1

    numWells = self.gconfig.simParams['numWells']
    numBarcodesPerWell = self.gconfig.simParams['numBarcodesPerWell']
    totalBarcodes = numWells * numBarcodesPerWell
    for (wellID, barcodeStart)  in zip(
      xrange(numWells),
      xrange(0, totalBarcodes, numBarcodesPerWell),
    ):
      input_map = {
        'cmd' : 'sim',
        'laneID' : laneID,
        'wellID' : wellID,
        'barcodeStart' : barcodeStart,
        'outputDir_path' : self.gconfig.simWells_path,
      }
      pre__fq_path = os.path.join(
        self.gconfig.simWells_path,
        '{0}_{1:03d}'.format(laneID, wellID),
      )
      output_list = [
        pre__fq_path + '_1.fq',
        pre__fq_path + '_2.fq',
      ]
      wconfig = WorkerConfig.spawn(
        self.gconfig,
        input_map,
        output_list,
      )
      configs.append(wconfig)
    return configs

  def getEstimatorConfigs(self):
    input_map = {
      'cmd' : 'estimate',
      'outputDir_path' : self.gconfig.wellStats_path,
      # hack to make uid work
      'laneID' : 0,
      'wellID' : 0,
    }
    wconfig = WorkerConfig.spawn(
      self.gconfig,
      input_map,
      [self.gconfig.params_path],
    )
    return [wconfig]

  def getAlignConfigs(self, mode):
    configs = []

    # determine input fqs and output dir based on mode
    fqIter = None
    outputDir_path = None
    if mode == 'sample':
      fqIter = self.gconfig.getWellFqPaths()
      outputDir_path = self.gconfig.sampleAlignments_path
    elif mode == 'sim':
      fqIter = self.gconfig.getSimWellFqPaths()
      outputDir_path = self.gconfig.simAlignments_path
    else:
      die

    for ((laneID, wellID), fq1, fq2) in fqIter:
      input_map = {
        'cmd' : 'align',
        'simulate' : (mode == 'sim'),
        'laneID' : laneID,
        'wellID' : wellID,
        'fq1_path' : fq1,
        'fq2_path' : fq2,
        'outputDir_path' : outputDir_path,
      }
      bam_path = os.path.join(
        outputDir_path,
        '{0}_{1:03d}.bam'.format(laneID, wellID),
      )
      output_list = [
        bam_path,
      ]
      wconfig = WorkerConfig.spawn(
        self.gconfig,
        input_map,
        output_list,
      )
      configs.append(wconfig)
      ## FIXME just one test well for now
      #break
    return configs

