args <- commandArgs(trailingOnly = TRUE)
network_file <- toString(args[1])
regulators_file <- toString(args[2])
target_genes_file <- toString(args[3])
phenotype_linked_genes_file <- toString(args[4])
cutoff_interval <- as.integer(args[5])
cutoff_stop <- as.integer(args[6])
output_dir <- toString(args[7])
output_file <- "PhenoProphet_Scores.txt"

network <- as.matrix(read.table(network_file))
regulators <- as.matrix(read.table(regulators_file))
target_genes <- as.matrix(read.table(target_genes_file))
phenotype_linked_genes <- as.matrix(read.table(phenotype_linked_genes_file))

## This function runs the PhenoProphet algorithm. PhenoProphet assigns each TF in a network a score representing the confidence that the deletion of that TF will yield a phenotypic change of interest. Given a network, the PhenoProphet score of a TF for a phenotype or interest is based on the enrichment of the TF's targets for genes that are known to linked to the phenotype of interest. Specifically, a hypergoemetic enrichment test is applied for each regulator over many total network sizes, ranging from the 500 most confident network edges to the 40,000 most confident network edges.
runPhenoProphet <- function(network, regulators, target_genes, phenotype_linked_genes, cutoff_interval=500, cutoff_stop=40000) {

  network <- abs(network)
  phenotype_linked_genes_indx <- match(intersect(phenotype_linked_genes, target_genes), target_genes)
  tf_exclude_indx <- match(regulators, target_genes)

  pheno.sigs <- c()
  for (cutoff in network[order(network,decreasing=TRUE)][seq(cutoff_interval,cutoff_stop,by=cutoff_interval)]) {
      ## Only consider top X network edges
      network.cut <- network >= cutoff
      ## Number of targets of each TF at this network cutoff
      target.count <- apply(network.cut[,-tf_exclude_indx],1,sum) 
      ## Number of phenotype-linked targets of each TF at this network cutoff
      pheno.target.count <- apply(network.cut[,setdiff(phenotype_linked_genes_indx,tf_exclude_indx)],1,sum)

    pheno.sig <- c()
    for (i in 1:length(regulators)) {
      if (pheno.target.count[i]>0) {
	## Use hypergeometric distribution to compute an enrichment pval for each TF
        pheno.sig <- c(pheno.sig, phyper(pheno.target.count[i] - 1, length(setdiff(phenotype_linked_genes_indx,tf_exclude_indx)), length(target_genes[-tf_exclude_indx])-length(setdiff(phenotype_linked_genes_indx,tf_exclude_indx)), target.count[i], lower.tail=FALSE))
      } else {
        pheno.sig <- c(pheno.sig,1)
      }
    }
    pheno.sigs <- rbind(pheno.sigs, pheno.sig)
  }
  
  scores <- apply(-log(pheno.sigs+1e-100),2,max)
  results <- cbind(regulators, scores)
  results <- results[order(scores, decreasing=TRUE),]
  colnames(results) <- c("REGULATOR", "PHENOPROPHET_SCORE")

  results
}

phenoProphetResults <- runPhenoProphet(network, regulators, target_genes, phenotype_linked_genes, cutoff_interval, cutoff_stop)
write.table(phenoProphetResults, file=paste(output_dir, "/", output_file, sep=""), row.names=FALSE, col.names=TRUE, sep="\t", quote=FALSE)

