function CreateSynLib(lib_design_dir, elements_file_name,context_file_name,vec_file_name, ...
    construct_set_file_name, primers_stab_file, min_coor,max_coor, filter_out_seqs,exclude_sets, ...
    primer5_start, primer5_end, primer3_start, primer3_end, barcodes_file, barcode_start, barcode_end)
%function CreateSynLib(lib_design_dir, elements_file_name,context_file_name,vec_file_name, ...
%	construct_set_file_name, primers_stab_file, min_coor,max_coor, filter_out_seqs,exclude_sets, ...
%	primer5_start, primer5_end, primer3_start, primer3_end, barcodes_file, barcode_start, barcode_end)%
%
% This function create synthetic library sequences from a set of design files
%
% lib_design_dir - working directory
% elements_file_name - Tabular file defining the library element (Sequence elements such as TF binding sites, polyT etc.). For file format - see example file and documentation in Sharon et al. 2012
% context_file_name - Tabular file defining the library context sequences (the background sequences) See example file format. For file format - see example file and documentation in Sharon et al. 2012
% vec_file_name- Tabular file defining vecotrs of elements 3' position. cordinates are 3'->5' and one based. See example file format. For file format - see example file and documentation in Sharon et al. 2012
% construct_set_file_name - file defing the library structure using the elements, context and vector files. See example file format. For file format - see example file and documentation in Sharon et al. 2012
% min_coor,max_coor - minimal and maximal corrdinate in which elements can be postioned. Construct with elements that overlap a corrdinate outside
%                     of these bounds are removed. This allow to reserve sequence segments for the flanked shared seqiuences
% filter_out_seqs - cell array of sequences that need to be filtered from the library (for example - restriction sites)
% exclude_sets - cell array. Names os sets in which similar promoters with duplicate barcode are allowed
% primers_stab_file - tab file containing 3' and 5' shared sequences (for example the sequences used for PCR amplification). See example for format
% primer5_start, primer5_end, primer3_start, primer3_end - 5' and 3' shared sequences start and end coordinates
% barcodes_file - tab file containing barcode sequences See example for format
% barcode_start, barcode_end - 5' and 3' barcode sequences start and end coordinates
%
% Output and log files are created in - <lib_design_dir>/Output. The script create files in several steps of the function for documentation
% the final output files are "SynLib_design_full.[fas|mat]" files

if ~exist([lib_design_dir '/' Output], 'dir')
    mkdir([lib_design_dir '/' Output]);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  loading the data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[reg_elements_array reg_elements_struct reg_elements_by_name_struct]= SynLibLoadRegElements(elements_file_name);
[context_array context_struct context_by_name_struct]= SynLibLoadRegElements(context_file_name);
[vecs_array vecs_struct]  = SynLibLoadPosVecFile(vec_file_name);
[construct_set_array]  = SynLibLoadConstructSetFile(construct_set_file_name);
[construct_array] = SynLibConstructSet2ConstructArray(construct_set_array, reg_elements_struct, context_by_name_struct, vecs_struct);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% QA - remove overlaps
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

remove_overlaps_log_file = [ lib_design_dir '/Output/SynLib_overlaps.log'];

error_if_found_error = false;
[is_array_ok, construct_array] = SynLibQAConstArrayCheckElemOverlap(construct_array, error_if_found_error, min_coor, max_coor, remove_overlaps_log_file);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% saving array before removing duplicates
% (this was done after sequencing so if there are stochasticly generated elements they might be different)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


construct_array_with_duplicates_mat_file = [ lib_design_dir '/Output/SynLib_withDup.mat'];
save(construct_array_with_duplicates_mat_file, 'construct_array', 'construct_set_array', 'reg_elements_struct', 'context_struct', 'vecs_struct');
construct_array_with_duplicates_fas_file = [ lib_design_dir '/Output/SynLib_withDup.fa'];
[construct_str_array construct_seq_array] = SynLibWriteConstructsArray2Fasta(construct_array, construct_array_with_duplicates_fas_file);



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% removing constructs with restriction sites
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

ok_inds = true(size(construct_seq_array));

for i=1:length(filter_out_seqs)
    ok_inds = ok_inds & cellfun(@isempty,strfind(construct_seq_array, filter_out_seqs{i}));
end


found_filter_out_seqs = find(ok_inds);



if length(found_filter_out_seqs) > 0
    disp(['Found  ' num2str(length(found_filter_out_seqs)) ' constructs with restriction site !!!!!!!!']);
    
    res_site_constructs = construct_array(found_filter_out_seqs);
    
    for i=1:length(res_site_constructs)
        cur_construct = res_site_constructs{i};
        
        is_elem = isfield(cur_construct,'Element') && cur_construct.Element{1}.Start > 0;
        if ~is_elem
            disp([cur_construct.Context.Name ', is element exist:' num2str(is_elem)]);
        else
            disp([cur_construct.Context.Name ', is element exist:' num2str(is_elem) ', elem:' cur_construct.Element{1}.Name]);
        end
    end
end



% not clean set
out_fasta_file_Notclean = [ lib_design_dir '/Output/SynLib_design_Notclean.fas'];
out_mat_file_Notclean = [ lib_design_dir '/Output/SynLib_design_Notclean.mat'];

[construct_str_array_Notclean construct_seq_array_Notclean] = SynLibWriteConstructsArray2Fasta(construct_array, out_fasta_file_Notclean);
save(out_mat_file_Notclean, 'construct_array', 'construct_str_array_Notclean', 'construct_seq_array_Notclean');

% clean set

construct_array_clean = construct_array;
construct_array_clean(found_filter_out_seqs) = [];

out_fasta_file_clean = [ lib_design_dir '/Output/SynLib_design_clean.fas'];
out_mat_file_clean = [ lib_design_dir '/Output/SynLib_design_clean.mat'];

[construct_str_array_clean construct_seq_array_clean] = SynLibWriteConstructsArray2Fasta(construct_array_clean, out_fasta_file_clean);

save(out_mat_file_clean, 'construct_array_clean', 'construct_str_array_clean', 'construct_seq_array_clean');


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% remove duplicate constructs
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

construct_array = construct_array_clean;

out_fasta_file = [ lib_design_dir '/Output/SynLib_design.fas'];
out_mat_file = [ lib_design_dir '/Output/SynLib_design.mat'];

[construct_str_array construct_seq_array] = SynLibWriteConstructsArray2Fasta(construct_array, out_fasta_file);
save(out_mat_file, 'construct_array', 'construct_str_array', 'construct_seq_array');

out_fasta_file = [ lib_design_dir '/Output/SynLib_design_full_withDup.fas'];
out_mat_file = [ lib_design_dir '/Output/SynLib_design_full_withDup.mat'];

[construct_str_array construct_seq_array] = SynLibWriteConstructsArray2Fasta(construct_array, out_fasta_file);
save(out_mat_file, 'construct_array', 'construct_str_array', 'construct_seq_array');


remove_duplicates_log_file = [ lib_design_dir '/Output/SynLib_duplicates.log'];
[construct_array_uniq removed_duplicates not_removed_duplicates] = SynLibConstructSetRemoveSeqDuplicates(construct_array,remove_duplicates_log_file, exclude_sets);


out_fasta_file_uniq = [ lib_design_dir '/Output/SynLib_design_uniq.fas'];
out_mat_file_uniq = [ lib_design_dir '/Output/SynLib_design_uniq.mat'];

[construct_str_array_uniq construct_seq_array_uniq] = SynLibWriteConstructsArray2Fasta(construct_array_uniq, out_fasta_file_uniq);

save(out_mat_file_uniq, 'construct_array_uniq', 'construct_str_array_uniq', 'construct_seq_array_uniq');



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% compute set sizes
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

out_file_sets_sizes = [ lib_design_dir '/Output/SynLibset_sizes_uniq.tab'];
[sets_names sets_sizes]  = SynLibCountSetsSizes(construct_array_uniq, out_file_sets_sizes);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% adding the primers
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

out_fasta_file_uniq_with_barcodes = [ lib_design_dir '/Output/SynLib_design_uniq_with_primers.fas'];
out_mat_file_uniq_with_barcodes = [ lib_design_dir '/Output/SynLib_design_uniq_with_primers.mat'];


construct_array_uniq_with_primers = SynLibAddPrimer(construct_array_uniq,primers_stab_file,primer5_start, primer5_end, primer3_start, primer3_end);
[construct_str_array_uniq_with_primers  construct_seq_array_uniq_with_primers ] = SynLibWriteConstructsArray2Fasta(construct_array_uniq_with_primers, out_fasta_file_uniq_with_barcodes);
save(out_mat_file_uniq_with_barcodes, 'construct_array_uniq_with_primers', 'construct_str_array_uniq_with_primers', 'construct_seq_array_uniq_with_primers');


out_fasta_file_uniq_with_barcodes = [ lib_design_dir '/Output/SynLib_design_uniq_with_barcodes.fas'];
out_mat_file_uniq_with_barcodes = [ lib_design_dir '/Output/SynLib_design_uniq_with_barcodes.mat'];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% adding the barcodes
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[barcodes barcodes_ids]  = LoadStab2SChv(barcode_file);
construct_array_uniq_with_barcodes = SynLibAddBarcodes(construct_array_uniq_with_primers,barcodes, barcode_start, barcode_end, true,filter_out_seqs);

[construct_str_array_uniq_with_barcodes construct_seq_array_uniq_with_barcodes] = SynLibWriteConstructsArray2Fasta(construct_array_uniq_with_barcodes, out_fasta_file_uniq_with_barcodes);
save(out_mat_file_uniq_with_barcodes, 'construct_array_uniq_with_barcodes', 'construct_str_array_uniq_with_barcodes', 'construct_seq_array_uniq_with_barcodes');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% final output files
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

out_fasta_file_full = [ lib_design_dir '/Output/SynLib_design_full.fas'];
out_mat_file_full = [ lib_design_dir '/Output/SynLib_design_full.mat'];
construct_array_full = construct_array_uniq_with_barcodes;
[construct_str_array_full construct_seq_array_full] = SynLibWriteConstructsArray2Fasta(construct_array_full, out_fasta_file_full);

save(out_mat_file_full, 'construct_array_full', 'construct_str_array_full', 'construct_seq_array_full');


