package edu.mit.compbio.flynet.genome;

import java.util.ArrayList;


/**
 * Abstract class for any region on a chromosome (e.g., a gene, a peak, a motif, etc.)
 */
public class Region implements java.io.Serializable {

	/** Serial version UID, for serialization */
	private static final long serialVersionUID = 1;

	//static public enum Strand = {PLUS};
	/** The start of the region */
	protected int start_ = -1;
	/** The end of the region */
	protected int end_ = -1;
	/** Confidence (only for predicted regions) */
	protected double confidence_ = -1;
	
	
	// ============================================================================
	// PUBLIC METHODS
	
	/** Default constructor */
	public Region() { }
	
	/** Constructor with initialization */
	public Region(int start, int end) {
		setRegion(start, end);
	}
	
	/** Constructor with initialization */
	public Region(int start, int end, double confidence) {
		setRegion(start, end);
		confidence_ = confidence;
	}
	
	
	// ----------------------------------------------------------------------------

	/** Set the start and the end of the region */
	public void setRegion(int start, int end) {
		start_ = start;
		end_ = end;
		assert start < end;
	}
	
	
	// ----------------------------------------------------------------------------

	/** Return true if this region contains the region passed as argument */
	public boolean contains(Region region) {
		return start_ <= region.getStart() && region.getEnd() <= end_;   
	}
	
	
	// ============================================================================
	// STATIC STUFF

	static public class CompareStart implements java.util.Comparator<Region> {
		
		/**  */
		public int compare(Region region1, Region region2) {
			return region1.getStart() - region2.getStart();
		}
	} 
	
	/** Check if the regions are sorted in order of their starting position */
	static public boolean isSorted(ArrayList<Region> regions) {
		
		for (int i=0; i<regions.size()-1; i++)
			if (regions.get(i).getStart() > regions.get(i+1).getStart())
				return false;
		
		return true;
	}

	
	// ============================================================================
	// GETTERS AND SETTERS
	
	public int getStart() { return start_;	}
	public int getEnd() { return end_; }
	public double getConfidence() { return confidence_; }
	public int getLength() { return end_ - start_; }
	
}
