package edu.mit.compbio.flynet.genome;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;

import edu.mit.compbio.flynet.*;


/**
 * Flybase genome annotation 
 */
public class Genome implements java.io.Serializable {

	/** Serial version UID, for serialization */
	private static final long serialVersionUID = 1;

	/** The chromosomes */
	LinkedHashMap<String, Chromosome> chromosomes_ = null;
	/** Identifiers of chromosomes that are excluded from the analysis */
	HashSet<String> excludedChromosomes_ = null;
	/** The genes */
	HashMap<String, Gene> genes_ = null;
	
	
	// ============================================================================
	// PUBLIC METHODS
	
	/** Default constructor */
	public Genome() {
			
		ArrayList<String> mainChromosomes = new ArrayList<String>();
		mainChromosomes.add("2L");
		mainChromosomes.add("2R");
		mainChromosomes.add("3L");
		mainChromosomes.add("3R");
		mainChromosomes.add("4");
		mainChromosomes.add("X");
		
		ArrayList<String> notSoInteresting = new ArrayList<String>();
		notSoInteresting.add("2LHet");
		notSoInteresting.add("2RHet");
		notSoInteresting.add("3LHet");
		notSoInteresting.add("3RHet");
		notSoInteresting.add("XHet");
		notSoInteresting.add("YHet");
		notSoInteresting.add("U");
		notSoInteresting.add("Uextra");
		notSoInteresting.add("dmel_mitochondrion_genome");
 		
		ArrayList<String> chromosomes = new ArrayList<String>();
		chromosomes.addAll(mainChromosomes);
		chromosomes.addAll(notSoInteresting);
		
		excludedChromosomes_ = new HashSet<String>();
		
		if (Settings.considerOnlyOneChromosome_ != null) {
			excludedChromosomes_.addAll(chromosomes);
			if (!excludedChromosomes_.remove(Settings.considerOnlyOneChromosome_))
				Flynet.error(new RuntimeException("The specified chromosome '" + Settings.considerOnlyOneChromosome_ + "' is not known"));
		
		} else if (Settings.considerOnlyMainChromosomes_) {
			excludedChromosomes_.addAll(notSoInteresting);
		}

		chromosomes_ = new LinkedHashMap<String, Chromosome>();
		for (int i=0; i<chromosomes.size(); i++) {
			String id = chromosomes.get(i);
			chromosomes_.put(id, new Chromosome(id));
		}

		genes_ = new HashMap<String, Gene>();
	}

	
	// ----------------------------------------------------------------------------

	/** Get a chromosome by it's identifier, gives a fatal error if the id does not correspond to a chromosome */
	public Chromosome getChromosome(String id) {
		try {
			return chromosomes_.get(id);
		} catch (NullPointerException e) {
			Flynet.error(e);
		}
		return null; // We'll never get here
	}

	
	// ----------------------------------------------------------------------------

	/** Add the given gene to the list of genes and to its chromosome */
	public void addGene(String chromosome, Gene gene) {
		
		String fbgn = gene.getFbgn();
		
		// check that no other gene with the same fbgn has been added
		if (genes_.containsKey(fbgn))
			Flynet.error(new RuntimeException("Gene " + fbgn + " has already been added"));
		
		// We add all genes, even those of excluded chromosomes, because otherwise we have an incomplete gene list
		//if (!excludedChromosomes_.contains(chromosome)) {
			genes_.put(fbgn, gene);
			getChromosome(chromosome).addGene(gene);
		//}
	}
	
	
	// ----------------------------------------------------------------------------
	
	/** Get a gene by its FBgn, fatal error if not found */
	public Gene getGene(String fbgn) {
		
		Gene gene = genes_.get(fbgn);
		if (gene == null)
			Flynet.error(new RuntimeException("Unknown gene " + fbgn));
		
		return genes_.get(fbgn);
	}


	// ----------------------------------------------------------------------------

	/** Print some statistics to the console */
	public void printStatistics() {
		
		System.out.println();
		System.out.println("Printing genome statistics:");
		System.out.println();
		System.out.println("CHROM\t#GENES");
		System.out.println("-------------");

		Iterator<Chromosome> iter = chromosomes_.values().iterator();
		
		while (iter.hasNext()) {
			Chromosome chrom = iter.next();
			
			if (!chromosomeIsExcluded(chrom.getName()))
				chrom.printStatistics();
		}
	}
	
	
	// ----------------------------------------------------------------------------

	/** Save this genome to a file using Java serialization */
	public void writeSerialization(String filename) {
		
		try {
			System.out.println();
			System.out.println("Writing genome to file " + filename + " ...");
			
			// Write to disk with FileOutputStream
			FileOutputStream f_out = new FileOutputStream(filename);
			// Write object with ObjectOutputStream
			ObjectOutputStream obj_out = new ObjectOutputStream(f_out);
			// Write object out to disk
			obj_out.writeObject(this);

		} catch (Exception e) {
			Flynet.error(e);
		}
	}
	
	
	// ----------------------------------------------------------------------------

	/** Read a serialized genome from a file */
	static public Genome readSerialization(String filename) {
		
		Genome genome = null;
		
		try {
			System.out.println();
			System.out.println("Reading genome from file " + filename + " ...");

			// Read from disk using FileInputStream
			FileInputStream f_in = new FileInputStream(filename);
			// Read object using ObjectInputStream
			ObjectInputStream obj_in = new ObjectInputStream (f_in);
			// Read an object
			genome = (Genome)obj_in.readObject();
			
		} catch (Exception e) {
			Flynet.error(e);
		}

		return genome;
	}
	
	
	// ----------------------------------------------------------------------------

	/** Returns true if this chromosome should be skipped */
	public boolean chromosomeIsExcluded(String chromosome) {
		return excludedChromosomes_.contains(chromosome);
	}

	
	// ============================================================================
	// GETTERS AND SETTERS

	public LinkedHashMap<String, Chromosome> getChromosomes() { return chromosomes_; }
	
}
