/*
Copyright (c) 2011-2012 Daniel Marbach(1,2)

(1) Massachusetts Institute of Technology, Cambridge MA, USA
(2) Broad Institute of MIT and Harvard
 
We release this software open source under an MIT license (see below). If this
software was useful for your scientific work, please cite our paper available at:
http://compbio.mit.edu/flynet

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */
package edu.mit.compbio.flynet;

import com.martiansoftware.jsap.FlaggedOption;
import com.martiansoftware.jsap.JSAP;
import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.Switch;
import com.martiansoftware.jsap.UnflaggedOption;


/**
 * Global settings / parameters
 */
public class Settings {

	/** Reference to Flynet */
	public static Flynet flynet_ = null;
	
	/** The root directory where networks are loaded from */
	public static String networkDir_ = ".";
	/** The filename of the gold standard network */
	public static String goldStandardNetwork_ = null;	
	/** The number of randomized networks used for functional enrichment analysis */
	public static int numRandomized_ = 10;
	/** The value used for missing values in the feature-specific networks */
	public static double missingValue_ = 0.1; 
	/** 
	 * Gene pairs with Jaccard coefficient of their regulator sets greater than this 
	 * threshold are considered co-regulated (used by FunctionalEnrichment).
	 */
	public static double coregulationThreshold_ = 0.5;
	
	/** Load only one chromosome (concerns only the package flynet.genome) */
	public static String considerOnlyOneChromosome_ = "2L";
	/** Load only main chromosomes */
	public static boolean considerOnlyMainChromosomes_ = true;
		
	// COMMAND-LINE ARGUMENTS
	
	/** JSAP (Java Simple Argument Parser), defines and parses the command line arguments */
	public static JSAP jsap_ = null;
	/** The JSAP parsed command line */
	public static JSAPResult config_ = null;

	/** Generate the supervised network (using pre-defined weights, to learn the weights use -s) */
	public static boolean generateSup_ = false;
	/** Generate the unsupervised network */
	public static boolean generateUnsup_ = false;
	/** Compute the recovery rates */
	public static boolean computeRecoveryRates_ = false;
	/** The file with the target genes (used to filter the networks for the recovery rates) */
	public static String tgFile_ = null;
	/** Compute the functional enrichments (options: ) */
	public static String computeEnrichment_ = null;
	/** Run the test() function */
	public static boolean runTest_ = false;
	
	/** Compute network intersection */
	public static boolean computeIntersection_ = false;
	/** Compute network union */
	public static boolean computeUnion_ = false;
	/** Truncate the networks to the given fraction of the total number of possible edges */
	public static double cutoff_ = -1;
	/** Filter the networks for a given list of TFs and targets */
	public static boolean filter_ = false;
	/** 
	 * The minimal weight considered when loading networks (for intersection, union, cutoff)
	 * Set to -1 to load all edges independently of their weight 
	 */
	public static double wmin_ = -1;
	
	/** Map secondary to primary FBgns */
	public static boolean cleanFBgns_ = false;
	/** The columns to be edited */
	public static int argColumn_ = 0;
	/** The number of header lines to be skipped */
	public static int argNumHeaderLines_ = 0;
	/** Delete lines that could not be translated */
	public static boolean argDelete_ = false;
	
	/** A list of input files */
	public static String[] argFileList_ = null;
	

	// ============================================================================
	// STATIC METHODS

	/** Print help */
	public static void printUsage() {
		
		Flynet.println("");
		Flynet.println("Usage: java flynet");
		Flynet.println("                "
                            + jsap_.getUsage());
		Flynet.println("");
		Flynet.println("Tip: depending on the analysis that you run, you have to increase the java heap size using -Xmx");
		Flynet.println("Example: java -jar -Xmx4g flynet.jar -e GO");

		System.exit(0);

	}


	// ----------------------------------------------------------------------------

	/** Parse the command-line arguments */
	public static void parseArguments(String[] args) {
				
		// parse the command line
		config_ = jsap_.parse(args);

		if (!config_.success()) {
			printUsage();
			System.exit(1);
		}
		
		if (config_.contains("networkDir"))
			networkDir_ = config_.getString("networkDir");
		
		generateSup_ = config_.getBoolean("generateSup");
		generateUnsup_ = config_.getBoolean("generateUnsup");

		if (config_.contains("recovery")) {
			computeRecoveryRates_ = true;
			goldStandardNetwork_ = config_.getString("recovery");
		}
		
		if (config_.contains("tgs"))
			tgFile_ = config_.getString("tgs");
		
		computeEnrichment_ = config_.getString("enrichment");
		computeIntersection_ = config_.getBoolean("intersection");
		computeUnion_ = config_.getBoolean("union");
		
		if (config_.contains("cutoff"))
			cutoff_ = config_.getDouble("cutoff");
		
		filter_ = config_.getBoolean("filter");
		
		if (config_.contains("wmin"))
			wmin_ = config_.getDouble("wmin");
		
		cleanFBgns_ = config_.getBoolean("cleanFB");
		
		if (config_.contains("column"))
			argColumn_ = config_.getInt("column") - 1;		
		
		if (config_.contains("numHeaderLines"))
			argNumHeaderLines_ = config_.getInt("numHeaderLines");
		
		argDelete_ = config_.getBoolean("delete");
		
		runTest_ = config_.getBoolean("test");
		
		argFileList_ = config_.getStringArray("fileList");

	}
	

	// ----------------------------------------------------------------------------
	
	/** Define the command line arguments, set up JSAP */
	public static void initializeJsap() {
		
		// Java Simple Argument Parser 
		jsap_ = new JSAP();

        FlaggedOption networkDir = new FlaggedOption("networkDir");
        networkDir.setLongFlag("dir");
        networkDir.setHelp("Specify the directory where the networks are located");
        networkDir.setUsageName("path");

        Switch generateSup = new Switch("generateSup");
        generateSup.setLongFlag("sup");
        generateSup.setHelp("Generate the supervised network using logistic regression");

        Switch generateUnsup = new Switch("generateUnsup");
        generateUnsup.setLongFlag("unsup");
        generateUnsup.setHelp("Generate the unsupervised network");

        FlaggedOption recovery = new FlaggedOption("recovery");
        recovery.setLongFlag("recovery");
        recovery.setHelp("Compute recovery rates");
		recovery.setUsageName("goldstandard");

        FlaggedOption tgs = new FlaggedOption("tgs");
        tgs.setLongFlag("tgs");
        tgs.setHelp("The file with the target genes (used only with --recovery)");
        tgs.setUsageName("file");

		FlaggedOption enrichment = new FlaggedOption("enrichment");
		enrichment.setShortFlag('e');
		enrichment.setHelp("Compute enrichment of co-regulated genes for GO, ImaGO, devExpr, or PPI");
		enrichment.setUsageName("type");

        Switch intersection = new Switch("intersection");
        intersection.setShortFlag('i');
        intersection.setHelp("Get the intersection of two or more networks");
        
        Switch union = new Switch("union");
        union.setShortFlag('u');
        union.setHelp("Get the union of two or more networks");
        
        FlaggedOption cutoff = new FlaggedOption("cutoff");
        cutoff.setLongFlag("cutoff");
        cutoff.setHelp("Truncate the networks to the given fraction of the total number of possible edges");
        cutoff.setStringParser(JSAP.DOUBLE_PARSER);

		Switch filter = new Switch("filter");
		filter.setLongFlag("filter");
		filter.setHelp("Filter the networks for the given list of TFs and targets: --filter tf.txt tg.txt net.txt");

        FlaggedOption wmin = new FlaggedOption("wmin");
        wmin.setLongFlag("wmin");
        wmin.setHelp("The minimal weight considered when loading networks (for intersection, union, cutoff; all edges are loaded if this flag is not specified");
        wmin.setStringParser(JSAP.DOUBLE_PARSER);
		
        Switch cleanFB = new Switch("cleanFB");
        cleanFB.setLongFlag("fbgn");
        cleanFB.setHelp("Map secondary FBgns to primary FBgns of the given annotation version");
        
        FlaggedOption column = new FlaggedOption("column");
        column.setShortFlag('c');
        column.setHelp("The column that should be translated (use with --fbgn)");
        column.setStringParser(JSAP.INTEGER_PARSER);

        FlaggedOption numHeaderLines = new FlaggedOption("numHeaderLines");
        numHeaderLines.setShortFlag('l');
        numHeaderLines.setHelp("The number of header lines that should be skipped (use with --fbgn)");
        numHeaderLines.setStringParser(JSAP.INTEGER_PARSER);
        
        Switch delete = new Switch("delete");
        delete.setShortFlag('d');
        delete.setHelp("Delete lines with an identifier that could not be translated (use with --fbgn)");

        Switch test = new Switch("test");
        test.setShortFlag('t');
        test.setHelp("Run the test() function");

        UnflaggedOption fileList = new UnflaggedOption("fileList");
        fileList.setHelp("A list of input files");
        fileList.setGreedy(true);
        fileList.setUsageName("file");

        // register the parameters
        try {
        	jsap_.registerParameter(networkDir);
        	jsap_.registerParameter(generateSup);
        	jsap_.registerParameter(generateUnsup);
        	jsap_.registerParameter(recovery);
        	jsap_.registerParameter(tgs);
        	jsap_.registerParameter(enrichment);
        	jsap_.registerParameter(intersection);
        	jsap_.registerParameter(union);
        	jsap_.registerParameter(cutoff);
        	jsap_.registerParameter(filter);
        	jsap_.registerParameter(wmin);
        	jsap_.registerParameter(cleanFB);
        	jsap_.registerParameter(column);
        	jsap_.registerParameter(numHeaderLines);
        	jsap_.registerParameter(delete);
        	jsap_.registerParameter(test);
        	jsap_.registerParameter(fileList);
		} catch (JSAPException e) {
			Flynet.error(e);
		}
	}

}
