/*
Copyright (c) 2011-2012 Daniel Marbach(1,2)

(1) Massachusetts Institute of Technology, Cambridge MA, USA
(2) Broad Institute of MIT and Harvard
 
We release this software open source under an MIT license (see below). If this
software was useful for your scientific work, please cite our paper available at:
http://compbio.mit.edu/flynet

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */
package edu.mit.compbio.flynet;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;


/**
 * Basic set operations on networks (edge lists): Intersection, union, complement...
 */
public class NetworkUtil {

	
	// ============================================================================
	// STATIC METHODS
	    
	/** The edges of net1 that are not in net2 (the relative complement of net2 with respect to net1) */
	public static Network complement(Network net1, Network net2) {
		
		Network compl = new Network();
		
		for (int i=0; i<net1.getNumEdges(); i++) {
			Edge edge = net1.getEdge(i);
			if (!net2.contains(edge))
				compl.addCopyOfEdge(edge);
		}
		
		return compl;
	}
	
	
    // ----------------------------------------------------------------------------

	/** 
	 * The edges of net1 that are also in net2 (the intersection of net1 with net2).
	 * If the flag is set, the edges that are part of the intersection are assigned the
	 * sum of the two weights, otherwise the weight of the edge of the first network is used.
	 */
	public static Network intersect(Network net1, Network net2, boolean addWeights) {
		
		Network inter = new Network();
		
		for (int i=0; i<net1.getNumEdges(); i++) {
			Edge edge1 = net1.getEdge(i);
			Edge edge2 = net2.getEdge(edge1);
			if (edge2 != null) {
				if (addWeights)
					inter.addEdge(edge1.getTf().getLabel(), edge1.getTarget().getLabel(), edge1.getWeight() + edge2.getWeight());
				else
					inter.addCopyOfEdge(edge1);
			}
		}
		Collections.sort(inter.getEdges());
		
		return inter;
	}

	
    // ----------------------------------------------------------------------------

	/** 
	 * The edges that are either in net1 or in net2 (the union of net1 and net2).
	 * If the flag is set, the edges that are part of both networks are assigned the sum 
	 * of the two weights (they are not added twice), otherwise the weight of the edge of the
	 * first network is used.
	 */
	public static Network union(Network net1, Network net2, boolean addWeights) {
		
		Network uni = new Network();
		
		// First, add all edges of net1
		for (int i=0; i<net1.getNumEdges(); i++)
			uni.addCopyOfEdge(net1.getEdge(i));
		
		// Add the edges of net2 that are not in net1, for those that are in both networks,
		// just add the weight if the flag is set
		for (int i=0; i<net2.getNumEdges(); i++) {
			Edge edge2 = net2.getEdge(i);
			Edge edgeUni = uni.getEdge(edge2);
			if (edgeUni == null)
				uni.addCopyOfEdge(edge2);
			else if (addWeights)
				edgeUni.setWeight(edgeUni.getWeight() + edge2.getWeight());
		}
		Collections.sort(uni.getEdges());
		
		return uni;
	}

	
	// ----------------------------------------------------------------------------

	/** Compute the Jaccard index between two generic hash sets */
	public static double jaccardIndex(HashSet<String> set1, HashSet<String> set2) {
		
		if (set1.size() == 0 || set2.size() == 0)
			return 0;
		
		int intersection = intersection(set1, set2); 
		
		return intersection / (double)(set1.size() + set2.size() - intersection);
	}

	
	// ----------------------------------------------------------------------------

	/** Count the number of elements that are in both sets */
	public static int intersection(HashSet<String> set1, HashSet<String> set2) {
		
		//HashSet intersection = new HashSet(set1);
		//intersection.retainAll(set2);

		HashSet<String> smallerSet;
		HashSet<String> biggerSet;
		if (set1.size() <= set2.size()) {
			smallerSet = set1;
			biggerSet = set2;
		} else {
			smallerSet = set2;
			biggerSet = set1;
		}
		int intersection = 0;
		
		Iterator<String> iter = smallerSet.iterator();
		while (iter.hasNext()) {
			if (biggerSet.contains(iter.next()))
				intersection++;
		}
		return intersection;
	}

}
