/*
Copyright (c) 2011-2012 Daniel Marbach(1,2)

(1) Massachusetts Institute of Technology, Cambridge MA, USA
(2) Broad Institute of MIT and Harvard
 
We release this software open source under an MIT license (see below). If this
software was useful for your scientific work, please cite our paper available at:
http://compbio.mit.edu/flynet

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */
package edu.mit.compbio.flynet;

import java.util.HashMap;
import java.util.HashSet;


/**
 * GO or ImaGO annotation for genes
 */
public class GeneAnnotation implements FunctionalDataset {

	/** The name of this dataset */
	private String name_ = null;
	/** The annotation for each gene */
	private HashMap<String, HashSet<String>> annotation_ = null;
	

	// ============================================================================
	// PUBLIC METHODS
	
	/** Load the annotation from the given file */
	public void load(String filename) {

		name_ = filename.substring(0, filename.indexOf("."));
		annotation_ = new HashMap<String, HashSet<String>>();
		
		String path = Settings.networkDir_ + "/" + filename;
		Flynet.println("Reading file: " + path);
		
		FileParser parser = new FileParser(path);
		String[] nextLine = parser.readLine();

		while (nextLine != null) {
			
			String gene = nextLine[0];
			String term = nextLine[1];
			
			HashSet<String> annot = annotation_.get(gene);
			if (annot == null) {
				annot = new HashSet<String>();
				annotation_.put(gene, annot);
			}
			annot.add(term);
			
			nextLine = parser.readLine();
		}
		
		parser.close();
	}

	
	// ----------------------------------------------------------------------------

	/** Returns true if the given gene has at least one annotation */
	public boolean contains(Node gene) {
		
		return annotation_.containsKey(gene.getLabel());
	}


	// ----------------------------------------------------------------------------

	/** Compute the Jaccard index for the sets of annotations of the two genes */
	public double functionalRelation(Node gene1, Node gene2) {

		HashSet<String> terms1 = annotation_.get(gene1.getLabel());
		HashSet<String> terms2 = annotation_.get(gene2.getLabel());
		
		if (terms1 == null || terms2 == null)
			throw new RuntimeException("One of the given genes is not annotated");
		
		return NetworkUtil.jaccardIndex(terms1, terms2);
	}
	    
	
	// ============================================================================
	// SETTERS AND GETTERS
	
	public void setName(String name) { name_ = name; }
	public String getName() { return name_; }

}
