/*
Copyright (c) 2011-2012 Daniel Marbach(1,2)

(1) Massachusetts Institute of Technology, Cambridge MA, USA
(2) Broad Institute of MIT and Harvard
 
We release this software open source under an MIT license (see below). If this
software was useful for your scientific work, please cite our paper available at:
http://compbio.mit.edu/flynet

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */
package edu.mit.compbio.flynet;

import java.util.ArrayList;
import java.util.HashSet;


/**
 * Functional enrichments of physical and integrative networks (Table III)
 */
public class FunctionalEnrichment {

	/** The supervised integrative network */
	private Network integrativeSup_ = null;
	/** The unsupervised integrative network */
	private Network integrativeUnsup_ = null;
	/** The ChIP network */
	private Network chip_ = null;
	/** The motif network */
	private Network motif_ = null;
	/** The REDfly network */
	private Network redfly_ = null;
	
	/** The functional dataset for which the enrichment is computed */
	private FunctionalDataset functionalDataset_ = null;
	
	
	// ============================================================================
	// PUBLIC METHODS

	/** Compute the enrichment specified by the type (GO, ImaGO, devExpr, or PPI) */
	public void run(String type) {
		
		if (type.equalsIgnoreCase("GO"))
			computeEnrichmentGO();
		else if (type.equalsIgnoreCase("ImaGO"))
			computeEnrichmentImaGO();
		else if (type.equalsIgnoreCase("devExpr"))
			computeEnrichmentDevExpression();
		else if (type.equalsIgnoreCase("PPI"))
			computeEnrichmentPPI();
		else if (type.equalsIgnoreCase("HiC"))
			computeEnrichmentHiCLongRange();
		else 
			throw new RuntimeException("ERROR: Unknown enrichment type: " + type);
	}


	// ----------------------------------------------------------------------------

	/** Compute enrichment of co-regulated genes for GO annotation */
	public void computeEnrichmentGO() {
		
		loadNetworks();
		
		Settings.networkDir_ = "resources/functional_enrichment";
		functionalDataset_ = new GeneAnnotation();
		functionalDataset_.load("go.txt");
		functionalDataset_.setName("GO");
		Flynet.println("");
		
		Settings.numRandomized_ = 10;
		Settings.coregulationThreshold_ = 0.5;
		computeEnrichment();
	}
	
	
	// ----------------------------------------------------------------------------

	/** Compute enrichment of co-regulated genes for ImaGO annotation */
	public void computeEnrichmentImaGO() {
		
		loadNetworks();
		
		Settings.networkDir_ = "resources/functional_enrichment";
		functionalDataset_ = new GeneAnnotation();
		functionalDataset_.load("imago.txt");
		functionalDataset_.setName("ImaGO");
		Flynet.println("");
		
		Settings.numRandomized_ = 10;		
		Settings.coregulationThreshold_ = 0.5;
		computeEnrichment();
	}

	
	// ----------------------------------------------------------------------------

	/** 
	 * Compute functional enrichment of co-regulated genes for all networks and all datasets.
	 * WARNING: This resets the loaded networks in the end (because they have been randomized)!
	 */
	public void computeEnrichmentDevExpression() {
		
		loadNetworks();
		
		Settings.networkDir_ = "resources/functional_enrichment";
		functionalDataset_ = new GeneExpression(false);
		functionalDataset_.load("rnaseq2.txt");
		functionalDataset_.setName("devExpression");
		Flynet.println("");
		
		Settings.numRandomized_ = 10;
		Settings.coregulationThreshold_ = 0.5;
		computeEnrichment();
	}

	
	// ----------------------------------------------------------------------------

	/** 
	 * Compute functional enrichment of co-regulated genes for all networks and all datasets.
	 * WARNING: This resets the loaded networks in the end (because they have been randomized)!
	 */
	public void computeEnrichmentPPI() {
		
		loadNetworks();
		
		Settings.networkDir_ = "resources/functional_enrichment";
		functionalDataset_ = new InteractionDataset();
		functionalDataset_.load("ppi.txt");
		functionalDataset_.setName("PPI");
		Flynet.println("");
		
		Settings.numRandomized_ = 100;
		Settings.coregulationThreshold_ = 0.25;
		computeEnrichment();
	}


	// ----------------------------------------------------------------------------

//	/** 
//	 * Compute functional enrichment of co-regulated genes for all networks and all datasets.
//	 * WARNING: This resets the loaded networks in the end (because they have been randomized)!
//	 */
//	public void computeEnrichmentHiC() {
//		
//		loadNetworks();
//		
//		Settings.networkDir_ = "resources/functional_enrichment/HiC";
//		functionalDataset_ = new GeneInteraction3D();
//		functionalDataset_.load("10k");
//		functionalDataset_.setName("HiC");
//		Flynet.println("");
//		
//		Settings.numRandomized_ = 1; // TODO change to 10!!!!!!!!!!!!!!!!!!!!!!!!!!!
//		Settings.coregulationThreshold_ = 0.5;
//		computeEnrichment();
//	}

	
	// ----------------------------------------------------------------------------

	/** 
	 * Compute functional enrichment of co-regulated genes for all networks and all datasets.
	 * WARNING: This resets the loaded networks in the end (because they have been randomized)!
	 */
	public void computeEnrichmentHiCLongRange() {
		
		loadNetworks();
		
		Settings.networkDir_ = "resources/functional_enrichment/HiC";
		functionalDataset_ = new InteractionDataset();
		functionalDataset_.load("long_range_interactions.txt");
		functionalDataset_.setName("HiC_long_range_interactions");
		Flynet.println("");
		
		Settings.numRandomized_ = 10;
		Settings.coregulationThreshold_ = 0.25;
		computeEnrichment();
	}

	// ----------------------------------------------------------------------------

	/** 
	 * Compute enrichment for the networks and functional dataset that have been previously loaded
	 * WARNING: This resets the loaded networks in the end (because they have been randomized)!
	 */
	public void computeEnrichment() {
		
		computeFunctionalEnrichment(integrativeSup_, functionalDataset_);
		computeFunctionalEnrichment(integrativeUnsup_, functionalDataset_);
		//computeFunctionalEnrichment(motif_, functionalDataset_);
		//computeFunctionalEnrichment(chip_, functionalDataset_);
		//computeFunctionalEnrichment(redfly_, functionalDataset_);
	}

	
	// ============================================================================
	// PRIVATE METHODS

	/** Load all networks */
	private void loadNetworks() {
		
		// Load the integrative networks
		Settings.networkDir_ = "resources/networks/integrative";

		integrativeSup_ = new Network();
		integrativeSup_.load("flynet_supervised_0.6.txt", false);
		integrativeSup_.setName("IntegrativeSup");
		
		integrativeUnsup_ = new Network();
		integrativeUnsup_.load("flynet_unsupervised_0.02.txt", false);
		integrativeUnsup_.setName("IntegrativeUnsup");
		
		// Load the physical networks
		Settings.networkDir_ = "resources/networks/physical";

		chip_ = new Network();
		chip_.load("chip_net.txt", true);
		chip_.setName("Chip");
		
		motif_ = new Network();
		motif_.load("motif_net.txt", false);
		motif_.setName("Motif");
		
		// The known network
		Settings.networkDir_ = "resources/networks";
		redfly_ = new Network();
		redfly_.load("redfly.txt", false);
		redfly_.setName("REDfly");
	}

	
	// ----------------------------------------------------------------------------

	/** 
	 * Compute functional enrichment of co-regulated genes in the given dataset.
	 * WARNING: This resets the loaded networks in the end (because they have been randomized)!
	 */
	private double computeFunctionalEnrichment(Network network, FunctionalDataset dataset) {
				
		Flynet.println("Computing functional enrichment: " + network.getName() + "_" + dataset.getName());
		if (network.getNumEdges() == 0)
			throw new RuntimeException("The network has no edges (maybe the network has been deleted because it was previously used and randomized by computeFunctionalEnrichment()");
		ArrayList<Double> functionalRelation = computeFunctionalRelation(network, dataset);

		double networkAvg = mean(functionalRelation);
		FileExport writer = new FileExport(network.getName() + "_" + dataset.getName() + ".txt");
		writer.print(functionalRelation);
		writer.close();
		
		ArrayList<Double> functionalRelationRandom = new ArrayList<Double>();
		
		System.out.print("Randomization: ");
		for (int i=0; i<Settings.numRandomized_; i++) {
			System.out.print(".");
			network.randomize();
			ArrayList<Double> nextRandom = computeFunctionalRelation(network, dataset);
			functionalRelationRandom.addAll(nextRandom);
		}
		System.out.print("\n");
		
		double randomAvg = mean(functionalRelationRandom);
		writer = new FileExport(network.getName() + "_" + dataset.getName() + "_randomized.txt");
		writer.print(functionalRelationRandom);
		writer.close();

		// Compute the enrichment
		double enrichment = networkAvg / randomAvg;
		Flynet.println("Enrichment:\t" + enrichment + "\n");
		
		// Delete the randomized network to make sure it's not being reused accidentally
		network.initialize();
		
		return enrichment;
	}

		
	// ----------------------------------------------------------------------------

	/** Return the sum of the given vector */
	private double sum(ArrayList<Double> vect) {
		
		double sum = 0;
		for (int i=0; i<vect.size(); i++)
			sum += vect.get(i);
		return sum;
	}

	
	// ----------------------------------------------------------------------------

	/** Return the mean of the given vector */
	private double mean(ArrayList<Double> vect) {
		
		return sum(vect) / vect.size();
	}

	
	// ----------------------------------------------------------------------------

	/** Compute the functional relation for co-regulated genes in the given network and dataset */
	private ArrayList<Double> computeFunctionalRelation(Network network, FunctionalDataset dataset) {
		
		ArrayList<Double> functionalRelation = new ArrayList<Double>();
		Node[] targets = network.getTargets().toArray(new Node[1]);
		
		// For every pair of target genes, where both genes have annotations
		for (int i=0; i<targets.length-1; i++) {
			// If the first gene has no annotations, continue
			if (!dataset.contains(targets[i]))
				continue;
			
			for (int j=i+1; j<targets.length; j++) {
				// If the second gene has no annotations, continue
				if (!dataset.contains(targets[j]))
					continue;
				
				// Check if the two genes are co-regulated
				HashSet<String> regulators_i = targets[i].getRegulators();
				HashSet<String> regulators_j = targets[j].getRegulators();
				
				// If the two genes share more than 50% of their regulators
				if (NetworkUtil.jaccardIndex(regulators_i, regulators_j) > Settings.coregulationThreshold_) {
					// Compute the functional relation between the two genes
					double f = dataset.functionalRelation(targets[i], targets[j]);
					if (f != Double.NaN)
						functionalRelation.add(f);
				}
			}
		}
		return functionalRelation;
	}

}
