/*
Copyright (c) 2011-2012 Daniel Marbach(1,2)

(1) Massachusetts Institute of Technology, Cambridge MA, USA
(2) Broad Institute of MIT and Harvard
 
We release this software open source under an MIT license (see below). If this
software was useful for your scientific work, please cite our paper available at:
http://compbio.mit.edu/flynet

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */
package edu.mit.compbio.flynet;

import java.util.ArrayList;
import java.util.Collections;


/**
 * Main class
 */
public class Flynet {


	// ============================================================================
	// STATIC METHODS

	/** 
	 * Main function.
	 */
	public static void main(String[] args) {
		
		try {
			Flynet flynet = new Flynet();
			Settings.flynet_ = flynet;
			flynet.run(args);
			
		} catch (Exception e) {
			error(e);
		}
	}


    // ----------------------------------------------------------------------------

	/** Print the stack trace of the exception and exit */
	static public void error(Exception e) {
		e.printStackTrace();
		System.exit(-1);
	}

	
    // ----------------------------------------------------------------------------

	/** Print a message */
	static public void println(String msg) {
		System.out.println(msg);
	}

	
    // ----------------------------------------------------------------------------

	/** Print a message */
	static public String insertBeforeFileExtension(String filename, String appendix) {
		int index = filename.lastIndexOf('.');
		if (index < 2)
			index = filename.length();
		return filename.substring(0, index) + appendix + ".txt";
	}
	
	
	// ============================================================================
	// PUBLIC METHODS
	
	/** Default constructor */
	public Flynet() {
	
		Settings.initializeJsap();
	}

	
	// ----------------------------------------------------------------------------

	/** Runs analysis based on input arguments */
	public void run(String[] args) {
		
		Settings.parseArguments(args);
		
		if (Settings.generateSup_ || Settings.generateUnsup_)
			new FlynetInference().run();
		else if (Settings.computeRecoveryRates_)
			new RecoveryRate().run();
		else if (Settings.computeEnrichment_ != null)
			new FunctionalEnrichment().run(Settings.computeEnrichment_);
		else if (Settings.computeIntersection_ || Settings.computeUnion_)
			networkOperations();
		else if (Settings.cutoff_ != -1)
			runTruncate();
		else if (Settings.filter_)
			runFilter();
		else if (Settings.cleanFBgns_)
			cleanFBgns();
		else if (Settings.runTest_)
			tests();
		else
			Settings.printUsage();
		
	}
	  		
	
	// ============================================================================
	// PRIVATE METHODS
		
	/** Compute the intersection or union between the networks passed as arguments */
	private void networkOperations() {
		
		String[] fileList = Settings.argFileList_;
		
		int numNetworks = fileList.length;
		if (numNetworks < 2)
			throw new IllegalArgumentException("At least two networks are required");
		
		// Load all networks
		ArrayList<Network> networks = new ArrayList<Network>(); 
		for (int i=0; i<numNetworks; i++) {
			Network net = new Network();
			if (Settings.wmin_ == -1)
				net.load(fileList[i], false);
			else
				net.loadEdgesAboveThreshold(fileList[i], Settings.wmin_, false);
			networks.add(net);
		}
		
		String filename = null;
		Network result = null;
		
		if (Settings.computeIntersection_) {
			// Intersect the first two networks
			result = NetworkUtil.intersect(networks.get(0), networks.get(1), true);
			// Intersect with the remaining networks
			for (int i=2; i<numNetworks; i++)
				result = NetworkUtil.intersect(result, networks.get(i), true);
			
			filename = "intersection.txt";
			Flynet.println("Intersection:");

		} else if (Settings.computeUnion_) {
			result = NetworkUtil.union(networks.get(0), networks.get(1), true);
			for (int i=2; i<numNetworks; i++)
				result = NetworkUtil.union(result, networks.get(i), true);
			
			filename = "union.txt";
			Flynet.println("Union:");
		}
		
		// Print info and save
		result.printInfo();
		Flynet.println("Writing file: " + filename);
		result.save(filename);

	}

	
	// ----------------------------------------------------------------------------

	/** Truncate the network to the fraction of total possible edges given in Settings.truncateNetworks_ */
	private void runTruncate() {
				
		String[] fileList = Settings.argFileList_;

		// check that an input file has been specified
		if (fileList == null || fileList.length < 1)
			throw new IllegalArgumentException("Specify at least one network: --trunc <fraction> <net1> <net2> ...");
				
		System.out.println("Truncating networks to top " + (100*Settings.cutoff_) + "% of total edges...");
		if (Settings.cutoff_ <= 0 || Settings.cutoff_ > 1)
			throw new IllegalArgumentException("The cutoff must be between 0 and 1");
		
		// load, truncate, and write the networks
		for (int i=0; i<fileList.length; i++) {
			Network net = new Network();
			if (Settings.wmin_ == -1)
				net.load(fileList[i], false);
			else
				net.loadEdgesAboveThreshold(fileList[i], Settings.wmin_, false);			
			ArrayList<Edge> edges = net.getEdges();
			int numEdgesBefore = edges.size();
			int numEdgesAfter = (int) Math.round(Settings.cutoff_*numEdgesBefore);
						
	    	// Sort the edge list according to weights
			// First we shuffle so that if several edges have the same weight, it will be different every time
			Collections.shuffle(edges);
	        Collections.sort(edges);
	        edges.subList(numEdgesAfter, edges.size()).clear(); // remove the tail of the list
	        assert net.getNumEdges() == numEdgesAfter;
	        	        
			println("Num edges after cutoff: " + numEdgesAfter);
			println("");
			
	        String filename = insertBeforeFileExtension(fileList[i], "_trunc");
	        net.save(filename);
		}
	}

	
	// ----------------------------------------------------------------------------

	/** Filter the networks for the given list of TFs and targets */
	private void runFilter() {
				
		String[] fileList = Settings.argFileList_;

		// check that at least three input files have been specified (TFs, targets, network)
		if (fileList == null || fileList.length < 3)
			throw new IllegalArgumentException("Specify at least three files: --filter <TFs> <targets> <network> ...");
						
		String tfList = fileList[0];
		String targetList = fileList[1];
		
		// load, filter, and write the networks
		for (int i=2; i<fileList.length; i++) {
			String inputFile = fileList[i];
			
			Network net = new Network();
			net.loadSubnet(inputFile, tfList, targetList, false);
			
			String outputFile = insertBeforeFileExtension(inputFile, "_filtered");
			net.save(outputFile);
		}
	}

	
	// ----------------------------------------------------------------------------
	
	/** Map secondary to primary FBgns */
	private void cleanFBgns() {
		
		String[] fileList = Settings.argFileList_;

		// check that an input file has been specified
		if (fileList == null || fileList.length != 1)
			throw new RuntimeException("You must specify exactly one input file, use: --fbgn <inputfile>");
				
		Translator translator = Translator.getInstance();
		translator.toPrimaryFBgn(fileList[0], Settings.argColumn_, Settings.argNumHeaderLines_, Settings.argDelete_);

	}

	
	// ----------------------------------------------------------------------------

	/** Sandbox */
	private void tests() {

//		Network net = new Network();
//		net.load("resources_private/final_networks/cutoff/flynet_supervised_0.6.txt", false);
//		
//		net = new Network();
//		net.load("resources_private/final_networks/cutoff/flynet_unsupervised_0.03.txt", false);
		
		//new SupervisedLearning().tests();
		
		//GenomeInteractionMapParser parser = new GenomeInteractionMapParser();
		//parser.parseMmc3();
		
//		ExpressionDatasetParser parser = new ExpressionDatasetParser();
//		//parser.parseBiniou();
//		//parser.parseTwist();
//		parser.parseMef2();
		
		System.exit(0);
	}

}
