/*
Copyright (c) 2011-2012 Daniel Marbach(1,2)

(1) Massachusetts Institute of Technology, Cambridge MA, USA
(2) Broad Institute of MIT and Harvard
 
We release this software open source under an MIT license (see below). If this
software was useful for your scientific work, please cite our paper available at:
http://compbio.mit.edu/flynet

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */
package edu.mit.compbio.flynet;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;


/**
 * Parse a tab-separated value file
 */
public class FileParser {

	/** The token used to separate the columns (usually a single character) */
	protected String separator_ = "\t";
	/** The buffered file reader */
	protected BufferedReader reader_ = null;
	/** Line counter */
	protected int lineCounter_ = 0;
	/** Next line */
	protected String nextLine_ = null;
	
	
	// ============================================================================
	// PUBLIC METHODS
	    
	/** Constructor */
	public FileParser(String filename) {

		try {
			FileInputStream fstream = new FileInputStream(filename);
			DataInputStream in = new DataInputStream(fstream);
			reader_ = new BufferedReader(new InputStreamReader(in));
		} catch (Exception e) {
			Flynet.error(e);
		}
	}
	
	
    // ----------------------------------------------------------------------------

	/** Read and return the next line, split using the separator_. Returns null if there is no more line to read. */
	public String[] readLine() {
		
		try {
			lineCounter_++;
			nextLine_ = reader_.readLine();
		} catch (IOException e) {
			Flynet.error(e);
		}
		
		if (nextLine_ == null)
			return null;

		return nextLine_.split(separator_, -1);
	}
	

    // ----------------------------------------------------------------------------

	/** Read all the lines (see readLine()) */
	public ArrayList<String[]> readAll() {
		
		ArrayList<String[]> data = new ArrayList<String[]>();
		
		try {
			while (true) {
				nextLine_ = reader_.readLine();
				if (nextLine_ == null)
					break;
				
				lineCounter_++;
				data.add(nextLine_.split(separator_));
			}
		} catch (IOException e) {
			Flynet.error(e);
		}
		
		return data;
	}

	
    // ----------------------------------------------------------------------------

	/** Read and return the next line */
	public String readRawLine() {
		
		try {
			lineCounter_++;
			nextLine_ = reader_.readLine();
		} catch (IOException e) {
			Flynet.error(e);
		}
		
		return nextLine_;
	}
	

    // ----------------------------------------------------------------------------

	/** Be polite and close the file reader */
	public void close() {
		
		try {
			reader_.close();
		} catch (IOException e) {
			Flynet.error(e);
		}
	}
	  
	
    // ----------------------------------------------------------------------------

	/** Skip N lines (useful to skip headers) */
	public void skipLines(int N) {
		
		try {
			for (int i=0; i<N; i++)
				reader_.readLine();
		} catch (IOException e) {
			Flynet.error(e);
		}
	}

	  	
	// ============================================================================
	// PRIVATE METHODS

    
	// ============================================================================
	// SETTERS AND GETTERS

	public void setSeparator(String separator) { separator_ = separator; }
    
	public int getLineCounter() { return lineCounter_; }
		
}
