/*
 * Decompiled with CFR 0.152.
 */
package weka.core.matrix;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.Serializable;
import java.io.StreamTokenizer;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.util.Locale;
import java.util.StringTokenizer;
import java.util.Vector;
import weka.core.RevisionHandler;
import weka.core.RevisionUtils;
import weka.core.Utils;
import weka.core.matrix.CholeskyDecomposition;
import weka.core.matrix.EigenvalueDecomposition;
import weka.core.matrix.LUDecomposition;
import weka.core.matrix.LinearRegression;
import weka.core.matrix.Maths;
import weka.core.matrix.QRDecomposition;
import weka.core.matrix.SingularValueDecomposition;

public class Matrix
implements Cloneable,
Serializable,
RevisionHandler {
    private static final long serialVersionUID = 7856794138418366180L;
    protected double[][] A;
    protected int m;
    protected int n;

    public Matrix(int m, int n) {
        this.m = m;
        this.n = n;
        this.A = new double[m][n];
    }

    public Matrix(int m, int n, double s) {
        this.m = m;
        this.n = n;
        this.A = new double[m][n];
        for (int i = 0; i < m; ++i) {
            for (int j = 0; j < n; ++j) {
                this.A[i][j] = s;
            }
        }
    }

    public Matrix(double[][] A) {
        this.m = A.length;
        this.n = A[0].length;
        for (int i = 0; i < this.m; ++i) {
            if (A[i].length == this.n) continue;
            throw new IllegalArgumentException("All rows must have the same length.");
        }
        this.A = A;
    }

    public Matrix(double[][] A, int m, int n) {
        this.A = A;
        this.m = m;
        this.n = n;
    }

    public Matrix(double[] vals, int m) {
        this.m = m;
        int n = this.n = m != 0 ? vals.length / m : 0;
        if (m * this.n != vals.length) {
            throw new IllegalArgumentException("Array length must be a multiple of m.");
        }
        this.A = new double[m][this.n];
        for (int i = 0; i < m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                this.A[i][j] = vals[i + j * m];
            }
        }
    }

    public Matrix(Reader r) throws Exception {
        String line;
        LineNumberReader lnr = new LineNumberReader(r);
        int currentRow = -1;
        while ((line = lnr.readLine()) != null) {
            StringTokenizer st;
            if (line.startsWith("%") || !(st = new StringTokenizer(line)).hasMoreTokens()) continue;
            if (currentRow < 0) {
                int rows = Integer.parseInt(st.nextToken());
                if (!st.hasMoreTokens()) {
                    throw new Exception("Line " + lnr.getLineNumber() + ": expected number of columns");
                }
                int cols = Integer.parseInt(st.nextToken());
                this.A = new double[rows][cols];
                this.m = rows;
                this.n = cols;
                ++currentRow;
                continue;
            }
            if (currentRow == this.getRowDimension()) {
                throw new Exception("Line " + lnr.getLineNumber() + ": too many rows provided");
            }
            for (int i = 0; i < this.getColumnDimension(); ++i) {
                if (!st.hasMoreTokens()) {
                    throw new Exception("Line " + lnr.getLineNumber() + ": too few matrix elements provided");
                }
                this.set(currentRow, i, Double.valueOf(st.nextToken()));
            }
            ++currentRow;
        }
        if (currentRow == -1) {
            throw new Exception("Line " + lnr.getLineNumber() + ": expected number of rows");
        }
        if (currentRow != this.getRowDimension()) {
            throw new Exception("Line " + lnr.getLineNumber() + ": too few rows provided");
        }
    }

    public static Matrix constructWithCopy(double[][] A) {
        int m = A.length;
        int n = A[0].length;
        Matrix X2 = new Matrix(m, n);
        double[][] C = X2.getArray();
        for (int i = 0; i < m; ++i) {
            if (A[i].length != n) {
                throw new IllegalArgumentException("All rows must have the same length.");
            }
            for (int j = 0; j < n; ++j) {
                C[i][j] = A[i][j];
            }
        }
        return X2;
    }

    public Matrix copy() {
        Matrix X2 = new Matrix(this.m, this.n);
        double[][] C = X2.getArray();
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                C[i][j] = this.A[i][j];
            }
        }
        return X2;
    }

    public Object clone() {
        return this.copy();
    }

    public double[][] getArray() {
        return this.A;
    }

    public double[][] getArrayCopy() {
        double[][] C = new double[this.m][this.n];
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                C[i][j] = this.A[i][j];
            }
        }
        return C;
    }

    public double[] getColumnPackedCopy() {
        double[] vals = new double[this.m * this.n];
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                vals[i + j * this.m] = this.A[i][j];
            }
        }
        return vals;
    }

    public double[] getRowPackedCopy() {
        double[] vals = new double[this.m * this.n];
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                vals[i * this.n + j] = this.A[i][j];
            }
        }
        return vals;
    }

    public int getRowDimension() {
        return this.m;
    }

    public int getColumnDimension() {
        return this.n;
    }

    public double get(int i, int j) {
        return this.A[i][j];
    }

    public Matrix getMatrix(int i0, int i1, int j0, int j1) {
        Matrix X2 = new Matrix(i1 - i0 + 1, j1 - j0 + 1);
        double[][] B = X2.getArray();
        try {
            for (int i = i0; i <= i1; ++i) {
                for (int j = j0; j <= j1; ++j) {
                    B[i - i0][j - j0] = this.A[i][j];
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new ArrayIndexOutOfBoundsException("Submatrix indices");
        }
        return X2;
    }

    public Matrix getMatrix(int[] r, int[] c) {
        Matrix X2 = new Matrix(r.length, c.length);
        double[][] B = X2.getArray();
        try {
            for (int i = 0; i < r.length; ++i) {
                for (int j = 0; j < c.length; ++j) {
                    B[i][j] = this.A[r[i]][c[j]];
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new ArrayIndexOutOfBoundsException("Submatrix indices");
        }
        return X2;
    }

    public Matrix getMatrix(int i0, int i1, int[] c) {
        Matrix X2 = new Matrix(i1 - i0 + 1, c.length);
        double[][] B = X2.getArray();
        try {
            for (int i = i0; i <= i1; ++i) {
                for (int j = 0; j < c.length; ++j) {
                    B[i - i0][j] = this.A[i][c[j]];
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new ArrayIndexOutOfBoundsException("Submatrix indices");
        }
        return X2;
    }

    public Matrix getMatrix(int[] r, int j0, int j1) {
        Matrix X2 = new Matrix(r.length, j1 - j0 + 1);
        double[][] B = X2.getArray();
        try {
            for (int i = 0; i < r.length; ++i) {
                for (int j = j0; j <= j1; ++j) {
                    B[i][j - j0] = this.A[r[i]][j];
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new ArrayIndexOutOfBoundsException("Submatrix indices");
        }
        return X2;
    }

    public void set(int i, int j, double s) {
        this.A[i][j] = s;
    }

    public void setMatrix(int i0, int i1, int j0, int j1, Matrix X2) {
        try {
            for (int i = i0; i <= i1; ++i) {
                for (int j = j0; j <= j1; ++j) {
                    this.A[i][j] = X2.get(i - i0, j - j0);
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new ArrayIndexOutOfBoundsException("Submatrix indices");
        }
    }

    public void setMatrix(int[] r, int[] c, Matrix X2) {
        try {
            for (int i = 0; i < r.length; ++i) {
                for (int j = 0; j < c.length; ++j) {
                    this.A[r[i]][c[j]] = X2.get(i, j);
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new ArrayIndexOutOfBoundsException("Submatrix indices");
        }
    }

    public void setMatrix(int[] r, int j0, int j1, Matrix X2) {
        try {
            for (int i = 0; i < r.length; ++i) {
                for (int j = j0; j <= j1; ++j) {
                    this.A[r[i]][j] = X2.get(i, j - j0);
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new ArrayIndexOutOfBoundsException("Submatrix indices");
        }
    }

    public void setMatrix(int i0, int i1, int[] c, Matrix X2) {
        try {
            for (int i = i0; i <= i1; ++i) {
                for (int j = 0; j < c.length; ++j) {
                    this.A[i][c[j]] = X2.get(i - i0, j);
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new ArrayIndexOutOfBoundsException("Submatrix indices");
        }
    }

    public boolean isSymmetric() {
        int nr = this.A.length;
        int nc = this.A[0].length;
        if (nr != nc) {
            return false;
        }
        for (int i = 0; i < nc; ++i) {
            for (int j = 0; j < i; ++j) {
                if (this.A[i][j] == this.A[j][i]) continue;
                return false;
            }
        }
        return true;
    }

    public boolean isSquare() {
        return this.getRowDimension() == this.getColumnDimension();
    }

    public Matrix transpose() {
        Matrix X2 = new Matrix(this.n, this.m);
        double[][] C = X2.getArray();
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                C[j][i] = this.A[i][j];
            }
        }
        return X2;
    }

    public double norm1() {
        double f = 0.0;
        for (int j = 0; j < this.n; ++j) {
            double s = 0.0;
            for (int i = 0; i < this.m; ++i) {
                s += Math.abs(this.A[i][j]);
            }
            f = Math.max(f, s);
        }
        return f;
    }

    public double norm2() {
        return new SingularValueDecomposition(this).norm2();
    }

    public double normInf() {
        double f = 0.0;
        for (int i = 0; i < this.m; ++i) {
            double s = 0.0;
            for (int j = 0; j < this.n; ++j) {
                s += Math.abs(this.A[i][j]);
            }
            f = Math.max(f, s);
        }
        return f;
    }

    public double normF() {
        double f = 0.0;
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                f = Maths.hypot(f, this.A[i][j]);
            }
        }
        return f;
    }

    public Matrix uminus() {
        Matrix X2 = new Matrix(this.m, this.n);
        double[][] C = X2.getArray();
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                C[i][j] = -this.A[i][j];
            }
        }
        return X2;
    }

    public Matrix plus(Matrix B) {
        this.checkMatrixDimensions(B);
        Matrix X2 = new Matrix(this.m, this.n);
        double[][] C = X2.getArray();
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                C[i][j] = this.A[i][j] + B.A[i][j];
            }
        }
        return X2;
    }

    public Matrix plusEquals(Matrix B) {
        this.checkMatrixDimensions(B);
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                this.A[i][j] = this.A[i][j] + B.A[i][j];
            }
        }
        return this;
    }

    public Matrix minus(Matrix B) {
        this.checkMatrixDimensions(B);
        Matrix X2 = new Matrix(this.m, this.n);
        double[][] C = X2.getArray();
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                C[i][j] = this.A[i][j] - B.A[i][j];
            }
        }
        return X2;
    }

    public Matrix minusEquals(Matrix B) {
        this.checkMatrixDimensions(B);
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                this.A[i][j] = this.A[i][j] - B.A[i][j];
            }
        }
        return this;
    }

    public Matrix arrayTimes(Matrix B) {
        this.checkMatrixDimensions(B);
        Matrix X2 = new Matrix(this.m, this.n);
        double[][] C = X2.getArray();
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                C[i][j] = this.A[i][j] * B.A[i][j];
            }
        }
        return X2;
    }

    public Matrix arrayTimesEquals(Matrix B) {
        this.checkMatrixDimensions(B);
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                this.A[i][j] = this.A[i][j] * B.A[i][j];
            }
        }
        return this;
    }

    public Matrix arrayRightDivide(Matrix B) {
        this.checkMatrixDimensions(B);
        Matrix X2 = new Matrix(this.m, this.n);
        double[][] C = X2.getArray();
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                C[i][j] = this.A[i][j] / B.A[i][j];
            }
        }
        return X2;
    }

    public Matrix arrayRightDivideEquals(Matrix B) {
        this.checkMatrixDimensions(B);
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                this.A[i][j] = this.A[i][j] / B.A[i][j];
            }
        }
        return this;
    }

    public Matrix arrayLeftDivide(Matrix B) {
        this.checkMatrixDimensions(B);
        Matrix X2 = new Matrix(this.m, this.n);
        double[][] C = X2.getArray();
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                C[i][j] = B.A[i][j] / this.A[i][j];
            }
        }
        return X2;
    }

    public Matrix arrayLeftDivideEquals(Matrix B) {
        this.checkMatrixDimensions(B);
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                this.A[i][j] = B.A[i][j] / this.A[i][j];
            }
        }
        return this;
    }

    public Matrix times(double s) {
        Matrix X2 = new Matrix(this.m, this.n);
        double[][] C = X2.getArray();
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                C[i][j] = s * this.A[i][j];
            }
        }
        return X2;
    }

    public Matrix timesEquals(double s) {
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                this.A[i][j] = s * this.A[i][j];
            }
        }
        return this;
    }

    public Matrix times(Matrix B) {
        if (B.m != this.n) {
            throw new IllegalArgumentException("Matrix inner dimensions must agree.");
        }
        Matrix X2 = new Matrix(this.m, B.n);
        double[][] C = X2.getArray();
        double[] Bcolj = new double[this.n];
        for (int j = 0; j < B.n; ++j) {
            for (int k = 0; k < this.n; ++k) {
                Bcolj[k] = B.A[k][j];
            }
            for (int i = 0; i < this.m; ++i) {
                double[] Arowi = this.A[i];
                double s = 0.0;
                for (int k = 0; k < this.n; ++k) {
                    s += Arowi[k] * Bcolj[k];
                }
                C[i][j] = s;
            }
        }
        return X2;
    }

    public LUDecomposition lu() {
        return new LUDecomposition(this);
    }

    public QRDecomposition qr() {
        return new QRDecomposition(this);
    }

    public CholeskyDecomposition chol() {
        return new CholeskyDecomposition(this);
    }

    public SingularValueDecomposition svd() {
        return new SingularValueDecomposition(this);
    }

    public EigenvalueDecomposition eig() {
        return new EigenvalueDecomposition(this);
    }

    public Matrix solve(Matrix B) {
        return this.m == this.n ? new LUDecomposition(this).solve(B) : new QRDecomposition(this).solve(B);
    }

    public Matrix solveTranspose(Matrix B) {
        return this.transpose().solve(B.transpose());
    }

    public Matrix inverse() {
        return this.solve(Matrix.identity(this.m, this.m));
    }

    public Matrix sqrt() {
        Matrix result = null;
        EigenvalueDecomposition evd = this.eig();
        Matrix v = evd.getV();
        Matrix d = evd.getD();
        Matrix s = new Matrix(d.getRowDimension(), d.getColumnDimension());
        for (int i = 0; i < s.getRowDimension(); ++i) {
            for (int n = 0; n < s.getColumnDimension(); ++n) {
                s.set(i, n, StrictMath.sqrt(d.get(i, n)));
            }
        }
        Matrix a = v.inverse();
        Matrix b = v.times(s).inverse();
        v = null;
        d = null;
        evd = null;
        s = null;
        result = a.solve(b).inverse();
        return result;
    }

    public LinearRegression regression(Matrix y, double ridge) {
        return new LinearRegression(this, y, ridge);
    }

    public final LinearRegression regression(Matrix y, double[] w, double ridge) {
        return new LinearRegression(this, y, w, ridge);
    }

    public double det() {
        return new LUDecomposition(this).det();
    }

    public int rank() {
        return new SingularValueDecomposition(this).rank();
    }

    public double cond() {
        return new SingularValueDecomposition(this).cond();
    }

    public double trace() {
        double t = 0.0;
        for (int i = 0; i < Math.min(this.m, this.n); ++i) {
            t += this.A[i][i];
        }
        return t;
    }

    public static Matrix random(int m, int n) {
        Matrix A = new Matrix(m, n);
        double[][] X2 = A.getArray();
        for (int i = 0; i < m; ++i) {
            for (int j = 0; j < n; ++j) {
                X2[i][j] = Math.random();
            }
        }
        return A;
    }

    public static Matrix identity(int m, int n) {
        Matrix A = new Matrix(m, n);
        double[][] X2 = A.getArray();
        for (int i = 0; i < m; ++i) {
            for (int j = 0; j < n; ++j) {
                X2[i][j] = i == j ? 1.0 : 0.0;
            }
        }
        return A;
    }

    public void print(int w, int d) {
        this.print(new PrintWriter(System.out, true), w, d);
    }

    public void print(PrintWriter output, int w, int d) {
        DecimalFormat format = new DecimalFormat();
        format.setDecimalFormatSymbols(new DecimalFormatSymbols(Locale.US));
        format.setMinimumIntegerDigits(1);
        format.setMaximumFractionDigits(d);
        format.setMinimumFractionDigits(d);
        format.setGroupingUsed(false);
        this.print(output, format, w + 2);
    }

    public void print(NumberFormat format, int width) {
        this.print(new PrintWriter(System.out, true), format, width);
    }

    public void print(PrintWriter output, NumberFormat format, int width) {
        output.println();
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                String s = format.format(this.A[i][j]);
                int padding = Math.max(1, width - s.length());
                for (int k = 0; k < padding; ++k) {
                    output.print(' ');
                }
                output.print(s);
            }
            output.println();
        }
        output.println();
    }

    public static Matrix read(BufferedReader input) throws IOException {
        int j;
        StreamTokenizer tokenizer = new StreamTokenizer(input);
        tokenizer.resetSyntax();
        tokenizer.wordChars(0, 255);
        tokenizer.whitespaceChars(0, 32);
        tokenizer.eolIsSignificant(true);
        Vector<Object> v = new Vector<Object>();
        while (tokenizer.nextToken() == 10) {
        }
        if (tokenizer.ttype == -1) {
            throw new IOException("Unexpected EOF on matrix read.");
        }
        do {
            v.addElement(Double.valueOf(tokenizer.sval));
        } while (tokenizer.nextToken() == -3);
        int n = v.size();
        double[] row = new double[n];
        for (j = 0; j < n; ++j) {
            row[j] = (Double)v.elementAt(j);
        }
        v.removeAllElements();
        v.addElement(row);
        while (tokenizer.nextToken() == -3) {
            row = new double[n];
            v.addElement(row);
            j = 0;
            do {
                if (j >= n) {
                    throw new IOException("Row " + v.size() + " is too long.");
                }
                row[j++] = Double.valueOf(tokenizer.sval);
            } while (tokenizer.nextToken() == -3);
            if (j >= n) continue;
            throw new IOException("Row " + v.size() + " is too short.");
        }
        int m = v.size();
        double[][] A = new double[m][];
        v.copyInto((Object[])A);
        return new Matrix(A);
    }

    private void checkMatrixDimensions(Matrix B) {
        if (B.m != this.m || B.n != this.n) {
            throw new IllegalArgumentException("Matrix dimensions must agree.");
        }
    }

    public void write(Writer w) throws Exception {
        w.write("% Rows\tColumns\n");
        w.write("" + this.getRowDimension() + "\t" + this.getColumnDimension() + "\n");
        w.write("% Matrix elements\n");
        for (int i = 0; i < this.getRowDimension(); ++i) {
            for (int j = 0; j < this.getColumnDimension(); ++j) {
                w.write("" + this.get(i, j) + "\t");
            }
            w.write("\n");
        }
        w.flush();
    }

    public String toString() {
        double maxval = 0.0;
        boolean fractional = false;
        for (int i = 0; i < this.getRowDimension(); ++i) {
            for (int j = 0; j < this.getColumnDimension(); ++j) {
                double current = this.get(i, j);
                if (current < 0.0) {
                    current *= -11.0;
                }
                if (current > maxval) {
                    maxval = current;
                }
                double fract = Math.abs(current - Math.rint(current));
                if (fractional || !(Math.log(fract) / Math.log(10.0) >= -2.0)) continue;
                fractional = true;
            }
        }
        int width = (int)(Math.log(maxval) / Math.log(10.0) + (double)(fractional ? 4 : 1));
        StringBuffer text = new StringBuffer();
        for (int i = 0; i < this.getRowDimension(); ++i) {
            for (int j = 0; j < this.getColumnDimension(); ++j) {
                text.append(" ").append(Utils.doubleToString(this.get(i, j), width, fractional ? 2 : 0));
            }
            text.append("\n");
        }
        return text.toString();
    }

    public String toMatlab() {
        StringBuffer result = new StringBuffer();
        result.append("[");
        for (int i = 0; i < this.getRowDimension(); ++i) {
            if (i > 0) {
                result.append("; ");
            }
            for (int n = 0; n < this.getColumnDimension(); ++n) {
                if (n > 0) {
                    result.append(" ");
                }
                result.append(Double.toString(this.get(i, n)));
            }
        }
        result.append("]");
        return result.toString();
    }

    public static Matrix parseMatlab(String matlab) throws Exception {
        String cells = matlab.substring(matlab.indexOf("[") + 1, matlab.indexOf("]")).trim();
        StringTokenizer tokRow = new StringTokenizer(cells, ";");
        int rows = tokRow.countTokens();
        StringTokenizer tokCol = new StringTokenizer(tokRow.nextToken(), " ");
        int cols = tokCol.countTokens();
        Matrix result = new Matrix(rows, cols);
        tokRow = new StringTokenizer(cells, ";");
        rows = 0;
        while (tokRow.hasMoreTokens()) {
            tokCol = new StringTokenizer(tokRow.nextToken(), " ");
            cols = 0;
            while (tokCol.hasMoreTokens()) {
                result.set(rows, cols, Double.parseDouble(tokCol.nextToken()));
                ++cols;
            }
            ++rows;
        }
        return result;
    }

    public String getRevision() {
        return RevisionUtils.extract("$Revision: 1.8 $");
    }

    public static void main(String[] args) {
        try {
            System.out.println("\nIdentity\n");
            Matrix I = Matrix.identity(3, 5);
            System.out.println("I(3,5)\n" + I);
            System.out.println("\nbasic operations - square\n");
            Matrix A = Matrix.random(3, 3);
            Matrix B = Matrix.random(3, 3);
            System.out.println("A\n" + A);
            System.out.println("B\n" + B);
            System.out.println("A'\n" + A.inverse());
            System.out.println("A^T\n" + A.transpose());
            System.out.println("A+B\n" + A.plus(B));
            System.out.println("A*B\n" + A.times(B));
            System.out.println("X from A*X=B\n" + A.solve(B));
            System.out.println("\nbasic operations - non square\n");
            A = Matrix.random(2, 3);
            B = Matrix.random(3, 4);
            System.out.println("A\n" + A);
            System.out.println("B\n" + B);
            System.out.println("A*B\n" + A.times(B));
            System.out.println("\nsqrt (1)\n");
            A = new Matrix(new double[][]{{5.0, -4.0, 1.0, 0.0, 0.0}, {-4.0, 6.0, -4.0, 1.0, 0.0}, {1.0, -4.0, 6.0, -4.0, 1.0}, {0.0, 1.0, -4.0, 6.0, -4.0}, {0.0, 0.0, 1.0, -4.0, 5.0}});
            System.out.println("A\n" + A);
            System.out.println("sqrt(A)\n" + A.sqrt());
            System.out.println("\nsqrt (2)\n");
            A = new Matrix(new double[][]{{7.0, 10.0}, {15.0, 22.0}});
            System.out.println("A\n" + A);
            System.out.println("sqrt(A)\n" + A.sqrt());
            System.out.println("det(A)\n" + A.det() + "\n");
            System.out.println("\nEigenvalue Decomposition\n");
            EigenvalueDecomposition evd = A.eig();
            System.out.println("[V,D] = eig(A)");
            System.out.println("- V\n" + evd.getV());
            System.out.println("- D\n" + evd.getD());
            System.out.println("\nLU Decomposition\n");
            LUDecomposition lud = A.lu();
            System.out.println("[L,U,P] = lu(A)");
            System.out.println("- L\n" + lud.getL());
            System.out.println("- U\n" + lud.getU());
            System.out.println("- P\n" + Utils.arrayToString(lud.getPivot()) + "\n");
            System.out.println("\nRegression\n");
            B = new Matrix(new double[][]{{3.0}, {2.0}});
            double ridge = 0.5;
            double[] weights = new double[]{0.3, 0.7};
            LinearRegression lr = A.regression(B, ridge);
            System.out.println("A\n" + A);
            System.out.println("B\n" + B);
            System.out.println("ridge = " + ridge + "\n");
            System.out.println("weights = " + Utils.arrayToString(weights) + "\n");
            System.out.println("A.regression(B, ridge)\n" + A.regression(B, ridge) + "\n");
            System.out.println("A.regression(B, weights, ridge)\n" + A.regression(B, weights, ridge) + "\n");
            System.out.println("\nWriter/Reader\n");
            StringWriter writer = new StringWriter();
            A.write(writer);
            System.out.println("A.write(Writer)\n" + writer);
            A = new Matrix(new StringReader(writer.toString()));
            System.out.println("A = new Matrix.read(Reader)\n" + A);
            System.out.println("\nMatlab-Format\n");
            String matlab = "[ 1   2;3 4 ]";
            System.out.println("Matlab: " + matlab);
            System.out.println("from Matlab:\n" + Matrix.parseMatlab(matlab));
            System.out.println("to Matlab:\n" + Matrix.parseMatlab(matlab).toMatlab());
            matlab = "[1 2 3 4;3 4 5 6;7 8 9 10]";
            System.out.println("Matlab: " + matlab);
            System.out.println("from Matlab:\n" + Matrix.parseMatlab(matlab));
            System.out.println("to Matlab:\n" + Matrix.parseMatlab(matlab).toMatlab() + "\n");
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }
}

