/**
 * Copyright 2003-2005 Barbara Engelhardt (bee@cs.berkeley.edu)
 * 
 * A single protein.
 * Used in each node of the reconciled tree.
 * 
 * @author Barbara Engelhardt
 */

package util;

import java.io.PrintStream;
import java.util.Vector;

public class PfunProtein {

    private String id;
    private String name;
    private String function;
    private String origin;
    private String ecNum;
    private String pFam;
    private Vector goNumbers;
    private Vector mocs;
    private Vector goReal;
    private String species;
    private String sequence;
    //private Hashtable fnProbabilities;
    private double[] fnLikelihoods;
    private Vector functionRatio;
    
    public PfunProtein (String idN) {
	id = idN;
	name = null;
	function = null;
	origin = null;
	ecNum = null;
	pFam = null;
	goNumbers = null;
	mocs = null;
	goReal = null;
	species = null;
	sequence = null;
	//fnProbabilities = null;
	fnLikelihoods = null;		
	functionRatio = null;
    }
    
    public String getID() {
	return id;
    }
    
    public String getName() {
	return name;
    }
    
    public void setName(String nameN) {
	if(nameN != null || name == null) {
	    name = nameN;
	}
    }
    
    public String getFunction() {
	return function;
    }
    
    public void addFunction(String moreFunction) {
	if(function == null) {
	    function = new String(moreFunction);
	} else {
	    function += (" "+moreFunction);
	}
    }
    
    public String getOrigin() {
	return origin;
    }
    
    public void setOrigin(String o)
    {
	origin = o;	
    }
    
    public String getSequence() {
	return sequence;
    }
    
    public void addSequence(String moreSequence) {
	if(sequence == null) {
	    sequence = new String(moreSequence);
	} else {
	    sequence += (moreSequence);
	}
    }
    
    public void addLikelihoods(double[] likelihoods)
    {
	fnLikelihoods = likelihoods;
    }

    public void addRatioLikelihoods()
    {
	if(functionRatio == null) return;
	fnLikelihoods = new double[functionRatio.size()];
	double total = 0;
	for(int i = 0; i < fnLikelihoods.length; i++) {
	    fnLikelihoods[i] = 
		((Double)functionRatio.elementAt(i)).doubleValue();
	    total += fnLikelihoods[i];
	}
	for(int i = 0; i < fnLikelihoods.length; i++) {
	    fnLikelihoods[i] = fnLikelihoods[i]/total;
	}
	
    }
    
    public double[] getLeafLikelihoods()
    {
	return fnLikelihoods;	
    }
    
    public String getECNumber() {
	return ecNum;
    }
    
    public void setECNumber(String ecNumN) {
	if(ecNumN != null || ecNum == null) {
	    ecNum = ecNumN;
	}
    }
    
    public String getSpecies() {
	return species;
    }
    
    public void setSpecies(String speciesN) {
	if(speciesN != null || species == null) {
	    species = speciesN;
	}
    }
    
    public void setGOReal(Vector gor) {
	if(goReal == null) {
	    goReal = new Vector();
	}
	goReal.addAll(gor);
    }
    
    public Vector getGOReal()
    {
	return goReal;
    }
    
    public void setFunctionRatio(Vector fnRatio)
    {
	if(functionRatio == null) 
	    functionRatio = new Vector();
	functionRatio.addAll(fnRatio);
    }

    public Vector getFunctionRatio()
    {
	return functionRatio;
    }

    public String getPFam() {
	return pFam;
    }
    
    public void setPFam(String pFamN) {
	if(pFamN != null || pFam == null) {
	    pFam = pFamN;
	}
    }
    
    public Vector getGONumber() {
	return goNumbers;
    }
    
    public void setGONumber(String goN) {
	if(goNumbers == null) {
	    goNumbers = new Vector();
	}
	if(goN != null) {
	    goNumbers.add(goN);
	}
    }
    
    public void setGONumbers(Vector goNs) {
	//if(name.indexOf("SCHPO") == -1) // Added for FUNGI experiments
	    goNumbers = goNs;
    }
    
    // MOC = method of categorization
    public Vector getMOC() {
	return mocs;
    }
    
    public void setMOC(String mocN) {
	if(mocs == null) {
			mocs = new Vector();
	}
	if(mocN != null) {
	    mocs.add(mocN);
	}
    }
    
    public void setMOCs(Vector mocNs) {
	//if(name.indexOf("SCHPO") == -1) // Added for FUNGI experiments
	    mocs = mocNs;
    }
    
    public boolean hasGORealAnnotation(Integer function)
    {
	if(getGOReal() == null) return false;
	Vector gra = getGOReal();
	if(gra.contains(function)) return true;
	return false;
    }
    
    public int numGORealAnnotation(int function)
	{
	    if(getGOReal() == null) return 0;
		return (getGOReal().size());
	}
    
    public boolean hasAnnotation(Integer function, String mocLevel)
    {
	if(getGONumber() == null) return false;
	Vector gon = getGONumber();
	Vector moc = getMOC();
	if(gon.contains(function) && moc.contains(mocLevel)) {
	    for(int i = 0; i < gon.size(); i++) {
		if(gon.elementAt(i) == function 
		   && ((String)moc.elementAt(i)).equals(mocLevel)) {
		    return true;
		}
			}
	}
	return false;
    }
    
    public boolean hasIDAorIMPEvidence()
    {
	Vector moc = getMOC();
	return(moc.contains("IDA") || moc.contains("IMP") 
	       || moc.contains("GOR"));
    }

    public boolean hasProperEvidence(Settings settings)
    {
	Vector moc = getMOC();
	boolean setp = ((Boolean)settings.getSetting("iea")).booleanValue();
	if(setp && moc.contains("IEA")) return true;
	setp = ((Boolean)settings.getSetting("ida")).booleanValue();
	if(setp && moc.contains("IDA")) return true;
	setp = ((Boolean)settings.getSetting("imp")).booleanValue();
	if(setp && moc.contains("IMP")) return true;
	setp = ((Boolean)settings.getSetting("tas")).booleanValue();
	if(setp && moc.contains("TAS")) return true;
	//setp = ((Boolean)settings.getSetting("iss")).booleanValue();
	//if(setp && moc.contains("ISS")) return true;
	return false;
    }


    // bee-- add in list of possible functions
    public boolean existsAnnotation(Integer function, String mocLevel)
    {
	if(getGONumber() == null) return false;
	Vector moc = getMOC();
	if(moc.contains(mocLevel)) {
			return true;
	}
	return false;
    }
    
    public void printProteinToFile(PrintStream fout) {
	printProteinToFile(fout, null);
    }
    
    public void printProteinToFile(PrintStream fout, String alignment) {
	
	fout.println("\t<Protein>");
	if(getName() != null) 
	    fout.println("\t\t<ProteinName>"+getName()+"</ProteinName>");
	if(getID() != null) 
	    fout.println("\t\t<ProteinNumber>"+getID()+"</ProteinNumber>");
	if(getOrigin() != null) 
	    fout.println("\t\t<ProteinLocation>"+getOrigin()+"</ProteinLocation>");
	if(getSpecies() != null)
				fout.println("\t\t<SpeciesName>"+getSpecies()+"</SpeciesName>");
	if(getECNumber() != null) 
	    fout.println("\t\t<ECNumber>"+getECNumber()+"</ECNumber>");
	if(getPFam() != null)
	    fout.println("\t\t<PFamNumber>"+getPFam()+"</PFamNumber>");
	if(getGONumber() != null)
	    fout.println("\t\t<GONumber>"+getGONumber()+"</GONumber>");
	if(getMOC() != null)
	    fout.println("\t\t<MOC>"+getMOC()+"</MOC>");
	if(getSequence() != null)
	    fout.println("\t\t<Sequence>"+getSequence()+"</Sequence>");
	if(alignment != null)
	    fout.println("\t\t<Alignment>"+alignment+"</Alignment>");
	if(getFunction() != null)
	    fout.println("\t\t<Function>"+getFunction()+"</Function>");
	fout.println("\t</Protein>");
	
    }
}
