/**
 * Data structure for the GO (gene ontology) DAG.
 *
 * Copyright 2003-2005 Barbara Engelhardt (bee@cs.berkeley.edu)
 * @author Barbara Engelhardt
 */

package util;

import java.util.*;
//import org.apache.regexp.*;
import java.io.*;
import stat.PFunTransMatrix;

public class PFunGODAG {
    
    // Denotes whether these members of the proteins
    // should be pulled out of the SwissProt DB.
    //private static boolean nameP;
    private static String goFileName;
    
    private static Vector functionList; // list of most recent pnums 
    //parents at each level.
    private static PFunDAG functions;   // DAG of functions
    private static int lineno;
    private static Entry currentEntry;
    //private static PfunProtein currentNode;
    private static double rValue;
    private static Vector leafList;
    private static PFunTransMatrix pfx;
    
    private Settings settings;
    private boolean verbose;
    
    

    public PFunGODAG(String fileName, boolean pName, 
		     boolean pGo, boolean pMoc)
    {
        functionList = new Vector();
        functions = new PFunDAG();
        lineno = 0;
        goFileName = fileName;
        rValue = (double) -1.0;
        leafList = null;
        pfx = null;
    }
    
    public void processRegexResults (String s) 
    {
	getEntry(s, true);
    }
    
    // Called ONLY by processRegexResults(s)
    // I think this basically parses things.
    public Entry getEntry(String s, boolean levelP)
    {
        // It looks like there's a bunch of parsing going on here...
        // PFunDAG functions: seems to add it stack-like to 
	// "functionList" and then
        // really really add it to PFunDAG.
        int level = -1;
        
        if(levelP) {  // seems to ALWAYS be called :)
            int level2;
            level = s.indexOf('%')-1;
            level2 = s.indexOf('<')-1;
            if((level2 > level && (level2 >= 0 && level < 0)) 
	       || (level2 < level && level2 >= 0 && level >= 0)) {
                level = level2;
            }
            if(level < 0) {
                //if (verbose) System.out.println("Error: no percents or" 
		//	+ " less thans in GO ontology line: "+lineno);
		return null;
	    }
        }
        s = s.substring(level+2, s.length());
        s = pullOneEntry(s, level);
        Entry e = currentEntry;
        // add parent as previous level
        Entry p = null;
        if(functionList.size() >= level && level-1 >= 0) 
	    p = (Entry)functionList.get(level-1);
        if(p != null) functions.addParent(e.getNumber(), p.getNumber());
        // add as new level
        functionList.add(level, e);
        
        // if any more %, create new, add as parent.		
        while(s.indexOf('%') >= 0 || s.indexOf('<') >= 0) {
            if(s.indexOf('%') >= 0) {
                s = pullOneEntry(s.substring(s.indexOf('%')+2), level-1);
                p = currentEntry;
            } else {
                s = pullOneEntry(s.substring(s.indexOf('<')+2), level-1);
                p = currentEntry;
            }
            if(p != null) functions.addParent(e.getNumber(), p.getNumber());
        }
        return e;
    }
    
    private String pullOneEntry(String s, int level)
    {
        Entry e = null;
        int semicolon = s.indexOf(';');
        String name = s.substring(0, semicolon-1);
        s = s.substring(semicolon+1, s.length());
        semicolon = s.indexOf(':');
        int id = new Integer(s.substring(semicolon+1, semicolon+8)).intValue();
        s = s.substring(semicolon+8);
        // not parsing EC numbers here, but could!
        // check if its already been seen.
        e = new Entry(id, name, level);
        if(!functions.contains(e.getNumber())) {
            functions.addNode(e.getName(), e.getNumber());
        }
        functions.addLevel(e.getNumber(), level);
        if(e != null) currentEntry = e;
        return s;
        
    }
    
    // Often called by PFunLibrary.java's buildDatasetGODAG().
    // Basically, it takes the .ont file and puts it in its own data structure.
    public void fillOutList()
    {
        String str = null;
        BufferedReader fin; 
        
        try {
            fin = new BufferedReader(new FileReader (goFileName));  
            
            try {
		// Read each line of the .ont Gene Ontology filename
                while ((str=fin.readLine()) != null) {
                    lineno++;
		    // If the line isn't blank and it's not a comment,
                    if(str.length() > 0 && !str.startsWith("!")) { 
			// if it matches a regular expression, 
                        processRegexResults(str);                    
                    }
                }
            }
            catch (Exception e) {
                // TODO: Why is this exception generalized?
                System.out.println("Error in fillOutList: " 
				   + str + " " + lineno + " " 
				   + e.getMessage());
                System.exit(1);
            }
            fin.close();
        }
        catch (Exception ioe) {
            // TODO: Why is this exception generalized? 
	    // I think it's for FileNotFoundError.
            System.err.println("fillOutList: " + goFileName + " " +
			       ioe.getMessage()); 
	    // TODO: Make error message friendlier.
            System.exit(1); // TODO: Have this propagate upwards. 
	    //In a GUI, I can't have this completely bring down the program.
        }
        //functions.printOutDAG();
    }
    
    
    //////////////////////////////////////////////////////////
    //  Statistics-Gathering section                        //
    //////////////////////////////////////////////////////////
    
    // Sent a vector with functions, a vector with experimental methods,
    // this function tallies all the counts from the induced subgraph
    // of each of the functions, and associated counts for each
    // experimental method.
    public void tallyObservedFunctions(Vector fns, Vector methods)
    {
        String m;
        int fn;
        for(int i = 0; i < fns.size(); i++) {
            fn = ((Integer)fns.elementAt(i)).intValue();
            m = null;
            if(methods.size() > i) m = (String)methods.get(i);
            if(m == null) {
                if (verbose) System.out.println("Function without a method: "
						+ fn);
            }
            tallyOneFunction(fn, m);
        }		
    }
    
    private void tallyOneFunction(int fn, String m)
    {    	
        Vector isg = new Vector();
        isg = functions.getInducedSubgraph(fn, isg);
        for(int i = 0; i < isg.size(); i++) {
            fn = ((Integer)isg.elementAt(i)).intValue();
            if(!functions.contains(fn)) { // should this be containskey?
                if (verbose) 
		    System.out.println("FunctionNode does not exist: " + fn);
            }
            Vector ex = ((Vector)functions.getAdditional(fn));
            //System.out.println(fn);
            if(ex == null) {
                ex = new Vector(15);
                for(int j = 0; j < 15; j++) ex.add(new Integer(0));
            }
            if(m.equals("IEA")) 
		ex.set(0,new Integer(((Integer)ex.elementAt(0)).intValue()+1));
            else if(m.equals("IMP")) 
		ex.set(1,new Integer(((Integer)ex.elementAt(1)).intValue()+1));
            else if(m.equals("IGI")) 
		ex.set(2,new Integer(((Integer)ex.elementAt(2)).intValue()+1));
            else if(m.equals("IPI")) 
		ex.set(3,new Integer(((Integer)ex.elementAt(3)).intValue()+1));
            else if(m.equals("ISS")) 
		ex.set(4,new Integer(((Integer)ex.elementAt(4)).intValue()+1));
            else if(m.equals("IDA")) 
		ex.set(5,new Integer(((Integer)ex.elementAt(5)).intValue()+1));
            else if(m.equals("IEP")) 
		ex.set(6,new Integer(((Integer)ex.elementAt(6)).intValue()+1));
            else if(m.equals("IEA")) 
		ex.set(7,new Integer(((Integer)ex.elementAt(7)).intValue()+1));
            else if(m.equals("TAS")) 
		ex.set(8,new Integer(((Integer)ex.elementAt(8)).intValue()+1));
            else if(m.equals("NAS")) 
		ex.set(9,new Integer(((Integer)ex.elementAt(9)).intValue()+1));
            else if(m.equals("RCA")) 
		ex.set(10,new Integer(((Integer)ex.elementAt(10)).intValue()+1));
            else if(m.equals("IGC")) 
		ex.set(11,new Integer(((Integer)ex.elementAt(11)).intValue()+1));
            else if(m.equals("ND")) 
		ex.set(12,
		       new Integer(((Integer)ex.elementAt(12)).intValue()+1));
            else if(m.equals("IC")) 
		ex.set(13,
		       new Integer(((Integer)ex.elementAt(13)).intValue()+1));
            else if(m.equals("P")) 
		ex.set(14,
		       new Integer(((Integer)ex.elementAt(14)).intValue()+1));
            else if(m.equals("E")) 
		ex.set(15,
		       new Integer(((Integer)ex.elementAt(15)).intValue()+1));
            else if(m.equals("NR")) 
		ex.set(16,
		       new Integer(((Integer)ex.elementAt(16)).intValue()+1));
            else if(m.equals("GOR")) 
		ex.set(17,
		       new Integer(((Integer)ex.elementAt(17)).intValue()+1));
            else 
                if (verbose) 
		    System.out.println("Methods not recognized: "+m);
            functions.addAdditional(fn, ex.clone());
        }
    }
    
    // prints this out into a nice, matlab-readable file
    public void printOutGOTally(String fileName)
    {
        PrintStream fout;
        fileName = fileName+".pdt";
        
        try {
            fout =  new PrintStream(new FileOutputStream(new File(fileName)));
            
            Enumeration pnames = functions.keys();
            while(pnames.hasMoreElements()) {
                int pfName = ((Integer)pnames.nextElement()).intValue();
                int total = 0;
                if(!functions.contains(pfName)) {
                    if (verbose) 
			System.out.println("FunctionNode does not exist: "
					   + pfName);
                }		
                int level = functions.getLevel(pfName);
                int pCount = functions.getNumParents(pfName);
                int cCount = functions.getNumChildren(pfName);
                Vector ex = (Vector)functions.getAdditional(pfName);
                if(ex == null) {
                    fout.println(pfName+" "+level+" "+pCount+" "
				 +cCount+" 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0");
                } else {
                    fout.print(pfName+" "+level+" "+pCount+" "+cCount+" ");
                    for(int i = 0; i < 15; i++) {
                        fout.print(ex.elementAt(i)+" ");
                        total += ((Integer)ex.elementAt(i)).intValue();
                    }
                    fout.println(total);
                }
            }
            fout.close();
        }
        catch (Exception ioe) {
            System.err.println("PrintOutGOTally: " + fileName + " " +
			       ioe.getMessage());
            System.exit(1);
        }	
    }
    
    
    //////////////////////////////////////////////////
    // Ontology building section                    //
    //////////////////////////////////////////////////
    
    public void tallyInducedFunctions(Vector fns, Vector ms)
    {	
        String m;
        int fn;
        for(int i = 0; i < fns.size(); i++) {
            fn = ((Integer)fns.elementAt(i)).intValue();
            m = null;
            if(ms.size() > i) m = (String)ms.get(i);
            if(m == null) {
                if (verbose) 
		    System.out.println("Function without a method: "+fn);
            }
            Vector isg = functions.getInducedSubgraph(fn);
            Vector dg = functions.getDescendantGraph(fn, false);
            isg.addAll(dg);
            for(int j = 0; j < isg.size(); j++) {
                fn = ((Integer)isg.elementAt(j)).intValue();
                if(!functions.contains(fn)) { // should this be containskey?
                    if (verbose) 
			System.out.println("FunctionNode does not exist: "+fn);
                }
                Integer ex = (Integer)functions.getAdditional(fn);
                //System.out.println(fn);
                if(ex == null) {
                    ex = new Integer(1);
                } else {
                    ex = new Integer(ex.intValue()+1);
                }
                functions.addAdditional(fn, ex);
            }
        }
    }
    
    // Only ever called from PFunLibrary.buildDatasetGODAG().
    // fns - vectorlist of all of the GO function codes from the XML file.
    // ms - vectorlist of all of the methods from the XML file.
    public void tallyFunctions(Vector fns, Vector ms)
    {	
        String m;
        int fn;
        for(int i = 0; i < fns.size(); i++) {
	    // fn - function
            fn = ((Integer)fns.elementAt(i)).intValue();
	    // m  - method
            m = null;       
            if(ms.size() > i) {
                m = (String)ms.get(i);
            }
            if(m == null) {
                if (verbose) 
		    System.out.println("Function without a method: "+fn);
            }
            else if(hasProperBackgroundSettings(m)) {	
                Integer ex = (Integer)functions.getAdditional(fn);   
                //if (verbose) System.out.println("Adding function: "+fn);
                if(ex == null) {
		    // add ...
                    ex = new Integer(1);
                } else {
                    ex = new Integer(ex.intValue()+1);
                }
		// add this additional function, this is a count
                functions.addAdditional(fn, ex);                     
                //System.out.print(ex + " ");
            }                                                        
	    // so it goes through and puts a count on the GO DAG 
	    // which proteins are used.
        }
    }
    
    private boolean hasProperBackgroundSettings(String m)
    {
	boolean setb = ((Boolean)settings.getSetting("bg")).booleanValue();
	boolean setp = ((Boolean)settings.getSetting("iea")).booleanValue();
	if(setp && m.equalsIgnoreCase("IEA") && !setb) return true;
	setp = ((Boolean)settings.getSetting("ida")).booleanValue();
	if(setp && m.equalsIgnoreCase("IDA")) return true;
	setp = ((Boolean)settings.getSetting("imp")).booleanValue();
	if(setp && m.equalsIgnoreCase("IMP")) return true;
	setp = ((Boolean)settings.getSetting("tas")).booleanValue();
	if(setp && m.equalsIgnoreCase("TAS")) return true;
	setp = ((Boolean)settings.getSetting("nas")).booleanValue();
	if(setp && m.equalsIgnoreCase("NAS") && !setb) return true;
	setp = ((Boolean)settings.getSetting("igi")).booleanValue();
	if(setp && m.equalsIgnoreCase("IGI")) return true;
	setp = ((Boolean)settings.getSetting("ipi")).booleanValue();
	if(setp && m.equalsIgnoreCase("IPI")) return true;
	setp = ((Boolean)settings.getSetting("rca")).booleanValue();
	if(setp && m.equalsIgnoreCase("RCA") && !setb) return true;
	setp = ((Boolean)settings.getSetting("ic")).booleanValue();
	if(setp && m.equalsIgnoreCase("IC") && !setb) return true;
	setp = ((Boolean)settings.getSetting("iss")).booleanValue();
	if(setp && m.equalsIgnoreCase("ISS") && !setb) return true;
	setp = ((Boolean)settings.getSetting("igc")).booleanValue();
	if(setp && m.equalsIgnoreCase("IGC") && !setb) return true;
	setp = ((Boolean)settings.getSetting("iep")).booleanValue();
	if(setp && m.equalsIgnoreCase("IEP") && !setb) return true;
	return false;
    }

    private boolean hasProperSettings(String m)
    {
	boolean setp = ((Boolean)settings.getSetting("iea")).booleanValue();
	if(setp && m.equalsIgnoreCase("IEA")) return true;
	setp = ((Boolean)settings.getSetting("ida")).booleanValue();
	if(setp && m.equalsIgnoreCase("IDA")) return true;
	setp = ((Boolean)settings.getSetting("imp")).booleanValue();
	if(setp && m.equalsIgnoreCase("IMP")) return true;
	setp = ((Boolean)settings.getSetting("tas")).booleanValue();
	if(setp && m.equalsIgnoreCase("TAS")) return true;
	setp = ((Boolean)settings.getSetting("nas")).booleanValue();
	if(setp && m.equalsIgnoreCase("NAS")) return true;
	setp = ((Boolean)settings.getSetting("ipi")).booleanValue();
	if(setp && m.equalsIgnoreCase("IPI")) return true;
	setp = ((Boolean)settings.getSetting("rca")).booleanValue();
	if(setp && m.equalsIgnoreCase("RCA")) return true;
	setp = ((Boolean)settings.getSetting("ic")).booleanValue();
	if(setp && m.equalsIgnoreCase("IC")) return true;
	setp = ((Boolean)settings.getSetting("igi")).booleanValue();
	if(setp && m.equalsIgnoreCase("IGI")) return true;
	setp = ((Boolean)settings.getSetting("iss")).booleanValue();
	if(setp && m.equalsIgnoreCase("ISS")) return true;
	return false;
    }

    public void tallyFunctions(Vector fns)
    {	
        int fn;
        for(int i = 0; i < fns.size(); i++) {
            fn = ((Integer)fns.elementAt(i)).intValue();
            Integer ex = (Integer)functions.getAdditional(fn);
            //System.out.println(fn);
            if(ex == null) {
                ex = new Integer(1);
            } else {
                ex = new Integer(ex.intValue()+1);
            }
            functions.addAdditional(fn, ex);
        }
    }
    
    public void tallyInducedFunctions(Vector fns)
    {	
        int fn;
        for(int i = 0; i < fns.size(); i++) {
            fn = ((Integer)fns.elementAt(i)).intValue();
            Vector isg = functions.getInducedSubgraph(fn);
            Vector dg = functions.getDescendantGraph(fn, false);
            isg.addAll(dg);
            for(int j = 0; j < isg.size(); j++) {
                fn = ((Integer)isg.elementAt(j)).intValue();
                if(!functions.contains(fn)) { // should this be containskey?
                    if (verbose) 
			System.out.println("FunctionNode does not exist: "+fn);
                }
                Integer ex = (Integer)functions.getAdditional(fn);
                //System.out.println(fn);
                if(ex == null) {
                    ex = new Integer(1);
                } else {
                    ex = new Integer(ex.intValue()+1);
                }
                functions.addAdditional(fn, ex);
            }
        }
    }
    
    public void pruneZeroHits()
    {
        Enumeration pnames = functions.keys();
        while(pnames.hasMoreElements()) {
            int pfName = ((Integer)pnames.nextElement()).intValue();
            if(!functions.contains(pfName)) {
                if (verbose) 
		    System.out.println("FunctionNode does not exist: "+pfName);
            }		
            Integer ex = (Integer)functions.getAdditional(pfName);	
            if(ex == null) {
                functions.removeNode(pfName);
            }
        }
    }
    
    // only called from PFunLibrary.java's buildDatasetGODAG 
    // family of functions.
    public void pruneZeroHitsAndLeaves()
    {
        Vector leaves = functions.getAllLeaves();
        for(int i = 0; i < leaves.size(); i++) {
            int pfName = ((Integer)leaves.elementAt(i)).intValue();
            if(!functions.contains(pfName)) {
                if (verbose) 
		    System.out.println("FunctionNode does not exist: "+pfName);
            }		
            Integer ex = (Integer)functions.getAdditional(pfName);	
            if(ex == null) {
                Vector p = functions
		    .getParents(((Integer)leaves.elementAt(i)).intValue());
                functions.removeNode(pfName);
                if(p != null) {
                    for(int j = 0; j < p.size(); j++) {
                        int node = ((PFunDAG.Node)p.elementAt(j)).getId();
                        if(node > -1 && functions.getNumChildren(node) < 1) {
                            leaves.add(new Integer(node));
                        }
                    }
                }
            }
        }
	// If there's only one function, add another
	// so that inference proceeds normally.
	if(leaves.size() == 1) {
	    functions.addNode("NOFUNCTION", 0);
	    Vector p = ((PFunDAG.Node)leaves.elementAt(0)).getParents();
	    if(p != null && p.size() > 0) { 
		functions.addParent(0, ((Integer)p.elementAt(0)).intValue());
	    }
	}
    }

    /* If the family has a singleton leaf in the
     * GO DAG, add a new "other" leaf as a sibling
     * of the singleton leaf, run normally */
    public void padSingletonLeaves()
    {
	if(functions.getNumLeaves() == 0) {
	    System.out.println("************** Error **************");
	    System.out.println("No ontology terms matching those found in the ontology file. Make sure that (a) ontology type matches and (b) .pli file contains at least one ontology term");
	    System.exit(1);
	}
	if(functions.getNumLeaves() > 1) return;
	
    }

    // prints this out into a nice, matlab-readable file
    public void printOutInducedTally(String fileName)
    {
        PrintStream fout;
        fileName = fileName+".pdt";
        
        try {
            fout =  new PrintStream(new FileOutputStream(new File(fileName)));
            Enumeration pnames = functions.keys();
            while(pnames.hasMoreElements()) {
                int pfName = ((Integer)pnames.nextElement()).intValue();
                if(!functions.contains(pfName)) {
                    if (verbose) 
			System.out.println("FunctionNode does not exist: "
					   +pfName);
                }		
                int level = functions.getLevel(pfName);
                int pCount = functions.getNumParents(pfName);
                int cCount = functions.getNumChildren(pfName);
                Integer ex = (Integer)functions.getAdditional(pfName);
                if(ex == null) {
                    fout.println(pfName+" "+level+" "+pCount+" "+cCount+" 0");
                } else {
                    fout.println(pfName+" "+level+" "
				 +pCount+" "+cCount+" "+ex);
                }
            }
            fout.close();
        }
        catch (Exception ioe) {
            System.err.println("PrintOutGOTally: " + fileName + " " +
			       ioe.getMessage());
            System.exit(1);
        }	
    }
    
    // prints this out into a nice, matlab-readable file
    public void printOutInducedLeafTally(String fileName)
    {
        PrintStream fout;
        fileName = fileName+".pdt";
        
        try {
            fout =  new PrintStream(new FileOutputStream(new File(fileName)));
            Vector leaves = functions.getAllLeaves();
            for(int i = 0; i < leaves.size(); i++) {
                int pfName = ((Integer)leaves.elementAt(i)).intValue();
                if(!functions.contains(pfName)) {
                    if (verbose) 
			System.out.println("FunctionNode does not exist: "
					   + pfName);
                }		
                int level = functions.getLevel(pfName);
                int pCount = functions.getNumParents(pfName);
                double prior = functions.getPrior(pfName);
                double posterior = functions.getPosterior(pfName);
                double likelihood = functions.getLikelihood(pfName);
                fout.println(pfName+" "+i+" "+level+" "
			     +pCount+" "+prior+" "+posterior+" "+likelihood);
            }
            fout.close();
        }
        catch (Exception ioe) {
            System.err.println("PrintOutGOTally: " + fileName + " " +
			       ioe.getMessage());
            System.exit(1);
        }	
    }
    
    
    // prints this out into a nice, matlab-readable file
    public void printOutInducedProbabilities(String fileName)
    {
        PrintStream fout;
        fileName = fileName+".pdt";
        
        try {
            fout =  new PrintStream(new FileOutputStream(new File(fileName)));
            Enumeration pnames = functions.keys();
            while(pnames.hasMoreElements()) {
                int pfName = ((Integer)pnames.nextElement()).intValue();
                if(!functions.contains(pfName)) {
                    if (verbose) 
			System.out.println("FunctionNode does not exist: "
					   + pfName);
                }		
                int level = functions.getLevel(pfName);
                int pCount = functions.getNumParents(pfName);
                int cCount = functions.getNumChildren(pfName);
                double prior = functions.getPrior(pfName);
                double posterior = functions.getPosterior(pfName);
                double likelihood = functions.getLikelihood(pfName);
                double evidence = functions.getEvidence(pfName);
                int leaves = functions.getNumDescendantLeaves(pfName);
                fout.println(pfName+" "+level+" "
			     +pCount+" "+cCount+" "+" "
			     +leaves+" "+prior+" "
			     +likelihood+" "+posterior+" "+evidence);
            }
            fout.close();
        }
        catch (Exception ioe) {
            System.err.println("PrintOutGOTally: " + fileName + " " +
			       ioe.getMessage());
            System.exit(1);
        }	
    }
    
    // calculates r value from paper.
    // According to Matlab, this calculation gets flakey
    // if leaves < 20. Anything above that and it is 
    // accurate up to approx 5 digits.
    public void findRValue()
    {
        // The additional one is the "sluff" variable
        // connected to all nodes (counts as single extra leaf).
        int leaves = functions.getNumLeaves();
        double r = 1;
        for (int i = 2; i <= leaves; i++) {
            double t = (double) 2.2605 * (double)((i*i)-i);
            r = r + ((double)1.0/t);
        }
        rValue = r*(double)leaves;	
        if (verbose) System.out.println("Multiplier: "+r
					+ ", Leaves: "+leaves
					+ ", R-value: "+rValue);
        Vector tempLeaves = functions.getAllLeaves();
        for(int i = 0; i < tempLeaves.size(); i++){
            if (verbose) System.out.println("Leaf: "+tempLeaves.elementAt(i));
        }
        if(functions != null) functions.setRValue(rValue);
    }
    
    // Accessor function
    public double getRValue()
    {
        if(rValue < 0) findRValue();
        return rValue;	
    }
    
    public double getPriorProbability(int fn)
    {
        return functions.getPrior(fn);	
    }
    
    // Given a DAG, take all evidence from gene ontology file 
    // with annotations for specific proteins. put all annotations into leaves.
    public void incorporateEvidenceShort(Vector fs, Vector ms, boolean noIEA)
    {
        // First put actual evidence in the appropriate positions.
        for(int i = 0; i < fs.size(); i++) {
            int pfName = ((Integer)fs.elementAt(i)).intValue();
            String method = (String)ms.elementAt(i);
            if(functions.contains(pfName) 
	       && hasProperSettings(method)) {
                double evidence = convertEvidence((String)ms.elementAt(i));
                //if (verbose) System.out.println("Adding evidence: "
		//			+evidence+" to "+pfName);
                functions.addEvidence(pfName, evidence);
                functions.propagateEvidenceDownward(pfName, evidence);
            }
        }
        synchronizeLikelihoods();
        functions.aPrioriEvidence();
    }
    
    public Hashtable switchOutProtein()
    {
        Hashtable pi = (Hashtable)(functions.removeProbabilityInstance());
        functions.zeroProbabilities();
        return pi;
    }
    
    public void clearDAGProbabilities()
    {
        functions.zeroProbabilities();	
    }
    
    public double[] pullOutLeafLikelihoods()
    {
        Vector leaves = getAllLeaves();
        double[] probs = new double[leaves.size()];
        for(int i = 0; i < leaves.size(); i++) {
            probs[i] = functions
		.getLikelihood(((Integer)leaves.elementAt(i)).intValue());
        }
        return probs;
    }
    
    /* sets the likelihood for leaves with evidence */
    public void synchronizeLikelihoods()
    {
        Vector leaves = getAllLeaves();
        double[] probs = new double[leaves.size()];
        double prior = getLeafSubsetPrior();
        double ltemp = 0.0;
        double totall = 0.0;
        //double remainder = 0.0;
        for(int i = 0; i < leaves.size(); i++) {
            ltemp = functions
		.getLikelihood(((Integer)leaves.elementAt(i)).intValue());
	    if(ltemp > 0) 
		totall = (1.0 - (1.0 - totall)*(1.0 - ltemp));
        }
        //remainder = getNumLeaves()*prior - totall;
        //if(remainder > 0) {
	//   remainder = remainder/(double)leaves.size();
	
	prior = (1 - totall)*prior;
	for(int i = 0; i < leaves.size(); i++) {
	    ltemp = functions
		.getLikelihood(((Integer)leaves.elementAt(i)).intValue());
	    probs[i] = ltemp;
	    probs[i] = 1.0 - ((1.0 - probs[i]) * (1.0 - prior));
            functions.setLikelihood(((Integer)leaves.elementAt(i)).intValue(),
				    probs[i]);
        }
    }
    
    public void insertLeafLikelihoods(double[] ll)
    {
        Vector leaves = getAllLeaves();
        for(int i = 0; i < ll.length-1; i++) {
            functions.setLikelihood(((Integer)leaves.elementAt(i)).intValue(),
				    ll[i]);
        }
    }
    
    private double convertEvidence(String m)
    {
        if(m.equals("IEA")) return 0.2;
        else if(m.equals("IMP")) return 0.8;
        else if(m.equals("IGI")) return 0.8;
        else if(m.equals("IPI")) return 0.8;
        else if(m.equals("ISS")) return 0.4;
        else if(m.equals("IDA")) return 0.9;
        else if(m.equals("IEP")) return 0.4;
        else if(m.equals("IGI")) return 0.4;
        else if(m.equals("TAS")) return 0.9;
        else if(m.equals("NAS")) return 0.3;
        else if(m.equals("RCA")) return 0.4;
        else if(m.equals("ND")) return 0.3;
        else if(m.equals("IC")) return 0.4;
        else if(m.equals("P")) return 0.2; // no idea what p is
        else if(m.equals("E")) return 0.6; // no idea what e is (evidence)
        else if(m.equals("NR")) return 0.3; // no idea what nr is (new result)
        else if(m.equals("GOR")) return 0.9; // from reading papers, my entry
        else 
            if (verbose) System.out.println("Methods not recognized: "+m);
        return 0;
    }
    
    public double getPosteriorProbability(int fn)
    {
        return functions.getPosterior(fn);		
    }
    
    public void initializeTransitionMatrix(PFunTransMatrix pftm, 
					   double scale) 
    {
        pfx = pftm;
        Vector leaves = functions.getAllLeaves();
        int[] dist = new int[leaves.size() + 1];
        Vector rowNames = new Vector();
        for(int i = 0; i < leaves.size(); i++) {
            Integer nodei = (Integer)leaves.elementAt(i);
            for(int j = 0; j < leaves.size(); j++) {
                if(i == j) dist[i] = 0;
                else dist[j] = functions.getTreeDistance(nodei.intValue(), 
			       ((Integer)leaves.elementAt(j)).intValue());
            }
            pfx.initializeToDistanceSimpleRate(i, dist, scale);
            rowNames.add((Integer)leaves.elementAt(i));
        }
        pfx.setNames(rowNames);
    }

    public Vector getLeafNames() 
    {
        Vector leaves = functions.getAllLeaves();
        Vector rowNames = new Vector();
        for(int i = 0; i < leaves.size(); i++) {
            rowNames.add((Integer)leaves.elementAt(i));
        }
	return rowNames;
    }
    
    public int getNumLeaves()
    {
        return functions.getNumLeaves();	
    }
    
    ////////////////////////////////////////////////////////////
    // Accessing DAG information for inference/learning       //
    ////////////////////////////////////////////////////////////
    
    // This is an approximation to two leaf node subsets
    // It checks to see if evidence associated with the
    // leaf and each (for three, pair of) other leaves
    // have evidence associated with them. If so, the
    // evidence is combined; otherwise it is added to
    // the appropriate leaf (pair) probability.
    
    // This function maps the integer onto the vector
    // of leaves that it has stored.
    public double getLeafSubsetProbability(int leaf)
    {
        Vector leaves = getAllLeaves();
        if(leaf >=0 && leaf < leaves.size()) {
            Integer leafID = (Integer)leaves.elementAt(leaf);
            return functions.getLeafSubsetProbability(leafID, leaves);
        }
        if (verbose) 
	    System.out.println("In PFunGODAG: entered wrong id for leaf: "
			       + leaf);
        return 0;
    }
    
    // Gets the prior probability of any leaf being present
    // in the function vector; this should be the same for
    // all leaf functions.
    public double getLeafSubsetPrior()
    {
        double lprior = 0.0;
        int[] leafv = functions.getLeafPolynomialVector(getNumLeaves());
        for(int i = 0; i < leafv.length; i++) {
            lprior += ((double)leafv[i]/(double)Math.pow(rValue, i+1));	
        }
        return lprior;	
    }
    
    public double getSingleLeafPrior()
    {
        return 1/rValue;
    }
    
    ////////////////////////////////////////////////////////////
    // Local class definition for an entry into the GO DAG    //
    ////////////////////////////////////////////////////////////
    
    private class Entry extends Object {
        public int goNum;
        public String goName;
        public int minLevel;
        
        public Entry(int num, String name, int level)
        {
            goNum =	num;
            while(name.indexOf('\\') >= 0) {
                name = name.substring(0, name.indexOf('\\'))
		    +name.substring(name.indexOf('\\')+1, 
				    name.length());
            }
            goName = name;
            minLevel = level;
        }
        
        public void setLevel(int level)
        {
            if(level < minLevel) minLevel = level;	
        }
        
        public String getName()
        {
            return goName;
        }
        
        public int getNumber()
        {
            return goNum;
        }
        
        public int getLevel()
        {
            return minLevel;
        }
    }
    
    /**
     * Method getAllLeaves.
     * @return Vector
     */
    public Vector getAllLeaves() {
        if (leafList == null) leafList = functions.getAllLeaves();
        return leafList;
    }
    
    /**
     * Method propagateLeavesUpwards.
     */
    /*
    private void propagateLeavesUpwards()
    {
        Enumeration pnames = functions.keys();
        while(pnames.hasMoreElements()) {
            Integer name = (Integer)pnames.nextElement();
            functions.propagateLeavesUpwards(name);
        }
	}*/
    
    /** Should we ever want to use an already existing Settings object,
     * we can direct our pointer to that.
     * @param settings Settings object to set to.
     * @see PFun.main()
     */
    public void setSettingsObject(Settings settings) {
        this.settings = settings;
        initSettings();
    }


    /** Once we get a new settings object, initialize any variables that we care about.
     * For example, I know PFunLibrary cares about verbose, family name, etc.
     */
    private void initSettings() {
        this.verbose = ((Boolean)this.settings.getSetting("verbose"))
	               .booleanValue();
    }
    
    
}
