//  filefinder.java
//  filefinder
//  for the Sifter project
//
//  Created by Steven Chan < www.stevenchan.us > on 10/16/05.
//
// Usage:
//  Filefinder finder = new Filefinder(".");
//  String[] default_values = {"molecular.ont", "molecular.ontology"};  // not case-sensitive
//  finder.seek("Molecular function ontology file",
//              "Downloadable from http://geneontology.org", 
//              "user-molecular.ont",
//              default_values);

package util;

import java.util.*;
import java.io.*;

public class Filefinder {

    private HashMap directoryMap;
    
    /**
    * Constructor. Prepares internal variables, such as the list of files in the specified directory.
     * @param startDirectory The directory to search for files.
     **/
    public Filefinder(String startDirectory) {
        this.directoryMap = this.hashDirectory(startDirectory);
        //this.startDirectory = startDirectory;
        // System.out.println( this.directoryMap.toString() ); // TODO - REMOVE LATER
    }
    /**
     * Constructor. If no parameters, assume that the starting directory is the current working directory.
     **/
    public Filefinder() {
        this(".");
    }

    /**
     * Test code. This is NOT necessary since Filefinder's purpose is not to be run.
     * If executed with "java Filefinder", however, it will run a few tests.
     * Feel free to delete this method if it causes problems.
     **/
    public static void main (String args[]) {
        // MAIN-RUNNING CODE
        //   This code is not necessary since it's not the main program
        //   being run, but useful for running tests.
	
        Filefinder f = new Filefinder("/Users/steven/sifter");
        try {
            String[] defaults = {"mol-function.ont"};
            System.out.println(f.seek("molecular function ontology file", "Downloadable from ONTOLOGY", null, defaults));
        } catch (FileNotFoundException e) {
            System.out.println("File not found: " + e.getMessage());
        }
        try {
            String[] defaults = {"mol-function.ont"};
            System.out.println(f.seek("molecular function ontology file", "Downloadable from ONTOLOGY", "nonexistent-file", defaults));
        } catch (FileNotFoundException e) {
            System.out.println("File not found: " + e.getMessage());
        }
        try {
            String[] defaults = {"function.ontology"};
            System.out.println(f.seek("molecular function ontology file", "Downloadable from ONTOLOGY", null, defaults));
        } catch (FileNotFoundException e) {
            System.out.println("File not found: " + e.getMessage());
        }
        
    }

    
    public String seek(String description,
                       String help,
                       String uservalue,
                       String[] default_filenames) throws FileNotFoundException {
        String res;
                           
        // If the user specifies a filename, then do something with it.
        if (uservalue != null) {
            if ((new File(uservalue).exists( ))) {
                return uservalue;
            } else {
                // Throw a new exception saying that the following files could work.
                try {
                    res = fileIn(default_filenames, this.directoryMap);
                } catch (FileNotFoundException e) {
                    // If the default files don't work, then give them the help message.
                    throw new FileNotFoundException (uservalue +
                                                     " does not exist. " + help);
                }
                throw new FileNotFoundException (uservalue +
                                                 " does not exist. This file may work, however: " +
                                                 res + ".");

            }
        } else {
            // If the user does not specify anything, then try to find files from the default filenames array.
            try {
                res = fileIn(default_filenames, this.directoryMap);
            } catch (FileNotFoundException e) {
                // If the default files don't work, then give them the help message.
                throw new FileNotFoundException (description + " does not exist. " + help);
            }
            return res;
        }
    }
    
    /**
     * Creates a HashMap representing all the files in the specified directory.
     * Searches recursively through the directory and adds key "filename", actual file object, into
     * the resulting HashMap.
     * I'm using File objects b/c that's the most effective and direct way to determine
     * whether a given File object represents an actual file or a directory,
     * since we're recursing.
     * 
     * @param startDirectory The directory to search for files.
     * @return HashMap object with String keys and File objects for values
     **/    
    private HashMap hashDirectory(String startDirectory) {
        HashMap res = new HashMap( );
        
        File[] list = new File(startDirectory).listFiles( );
        for (int i=0; i<list.length; i++) {
            File item = list[i];
            // System.out.println(item);    // Print the item.
          
            // If it's a directory,
            //    recursively call hashDirectory() on it. Save it in hash.
            //    Add all those hashes to res.
            if (item.isDirectory()) {
                HashMap recursedMap = hashDirectory(item.getPath());
                res.putAll(recursedMap);
            } else {
                // Add the lower-cased filename to hash, plus the file object.
                String lowerCaseName = item.getName().toLowerCase();
                res.put(lowerCaseName, item);
            }
        }
        return res;
    }
    
    /**
     * Given an array of possible filenames (not pathnames!) and a List of all the files,
     * tries to see if there are matches in the List.
     * (Does not use case sensitivity.)
     * 
     * @param possibilities
     * @param fileMap
     * @return String containing a perfectly valid filename
     * @throws FileNotFoundException
     **/
    private String fileIn(String[] possibilities, HashMap fileMap) throws FileNotFoundException {
        for (int i=0; i<possibilities.length; i++) { // i represents index for possibilities.
            String possibility = possibilities[i];
            if (fileMap.containsKey(possibility)) {  // If a possible filename does exist,
                return ((File)fileMap.get(possibility)).getPath();  // just return it!
            }
        }
        // At this point, we've exhausted our list of possibilities,
        // so complain by throwing a filenotfoundexception.
        // TODO: pass possibilities array to the new Exception. Can we do this in Java?
        throw new FileNotFoundException();
    }

}
