/**
 * SIFTER project. This handles the command-line interface *and nothing else*
 * (or, it shouldn't, at least in future revisions!).
 * $Id: PFun.java,v 1.2 2005/05/27 06:56:01 sprite Exp $
 * 
 * Copyright (c) 2010, Barbara Engelhardt (bee@compbio.berkeley.edu)
 *
 * @author Barbara Engelhardt (primary investigator and author)
 * @author Steven R. Chan (later code hacking, documentation, GUI)
 * @version 1.0
 */

import java.io.*;
import org.apache.commons.cli.*; // for command-line parsing
import stat.PFunTransMatrix;
import util.PFunGODAG;
import util.Settings; // to preserve options.
import util.Filefinder; // to figure out where the data files are

// Insert constants here?


public class PFun {
    /**
     * Holds an Options object, which is used for the Apache commons CLI parser.
     */
    private static Options options;

    /**
     * Holds the Settings object (a place to store CLI user-defined parameters)
     */
    private static Settings settings;

    /**
     * Toggles verbose mode.
     */
    static boolean verbose;
    
    /**
     * Is a file's existence really required? (i.e. should we assert its existence)
     */
    private static boolean FILE_EXISTENCE_REQUIRED = true;
    private static boolean FILE_EXISTENCE_OPTIONAL = false;
    
    private static int ITERATIONS_DEFAULT = 4000;
    private static double STEPSIZE_DEFAULT = 0.01;
    private static double CUTOFF_DEFAULT = 0.000115;
    private static int FOLDS_DEFAULT = 0;
    private static int TRUNCATION_DEFAULT = 4;
    

    /**
     * Sole entry point to the command-line application. Tons and tons of
     * command-line parsing, error-checking on parameters, and string-pulling.
     * 
     * @author Steven R. Chan
     * 
     * @param args array of string arguments
     * @return No return value
     */
    public static void main(String[] args) throws ParseException, IllegalArgumentException, FileNotFoundException {
        verbose = false;
        String family;
        CommandLineParser parser = new PosixParser();
        options = buildOptions();

        try {
            // This block of code parses command-line arguments and stores them
            // in settings.
            CommandLine line = parser.parse(options, args);
            args = line.getArgs();
            settings = new Settings();

            // Does the user want HELP?
            if (line.hasOption("help")) {
                printHelp();
                System.exit(0);
            }

            // Set verbose options.
            if (line.hasOption("v")) {
                settings.setSetting("verbose", new Boolean(true));
                verbose = true;
                System.out.println("**************************************************");
                System.out.println("JOB SUMMARY");
            } else {
                settings.setSetting("verbose", new Boolean(false));
                verbose = false;
            }

            
            // Set include iea option.
            if (line.hasOption("iea")) {
                settings.setSetting("iea", new Boolean(true));
            } else {
                settings.setSetting("iea", new Boolean(false));
            }
            if (line.hasOption("bg")) {
                settings.setSetting("bg", new Boolean(true));
            } else {
                settings.setSetting("bg", new Boolean(false));
            }
            if (line.hasOption("tas")) {
                settings.setSetting("tas", new Boolean(true));
            } else {
                settings.setSetting("tas", new Boolean(false));
            }
            if (line.hasOption("nas")) {
                settings.setSetting("nas", new Boolean(true));
            } else {
                settings.setSetting("nas", new Boolean(false));
            }
            if (line.hasOption("ipi")) {
                settings.setSetting("ipi", new Boolean(true));
            } else {
                settings.setSetting("ipi", new Boolean(false));
            }
            if (line.hasOption("iss")) {
                settings.setSetting("iss", new Boolean(true));
            } else {
                settings.setSetting("iss", new Boolean(false));
            }
            if (line.hasOption("rca")) {
                settings.setSetting("rca", new Boolean(true));
            } else {
                settings.setSetting("rca", new Boolean(false));
            }
            if (line.hasOption("iep")) {
                settings.setSetting("iep", new Boolean(true));
            } else {
                settings.setSetting("iep", new Boolean(false));
            }
            if (line.hasOption("igc")) {
                settings.setSetting("igc", new Boolean(true));
            } else {
                settings.setSetting("igc", new Boolean(false));
            }
            if (line.hasOption("ic")) {
                settings.setSetting("ic", new Boolean(true));
            } else {
                settings.setSetting("ic", new Boolean(false));
            }
            if (line.hasOption("igi")) {
                settings.setSetting("igi", new Boolean(true));
            } else {
                settings.setSetting("igi", new Boolean(false));
            }
	    settings.setSetting("imp", new Boolean(true));
	    settings.setSetting("ida", new Boolean(true));

            // Make sure we have exactly one argument for FAMILY setting.
            if (args.length != 1) {
                throw new IllegalArgumentException();
            } else {
                family = args[0].toLowerCase();
                settings.setSetting("family", family);
                if (verbose)
                    System.out.println("Family set to " + family);
                // TODO: Write clearer Verbose handling; 
		// substitute for all system.out.println.
            }

            /**
             * FILENAME INFORMATION
             * [option, setting, description, help]
             */
            
            // Initialize possible string names.
            String[] FILE_FX_DEFAULTS = {"infer-" + family + ".fx"};
            String[] FILE_FX = {"fx",
                "familyFilename",
                "Parameter file (.fx)",
                "To obtain a .fx file, run 'sifter --generate'. See README.txt."};
            String[] FILE_SCALE_DEFAULTS = {"scale-" + family + ".fx"};
            String[] FILE_SCALE = {"sfx",
                "scaleParamsFilename",
                "Scale parameter file (.fx)",
                "To obtain a .fx scale file, run with --generate. See README.txt."};
            String[] FILE_ALPHA_DEFAULTS = {"alpha-" + family + ".fx"};
            String[] FILE_ALPHA = {"afx",
                "alphaParamsFilename",
                "Alpha parameter file (.fx)",
                "To obtain a .fx scale file, run with --generate. See README.txt."};
            String[] FILE_PLI_DEFAULTS = {"proteinfamily_" + family + ".pli",
					  "proteinfamily_" 
					  + family.toUpperCase() + ".pli"};
            String[] FILE_PLI = {"pli",
                "proteinFilename",
                "Protein family file (.pli)",
                "To obtain a .pli file, use 'pfam2pli' script. See README.txt."};
            String[] FILE_NEX_DEFAULTS = {"reconciled_" + family +".nex", 
					  "reconciled_" + family +".nhx",
					  "reconciled_" + family.toUpperCase() 
					  +".nex", 
					  "reconciled_" + family.toUpperCase() 
					  +".nhx"};
            String[] FILE_NEX = {"nex",
                "reconciledFilename (.nex/.nhx)",
                "Reconciled tree in New Hampshire Extended format",
                "To obtain a reconciled tree file, use 'pli2tree' script. See README.txt."};
            String[] FILE_ONTOLOGY_DEFAULTS = {"function.ontology", "function.ont"};
            String[] FILE_ONTOLOGY = {"ontology",
                "ontology",
                "Functions gene ontology database",
                "Get function.ontology from www.geneontology.org."};
            String[] FILE_OUTPUT_DEFAULTS = {"output/default.rdata"};
            String[] FILE_OUTPUT = {"output",
                "output",
                "Resulting output file",
                "File in which Sifter outputs its final results."};

            prepareFilename(FILE_ONTOLOGY, FILE_ONTOLOGY_DEFAULTS, 
			    family, line, FILE_EXISTENCE_REQUIRED);
            prepareFilename(FILE_OUTPUT, FILE_OUTPUT_DEFAULTS, 
			    family, line);

            ////// PROGRAM ENTRY: The action starts here.
            if (line.hasOption("g")) { // User wants to generate
                                              // parameters.
                String fx = prepareFilename(FILE_FX, FILE_FX_DEFAULTS, 
					    family, line);
                String scale = prepareFilename(FILE_SCALE, FILE_SCALE_DEFAULTS, 
					    family, line);
                String alpha = prepareFilename(FILE_ALPHA, FILE_ALPHA_DEFAULTS, 
					    family, line);
                String pli = prepareFilename(FILE_PLI, FILE_PLI_DEFAULTS, 
					     family, line, 
					     FILE_EXISTENCE_REQUIRED);
                String nex = prepareFilename(FILE_NEX, FILE_NEX_DEFAULTS, 
					     family, line, 
					     FILE_EXISTENCE_REQUIRED);
		if (line.hasOption("folds")) { // Set iteration options
		    settings.setSetting("folds", 
					Integer.valueOf(line
						       .getOptionValue("folds")));
		} else {
		    settings.setSetting("folds", 
					new Integer(FOLDS_DEFAULT));
		}
		if (line.hasOption("truncation")) { // Set iteration options
		    settings.setSetting("truncation", 
					Integer.valueOf(line
						       .getOptionValue("truncation")));
		} else {
		    settings.setSetting("truncation", 
					new Integer(TRUNCATION_DEFAULT));
		}
                initGenerate(fx, scale, alpha, pli);
	    }   else if (line.hasOption("xval")) {
                String fx = prepareFilename(FILE_FX, FILE_FX_DEFAULTS, 
					    family, line);
                String scale = prepareFilename(FILE_SCALE, 
					       FILE_SCALE_DEFAULTS, 
					       family, line);
                String alpha = prepareFilename(FILE_ALPHA, 
					       FILE_ALPHA_DEFAULTS, 
					       family, line);
                String pli = prepareFilename(FILE_PLI, FILE_PLI_DEFAULTS,  
					     family, line, 
					     FILE_EXISTENCE_REQUIRED);
                String nex = prepareFilename(FILE_NEX, FILE_NEX_DEFAULTS, 
					     family, line, 
					     FILE_EXISTENCE_REQUIRED); 
		if (line.hasOption("iter")) { // Set iteration options
		    settings.setSetting("iterations", 
					Integer.getInteger(line
						     .getOptionValue("iter")));
		} else {
		    settings.setSetting("iterations", 
					new Integer(ITERATIONS_DEFAULT));
		}
                
		if (line.hasOption("step")) { // Set iteration options
		    settings.setSetting("stepsize", 
					Double.valueOf(line
						     .getOptionValue("step")));
		} else {
		    settings.setSetting("stepsize", 
					new Double(STEPSIZE_DEFAULT));
		}
		if (line.hasOption("cutoff")) { // Set iteration options
		    settings.setSetting("cutoff", 
					Double.valueOf(line
						       .getOptionValue("cutoff")));
		} else {
		    settings.setSetting("cutoff", 
					new Double(CUTOFF_DEFAULT));
		}
		if (line.hasOption("folds")) { // Set iteration options
		    settings.setSetting("folds", 
					Integer.valueOf(line
						       .getOptionValue("folds")));
		} else {
		    settings.setSetting("folds", 
					new Integer(FOLDS_DEFAULT));
		}
		if (line.hasOption("truncation")) { // Set iteration options
		    settings.setSetting("truncation", 
					Integer.valueOf(line
						       .getOptionValue("truncation")));
		} else {
		    settings.setSetting("truncation", 
					new Integer(TRUNCATION_DEFAULT));
		}
		if (line.hasOption("em")) { // Set em options
		    settings.setSetting("em", new Boolean(true));
		}
		initXValidation(fx, scale, alpha, pli, nex);
            } else if (line.hasOption("em")) {
                String fx = prepareFilename(FILE_FX, FILE_FX_DEFAULTS, 
					    family, line);
                String scale = prepareFilename(FILE_SCALE, 
					       FILE_SCALE_DEFAULTS, 
					       family, line);
                String alpha = prepareFilename(FILE_ALPHA, 
					       FILE_ALPHA_DEFAULTS, 
					       family, line);
                String pli = prepareFilename(FILE_PLI, FILE_PLI_DEFAULTS,  
					     family, line, 
					     FILE_EXISTENCE_REQUIRED);
                String nex = prepareFilename(FILE_NEX, FILE_NEX_DEFAULTS, 
					     family, line, 
					     FILE_EXISTENCE_REQUIRED); 
		
		if (line.hasOption("iter")) { // Set iteration options
		    settings.setSetting("iterations", 
					Integer.valueOf(line
						     .getOptionValue("iter")));
		} else {
		    settings.setSetting("iterations", 
					new Integer(ITERATIONS_DEFAULT));
		}
		if (line.hasOption("step")) { // Set iteration options
		    settings.setSetting("stepsize", 
					Double.valueOf(line
						     .getOptionValue("step")));
		} else {
		    settings.setSetting("stepsize", 
					new Double(STEPSIZE_DEFAULT));
		}
		if (line.hasOption("folds")) { // Set iteration options
		    settings.setSetting("folds", 
					Integer.valueOf(line
						       .getOptionValue("folds")));
		} else {
		    settings.setSetting("folds", 
					new Integer(FOLDS_DEFAULT));
		}
		if (line.hasOption("truncation")) { // Set iteration options
		    settings.setSetting("truncation", 
					Integer.valueOf(line
						       .getOptionValue("truncation")));
		} else {
		    settings.setSetting("truncation", 
					new Integer(TRUNCATION_DEFAULT));
		}
		if (line.hasOption("cutoff")) { // Set iteration options
		    settings.setSetting("cutoff", 
					Double.valueOf(line
						       .getOptionValue("cutoff")));
		} else {
		    settings.setSetting("cutoff", 
					new Double(CUTOFF_DEFAULT));
		}
		initEM(fx, scale, alpha, pli, nex);
	    } else { // We want exact parameters.
                if (verbose)
                    System.out.println("Mode: Use Exact Network Parameters.");
                if (verbose)
                    System.out
	        .println("**************************************************");
                String fx = prepareFilename(FILE_FX, FILE_FX_DEFAULTS, 
					    family, line);
                String scale = prepareFilename(FILE_SCALE, FILE_SCALE_DEFAULTS, 
					    family, line);
                String alpha = prepareFilename(FILE_ALPHA, FILE_ALPHA_DEFAULTS, 
					    family, line);
                String pli = prepareFilename(FILE_PLI, FILE_PLI_DEFAULTS,  
					     family, line, 
					     FILE_EXISTENCE_REQUIRED);
                String nex = prepareFilename(FILE_NEX, FILE_NEX_DEFAULTS, 
					     family, 
					     line, FILE_EXISTENCE_REQUIRED); 
		
		settings.setSetting("reconciledFilename", nex);
		if (line.hasOption("truncation")) { // Set iteration options
		    settings.setSetting("truncation", 
					Integer.valueOf(line
						       .getOptionValue("truncation")));
		} else {
		    settings.setSetting("truncation", 
					new Integer(TRUNCATION_DEFAULT));
		}
		if (line.hasOption("folds")) { // Set iteration options
		    settings.setSetting("folds", 
					Integer.valueOf(line
						       .getOptionValue("folds")));
		} else {
		    settings.setSetting("folds", 
					new Integer(FOLDS_DEFAULT));
		}

                performExact();
            }

        // Error catching.
        } catch (IllegalArgumentException e) {
            System.err
                    .println("ERROR: Please specify one protein family file to process: ");
	    for(int i = 0; i < args.length; i++) System.out.println(args[i]);
            printHelp();
        } catch (UnrecognizedOptionException e) {
            System.err.println("ERROR: Unrecognized options. ("
                    + e.getMessage() + ")");
            printHelp();
        } catch (MissingArgumentException e) {
            System.err.println("ERROR: Incomplete arguments. Some arguments require values.");
            printHelp();
        } catch (FileNotFoundException e) {        // TODO: Clean this up.
            System.err.println("ERROR: File not found.\n   " 
			       + e.getMessage() + ".\n" 
			       + "   To specify files, please use the options "
			       + "listed under Sifter's help by entering:\n" 
			       + "    java -jar sifter.jar --help");
        }
    }

    
    private static String prepareFilename(String[] fileinfo, 
					  String[] defaultFilenames, 
					  String family, CommandLine line)
	throws FileNotFoundException {
	return prepareFilename(fileinfo, defaultFilenames, 
			       family, line, FILE_EXISTENCE_OPTIONAL);
    }

    /** Prepares filenames for use by the rest of the program 
     * (like initGenerate).
     * Checks for its existence as well.
     * 
     * @param fileinfo Fileinfo arrays, such as FILE_FX coded in main().
     * @param defaultFilenames Default file name arrays, such as that coded in main().
     * @param family A protein family name, such as 'pf20005'
     * @param line Command-line options
     * @param existence_required Does the file need to exist?
     * @return String final filename to use
     * @throws FileNotFoundException If user specifies an invalid file, 
     *         or the default file does not exist, throw an exception.
     */
    private static String prepareFilename(String[] fileinfo, 
					  String[] defaultFilenames, 
					  String family, CommandLine line, 
					  boolean existence_required) 
	throws FileNotFoundException {
	
        String optionName = fileinfo[0];
        String settingName = fileinfo[1];
        String desc = fileinfo[2];
        String help = fileinfo[3];    
        String userFilename = null;
        String finalFilename = null;
        
        if (line.hasOption(optionName)) { 
	    userFilename = line.getOptionValue(optionName); 
	}
    
        Filefinder f = new Filefinder();
        try {
            finalFilename = f.seek(desc, help, userFilename, defaultFilenames);
        } catch (FileNotFoundException e) {
            if (existence_required) {
                throw e;
            } else { // What if the user specifies an invalid file
		//and we don't need it to exist?
                if (userFilename != null) {
                    finalFilename = userFilename;
                } else {
                    finalFilename = defaultFilenames[0];
                }
            }
        }
        
        settings.setSetting(settingName, finalFilename);
        if (verbose) {
            System.out.println(desc + " ("+settingName+"): " + finalFilename);
        }
        return finalFilename;
    }

    /** Prepare parameters for "Generate" mode, 
     *  passing things off to performGenerate().
     * @see performGenerate()
     * @param familyFilename
     * @param scaleParamsFilename
     * @param alphaParamsFilename
     * @param proteinFilename
     */
    private static void initGenerate(String familyFilename, 
				     String scaleParamsFilename, 
				     String alphaParamsFilename, 
				     String proteinFilename) {
        if (verbose)
            System.out.println("Mode: Generating Network Parameters.");
        if (verbose)
            System.out
                    .println("**********************************************");
        performGenerate(familyFilename, scaleParamsFilename, 
			alphaParamsFilename, 20.0, proteinFilename); //
        // speciation: .fx files are matrices that are at least of length 2. 
        // Show the chance that it would change functions.
    }

    /** Prepare parameters for "EM" mode, 
     *  passing things off to performEM().
     * @see performEM()
     * @param familyFilename
     * @param scaleParamsFilename
     * @param alphaParamsFilename
     * @param proteinFilename
     * @param nexFilename
     */
    private static void initEM(String familyFilename, 
			       String scaleParamsFilename, 
			       String alphaParamsFilename, 
			       String proteinFilename,
			       String nexFilename) {
        if (verbose)
            System.out.println("Mode: Estimating Network Parameters.");
        if (verbose)
            System.out
		.println("**********************************************");
        performEM(familyFilename, proteinFilename, 
		  scaleParamsFilename, alphaParamsFilename, 
		  nexFilename); //
        // speciation: .fx files are matrices that are at least of length 2. 
        // Show the chance that it would change functions.
    }

    /** Prepare parameters for cross-validation mode, 
     *  passing things off to performXValidation().
     * @see performXValidation()
     * @param familyFilename
     * @param scaleParamsFilename
     * @param alphaParamsFilename
     * @param proteinFilename
     * @param nexFilename
     */
    private static void initXValidation(String familyFilename, 
					String scaleParamsFilename, 
					String alphaParamsFilename, 
					String proteinFilename,
					String nexFilename) {
        if (verbose)
            System.out.println("Mode: Cross Validation, Estimating Network Parameters.");
        if (verbose)
            System.out
		.println("**********************************************");
        performXValidation(familyFilename, proteinFilename, 
			   scaleParamsFilename, alphaParamsFilename,
			   nexFilename); //
        // speciation: .fx files are matrices that are at least of length 2. 
        // Show the chance that it would change functions.
    }


    /**
     * Prints command-line help.
     * @see main
     */
    private static void printHelp() {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp("java -jar sifter.jar [OPTIONS] FAMILYNAME",
                        options);
    }

    /**
     * Primary function that pulls the strings. :) Performs gene ontology
     * DAG/graph calculations, reads in reconciled tree, and does inference
     * using the PFunLibrary class.
     * 
     * Depends on PFunGODAG, and PFunLibrary.
     * Used to be called parseTestNetworkInputExact().
     * 
     * @author Barbara Engelhardt
     */
    private static void performExact() {
        // TODO: parseGenerateAndTestNetworkInput will call this many, many
        // times, which will clobber any output files generated... could we
        // streamline this?
        boolean noIEA = !((Boolean)settings.getSetting("iea")).booleanValue();
        PFunLibrary pfl = new PFunLibrary();
        pfl.setSettingsObject(settings);

	String filename = ((String) settings.getSetting("proteinFilename"));
	String scaleFilename = ((String) settings.getSetting("scaleParamsFilename"));
	String alphaFilename = ((String) settings.getSetting("alphaParamsFilename"));
	String ontology = ((String) settings.getSetting("ontology"));
	String nexfile = ((String) settings.getSetting("reconciledFilename"));
        // GO DAG probabilities stuff
	if(verbose)
	    System.out.println("Running: "+filename+", with ontology "+ontology);
        PFunGODAG pfgodag = pfl.buildDatasetGODAG(filename,
						  ontology, noIEA);
	// Read in reconciled tree.  [maybe parameters file, .fx files]
        pfl.buildGraphicalModel(pfgodag, nexfile, noIEA); 
        pfl.inferWithTransitionMatrixExact(pfgodag, 
					   pfl.getFamily()); 
    }

    public static double performEM(String fileName, String proteinFilename, 
				   String scaleFilename, String alphaFilename, 
				   String nexFilename) {

	boolean noIEA = !((Boolean)settings.getSetting("iea")).booleanValue();
	PFunLibrary pfl = new PFunLibrary();
	pfl.setSettingsObject(settings);
	String ontology = ((String) settings.getSetting("ontology"));
	PFunGODAG pfgodag = pfl.buildDatasetGODAG(proteinFilename, 
						  ontology, 
						  noIEA);
	pfl.buildGraphicalModel(pfgodag, nexFilename, noIEA);
	double correct = pfl.estimateParameters(pfgodag, pfl.getFamily());
	return correct;
    }

    public static double performXValidation(String fileName, 
					    String proteinFilename, 
					    String scaleFilename,
					    String alphaFilename,
					    String nexFilename) 
    {
	boolean noIEA = !((Boolean)settings.getSetting("iea")).booleanValue();
	PFunLibrary pfl = new PFunLibrary();
	pfl.setSettingsObject(settings);
	String ontology = ((String) settings.getSetting("ontology"));
	PFunGODAG pfgodag = pfl.buildDatasetGODAG(proteinFilename, 
						  ontology, 
						  noIEA);
	pfl.buildGraphicalModel(pfgodag, nexFilename, noIEA);
	double correct = pfl.crossValidation(pfgodag, pfl.getFamily());
	return correct;
    }

    /**
     * Generates a set of input parameters for the inference problem. All
     * parameters must be >= 0. The lower the theta_m,n parameters, the
     * more likely function m will mutate to function n. The s parameter
     * is a rate of mutation.
     * 
     * Formerly generateNetworkParameters()
     * 
     * @param fileName
     * @param scaleFilename
     * @param alphaFilename
     * @param proteinFilename
     * @author Barbara Engelhardt
     */
    private static void performGenerate(String fileName, String scaleFilename,
					String alphaFilename, double scale, 
					String proteinFilename) {
        PFunLibrary pfl = new PFunLibrary();
        pfl.setSettingsObject(settings);

	// true means, don't include iea data
        boolean noIEA = !((Boolean)settings.getSetting("iea")).booleanValue();
        String ontology = ((String)settings.getSetting("ontology"));
        PFunGODAG pfgodag = pfl.buildDatasetGODAG(proteinFilename,
						  ontology, noIEA); 

        int numLeaves = pfgodag.getNumLeaves();
        System.out.println("There are " + numLeaves + " candidate functions.");
        // create a transition matrix
        PFunTransMatrix pfx = new PFunTransMatrix(numLeaves, numLeaves,
                true);
        pfx.setScale("species", 0.03);
        pfx.setScale("duplication", 0.05);
	for(int i = 0; i < numLeaves; i++) pfx.setAlpha(i, 1.0);
        pfgodag.initializeTransitionMatrix(pfx, scale);
        pfx.printOutMatrix(fileName); // TODO: Distinguish between
                                            // filename and fileName.
        pfx.printOutScale(scaleFilename);
        pfx.printOutAlpha(alphaFilename);
        System.out.println("Output written to "+fileName
			   +", "+scaleFilename);
    }

    
    /** Builds the set of possible options for this program.
     * @return An Options file that can be used by the CommandLine parser.
     */
    private static Options buildOptions () {
        Options res = new Options();
	
        res.addOption("g", "generate", false,
		      "Generates a set of input parameters for" 
		      + " the inference problem.");
        res.addOption("iea", "with-iea", false,
		"Use protein annotations inferred by electronic annotation.");
        res.addOption("tas", "with-tas", false,
		"Use protein annotations from traceable author statements.");
        res.addOption("nas", "with-nas", false,
		"Use protein annotations from non-traceable author statements.");
        res.addOption("igi", "with-igi", false,
		"Use protein annotations from those inferred from genetic interaction.");
        res.addOption("ipi", "with-ipi", false,
		"Use protein annotations from those inferred from physical interaction.");
        res.addOption("iss", "with-iss", false,
		"Use protein annotations from those inferred from sequence similarity.");
        res.addOption("ic", "with-ic", false,
		"Use protein annotations from those inferred from curator");
        res.addOption("rca", "with-rca", false,
		"Use protein annotations from those reconstructed from computational analyses.");
        res.addOption("iep", "with-iep", false,
		"Use protein annotations from those inferred from expression profiles.");
        res.addOption("igc", "with-igc", false,
		"Use protein annotations from those inferred from genomic context.");
        res.addOption(OptionBuilder
		      .withLongOpt("ida-background")
		      .withDescription("Use only experimental annotations (IDA, IMP, TAS, IGI, IPI) to generate candidate functions")
		      .create("bg"));
        res.addOption("em", "em", false,
		"Perform EM to estimate parameters");
        res.addOption("xval", "xvalidation", false,
		"Use cross-validation with EM.");
        res.addOption(OptionBuilder
		      .withLongOpt("output")
		      .withDescription("Set output file (default: output/default.rdata)")
		      .withArgName("filename")
		      .hasArg()
		      .create("output"));
        res.addOption(OptionBuilder
                .withLongOpt("protein")
                .withDescription("Set protein file (default: proteins/proteinfamily_<FAMILY>.pli)")
                .withArgName("filename")
                .hasArg()
                .create("pli"));
        res.addOption(OptionBuilder
                .withLongOpt("reconciled")
                .withDescription("Set reconciled .nex tree (default: reconciled/reconciled_<FAMILY>.nex)")
                .withArgName("filename")
                .hasArg()
                .create("nex"));
        res.addOption(OptionBuilder
                .withLongOpt("familyfile")
                .withDescription("Set family .fx parameter filename (default: data/infer-<FAMILY>.fx)")
                .withArgName("filename")
                .hasArg()
                .create("fx"));
        res.addOption(OptionBuilder
                .withLongOpt("scale")
                .withDescription("Set family .fx scale filename (default: data/scale-<FAMILY>.fx)")
                .withArgName("filename")
                .hasArg()
                .create("sfx"));
        res.addOption(OptionBuilder
                .withLongOpt("alpha")
                .withDescription("Set family .fx alpha filename (default: data/alpha-<FAMILY>.fx)")
                .withArgName("filename")
                .hasArg()
                .create("afx"));
        res.addOption(OptionBuilder
                .withLongOpt("ontology")
                .withDescription("Specify which ontology file you want (default: \"data/function.ontology\")")
                .withArgName("filename")
                .hasArg()
                .create("ontology"));
        res.addOption(OptionBuilder
                .withLongOpt("help")
                .withDescription(
                        "Show help for arguments. (More help is available via README.txt)")
                .create());
        res.addOption(OptionBuilder
                      .withLongOpt("iter")
                      .withDescription("Number of iterations. At the moment, this applies only to EM. (default: " + ITERATIONS_DEFAULT + ")")
                      .withArgName("number")
                      .hasArg()
                      .create("iter"));
        res.addOption(OptionBuilder
                      .withLongOpt("step")
                      .withDescription("Step size for gradient ascent in EM (M-step) (default: " + STEPSIZE_DEFAULT + ")")
                      .withArgName("number")
                      .hasArg()
                      .create("step"));
        res.addOption(OptionBuilder
                      .withLongOpt("cutoff")
                      .withDescription("Cutoff delta for gradient ascent in EM (M-step) (default: " + CUTOFF_DEFAULT + ")")
                      .withArgName("number")
                      .hasArg()
                      .create("cutoff"));
        res.addOption(OptionBuilder
                      .withLongOpt("folds")
                      .withDescription("Number of folds in cross validation, leave-one-out is 0 (default: " + FOLDS_DEFAULT + ")")
                      .withArgName("number")
                      .hasArg()
                      .create("folds"));
        res.addOption(OptionBuilder
                      .withLongOpt("truncation")
                      .withDescription("Number of functions to truncate to in approximation (default: " + TRUNCATION_DEFAULT + ")")
                      .withArgName("number")
                      .hasArg()
                      .create("truncation"));
        res.addOption("v", "verbose", false, "Verbose operation.");
        return res;
    }
}
