function [B] = medianFilter(A, block_size)

% Performs a median filter on gray image given by A.
% Works on a block of of size block_size x block_size (3x3, 5x5, 9x9).
% We assume block_size is odd.

[m,n] = size(A);

% We need to be able to apply median filter at the boundary pixels.
% Assume the image has Neumann boundary conditions.
% The easiest way to handle this is to set up a matrix A2 that has 
% the original image A in the center and copies the adjacent
% pixel value around the edges.

k = floor(block_size/2);  % How far we need to extend border.
A2 = zeros(m+2*k,n+2*k);
A2(k+1:m+k,k+1:n+k) = A;   % Original image in center.

% Extend border pixels.
% Handle the 4 corners first.
A2(1:k,1:k) = A(1,1);                  % Upper left corner.
A2(1:k,n+k+1:n+2*k) = A(1,n);          % Upper right corner.
A2(m+k+1:m+2*k,n+k+1:n+2*k) = A(m,n);  % Lower right corner.
A2(m+k+1:m+2*k,1:k) = A(m,1);          % Lower left corner.

% To handle the sides, we need to copy 1 row/column at a time.
for i = 1:k
  A2(i,k+1:n+k) = A(1,1:n);            % Upper.
  A2(k+1:m+k,n+k+i) = A(1:m,n);        % Right.
  A2(m+k+i,k+1:n+k) = A(m,1:n);        % Lower.
  A2(k+1:m+k,i) = A(1:m,1);            % Left.
end;

% Apply median filter to form matrix B.
B=zeros(m,n);
pixel_block = zeros(1,block_size^2);
for i = 1:m
  for j = 1:n
        % Determine pixel block.
        k = floor(block_size/2);
        pixel_block = reshape(A2(i:i+2*k,j:j+2*k),block_size^2,1);
        
        % Find median without using median function.
        sorted_block = sort(pixel_block);
        median_position = ceil(block_size^2 / 2); 
        block_median = sorted_block (median_position);
	
        B(i,j) = block_median;
  end;
end;
