%==========================================================================
%
%   analyseImages.m designed to perform an analysis of one cell data set
%
%   Copyright  2010, Steffen Remmele, Julia Ritzerfeld
%
%   This program is free software; you can redistribute it and/or modify it 
%   under the terms of the GNU General Public License as published by the 
%   Free Software Foundation; either version 3 of the License, 
%   or (at your option) any later version.
%
%   This program is distributed in the hope that it will be useful, but 
%   WITHOUT ANY WARRANTY; without even the implied warranty of 
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
%   General Public License for more details.
%
%   You should have received a copy of the GNU General Public License along 
%   with this program; if not, see <http://www.gnu.org/licenses/>.
%
%   CONTACT:
%   Steffen Remmele 
%   Experimental Radiation Oncology 
%   Mannheim Medical Center - University of Heidelberg 
%   Phone: 0621-383-6916
%   Mail: steffen.remmele@ziti.uni-heidelberg.de
%   HP: www.medical-physics.de
%
%   Dr. Julia Ritzerfeld
%   Biochemie-Zentrum der Universitt Heidelberg (BZH)
%   Phone: 06221-544180
%   Mail: julia.ritzerfeld@bzh.uni-heidelberg.de
%   HP: www.uni-heidelberg.de/zentral/bzh/
%
%==========================================================================
function Erg = analyseImages(imgGFP, imgCherry, imgHoechst, boolPause, boolCherryImage, boolCherrySegmentation)
%==========================================================================
%
% Analyse two different cell images with additional information in nucleus
% image and differentiate between the phenotypes membran, cytoplasma, golgi
% and golgi strong. In an early version undecided cells could be stored and 
% classified manually in an extra step (removed). Cells that do not match
% the quality requirements or the classification criteria will be marked as
% "invalid"
%
% This function is the kernel of an GUI supported workflow and handles one
% group of cell images belonging together (Hoechst, GFP and Cherry). First, 
% the cell nuclei are segmented using an adaptive threshold method on basis
% of the Hoechst image. Then the cell segmentation is done using the GFP 
% image (region growing) and the segmentation result is used for both GFP 
% and Cherry image classification. A decision function is used to classify
% the cells. This function can be adapted for other types of cell images on
% basis of the provided features (mean and standard deviation). It is
% possible to have an dataset without Cherry or to do a segmentation with
% an available Cherry image.
%
% INPUT:
%       imgGFP                  - GFP cell image
%       imgCherry               - Cherry cell image
%       imgHoechst              - Hoechst nucleus image
%       boolPause               - show details of analyses and stop to show each cell 
%       boolCherryImage         - shows whether there is a cherry image
%       boolCherrySegmentation  - uses Cherry image for segmentation
%
% OUTPUT:
%       Erg  - result of the analyses  
%           (Erg = zeros(1,14)) -directly assigned
%           14 integer values containing the result of the analysis 
%           first the results for the GFP analysis then Cherry
%           1. amount of cells found in the image
%           2. amount of cytoplasma cells
%           3. amount of golgi cells
%           4. amount of membram cells
%           5. amount of manual cells - which are stored separately (no 
%              longer used - empty slot for expansion)
%           6. amount of golgi strong cells - are named undecided (old)
%           7. amount of invalid cells - due to size, location or amount of
%              cell nuclei (only one nucleus per cell allowed)
%
% PARAMETER:
%       The following parameter are set due to experience but can be
%       changed for a different type of image. Each parameter is commented
%       and its function is explained.
%
% STEPS:
%   1: Start and check input data
%   2: Segment cell nuclei in hoechst -> imgCellKernelSegment
%   3: Segment cells in gfp using nuclei results -> imgCellSegment
%   4: Cut cell out and classify it. Both GFP & Cherry -> Erg
%==========================================================================

%PARAMETER
%=========

%general
%~~~~~~~
labelCon = 8; %connectivity for neighbourhood area

medianSize = 5; %size of mask for median filter for denoising

%disksizes for morphological operations
disksize = 3;    %general
disksize2 = 2;   %smooth the kernel
disksize3 = 3;   %detect the borders

%nucleus segmentation
%~~~~~~~~~~~~~~~~~~~~
%quality control
minCellKernelSize = 100;     %minumum nucleus size
maxCellKernelSize = 1000;    %maximum nucleus size

%quality control
distanceCellKernelToBorder = 10;     %minimum distance to image border - otherwise only a too small part of the cell is imaged.

thresholdMedianCellKernelSize = 370;  %demanded median size of nuclei

markMedianCellKernelSize = 200;  %mark to adapt search step - median size of nuclei
minimumAmountOfNuclei = 15;      %mark to adapt search step - minimum amount of cell nucluei

%search step sizes for threshold segmentation
highNucleiThresholdStep = 200;  
normalNucleiThresholdStep = 100;
lowNucleiThresholdStep = 25;

%cell segmentation
%~~~~~~~~~~~~~~~~~
iterationRegionGrowing = 25;   %iteration steps in the region growing algo

%quality control
minCellSize = 400;     %minumum cell size
maxCellSize = 2500;    %maximum cell size

maximumGradientValue = 1;   %second property for cell segmentation

%classification
%~~~~~~~~~~~~~~
sizeROIX = 100;      %size of ROI considered when classifying a single cell
sizeROIY = 100;

midROIX = 50;        %mid of the ROI area.
midROIY = 50;

%quality control
minAmountInteriorValues = 60;   %robustness: minimum required amount of interior values

%use parts of areas
percentageBoundary = 0.4;    %use highest percentage of boundary values - due to blurring of thin membran
percentageKernel = 0.8;      %use lowest percentage of nucleus values - bright golgi might influence high values  

weightInterior = 0.95; %golgi strong - weight of interior when compared to boundary GFP
weightInteriorCherry = 0.95; %golgi strong - weight of interior when compared to boundary Cherry

minStAbwTotal = 40; %glogi strong - minimal required standard deviation for total cell GFP
minStAbwTotalCherry = 40; %glogi strong - minimal required standard deviation for total cell Cherry

%decision function (GFP:
%   if meanBou < weightInterior*meanInt && meanKer2 < meanInt && stabwTotal > minStAbwTotal
%       fprintf('GOLGI STRONG\n');
%   elseif meanBou < meanInt && meanKer2 < meanInt
%       fprintf('GOLGI\n');
%   elseif meanBou > meanKer2 && meanBou > meanInt 
%       fprintf('MEMBRAN\n');
%   elseif meanKer2 > meanInt 
%       fprintf('CYTOPLASMA\n');
%   else
%       fprintf('***INVALID - out of function***\n');
%   end     
        
% Classification function GFP start line 895 and Cherry in line 1027. 
% It is a custom function using if/else constructs to realize the decision 
% function. The decision function is tree like so the features for golgi 
% are tested first. If the cell does not match any criteria it is marked as 
% INVALID.

%==========================================================================

%test input parameter boolCherryImage, boolCherrySegmentation

if boolCherryImage == 0 && boolCherrySegmentation == 1  %impossible, cannot do a cherry segmentation with no cherry image.
    
    if boolPause == 1
       fprintf('Segmentation with Cherry not possible since there is no Cherry image\n'); 
    end
    
    Erg = [-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1];
else

if boolPause == 1
    %show images
    figure(1);
    imagesc(imgGFP);
    colormap(gray);
    title('GFP image'); 
        
    if boolCherryImage == 1 
        figure(2);
        imagesc(imgCherry);
        colormap(gray);
        title('Cherry image');          
    end

    figure(3);
    imagesc(imgHoechst);
    colormap(gray);
    title('Hoechst image');      
end

%Step1: Start and check input data
%=================================
if boolPause == 1
    fprintf('Analysis started...\n');
end

se = strel('disk',disksize);
se2 = strel('disk',disksize2);
se3 = strel('disk',disksize3);

imgSizeGFP      = size(imgGFP);      
imgSizeCherry   = size(imgCherry);      
imgSizeHoechst  = size(imgHoechst);    

if imgSizeGFP(1) ~= imgSizeHoechst(1) && imgSizeGFP(2) ~= imgSizeHoechst(2) && imgSizeGFP(1) ~= imgSizeCherry(1) && imgSizeGFP(2) ~= imgSizeCherry(2)
    fprintf('input data do not have the same size and can not be analysed!\n');
end %sizecheck

%Step2: Segment cell nuclei in hoechst
%------------------------------------------
%INPUT:  imgHoechst
%OUTPUT: imgCellKernelSegment (binary)
%First the Hoechst image is denoised using a median fiter of size medianSize.
%Take the rounded(0.01) max value of the Hoechst image as startThreshold 
%and downsize the threshold adaptively during the search according to 
%several properties. The idea is to beginn with a coarse stepsize and have
%a fine stepsize when being near to the demanded result.
%The optimal threshold is found if the median of the size of the found
%objects reaches a special mark given by thresholdMedianCellKernelSize.
%Then an opening is done to get a smooth boundary for the cell kernels.
%After that a labeling is done and all too small and too big objects are
%removed. In addition the background areas inside the cell nuclei are 
%closed and the result is given as binary image.

if boolPause == 1
    fprintf('Segmeting cell nuclei...\n');
end

%used matrices
%imgHoechstSeg = zeros(imgSizeHoechst);
%imgThresHoechst = zeros(imgSizeHoechst);
%imgInvertThresHoechst = zeros(imgSizeHoechst);
%imgLabelHoechst = zeros(imgSizeHoechst);
%imgInvertLabelHoechst = zeros(imgSizeHoechst);
%imgCellKernelSegment = zeros(imgSizeHoechst);

%denoise the image first to get a better
imgHoechstSeg = medianFilter(imgHoechst, medianSize);

%detect max image value in Hoechst Image and use a rounded version 
%as start for the adaptive threshold segmentation
maxHoechst = max(max(imgHoechstSeg));
maxHoechst = round(maxHoechst/100)*100;

%used parameter
boolCond = 1; %search in progress
backgroundThresholdHoechst = maxHoechst;

%adaptive threshold segmentation
imgThresHoechst = zeros(imgSizeHoechst);
while(boolCond)
    
    for i=1:imgSizeHoechst(1)
        for j=1:imgSizeHoechst(2)
            if imgHoechstSeg(i,j) > backgroundThresholdHoechst
               imgThresHoechst(i,j) = 1;
            else
               imgThresHoechst(i,j) = 0;
            end %if1
        end %for2
    end %for1
    
    imgLabelHoechst = bwlabel(imgThresHoechst, labelCon);
    
    amountCellKernels = max(max(imgLabelHoechst));

    sizeCellKernels = zeros(amountCellKernels, 1);

    for i=1:imgSizeHoechst(1)
        for j=1:imgSizeHoechst(2)
            if imgLabelHoechst(i,j) ~= 0
                sizeCellKernels(imgLabelHoechst(i,j)) = sizeCellKernels(imgLabelHoechst(i,j)) + 1;
            end %if1
        end %for2
    end %for1

    medianCellKernelSize = median(sizeCellKernels);
    
    if medianCellKernelSize > thresholdMedianCellKernelSize || backgroundThresholdHoechst == 0
        if amountCellKernels > minimumAmountOfNuclei 
            boolCond = 0;
        else
            backgroundThresholdHoechst = backgroundThresholdHoechst - normalNucleiThresholdStep;
        end
    else
        if amountCellKernels > minimumAmountOfNuclei && medianCellKernelSize > markMedianCellKernelSize
            backgroundThresholdHoechst = backgroundThresholdHoechst - lowNucleiThresholdStep;
        elseif amountCellKernels > minimumAmountOfNuclei || medianCellKernelSize > markMedianCellKernelSize
            backgroundThresholdHoechst = backgroundThresholdHoechst - normalNucleiThresholdStep;
        else
            backgroundThresholdHoechst = backgroundThresholdHoechst - highNucleiThresholdStep;    
        end %if2
    end % if1
    
    %check if anaylsis still reasonable
    if backgroundThresholdHoechst < 0
        boolCond = 0;
    end
    
end %while1

%perform an opening to smooth the edges
imgThresHoechst = imerode(imgThresHoechst,se);
imgThresHoechst = imdilate(imgThresHoechst,se);

%remove all too small and too big objects
%labelling, counting of sizes, removal.
imgLabelHoechst= bwlabel(imgThresHoechst, labelCon);
    
amountCellKernels = max(max(imgLabelHoechst));
sizeCellKernels = zeros(amountCellKernels, 1);
    
for i=1:imgSizeHoechst(1)
    for j=1:imgSizeHoechst(2)
        if imgLabelHoechst(i,j) ~= 0
            sizeCellKernels(imgLabelHoechst(i,j)) = sizeCellKernels(imgLabelHoechst(i,j)) + 1;
        end %if1
    end %for2
end %for1
    
for i=1:imgSizeHoechst(1)
    for j=1:imgSizeHoechst(2)
        if imgLabelHoechst(i,j) ~= 0
            if sizeCellKernels(imgLabelHoechst(i,j)) < minCellKernelSize || sizeCellKernels(imgLabelHoechst(i,j)) > maxCellKernelSize
                imgThresHoechst(i,j) = 0;
            end %if2
        end %if1
    end %for2
end %for1

imgCellKernelSegment = imgThresHoechst;

%close all areas within the cell kernels
%invert the image, search background as biggest labeled area,
%set all non background areas to 1 in threshold image
imgInvertThresHoechst = 1 - imgThresHoechst;

imgInvertLabelHoechst = bwlabel(imgInvertThresHoechst, labelCon);

amountAreas = max(max(imgInvertLabelHoechst));
sizeAreas = zeros(amountAreas, 1);

for i=1:imgSizeHoechst(1)
    for j=1:imgSizeHoechst(2)
        if imgInvertLabelHoechst(i,j) ~= 0
            sizeAreas(imgInvertLabelHoechst(i,j)) = sizeAreas(imgInvertLabelHoechst(i,j)) + 1;
        end %if1
    end %for2
end %for1

for i=1:amountAreas
    if sizeAreas(i) == max(sizeAreas)
        backgroundArea = i;
    end %if1
end %for1

for i=1:imgSizeHoechst(1)
    for j=1:imgSizeHoechst(2)
        if imgInvertLabelHoechst(i,j) ~= 0 && imgInvertLabelHoechst(i,j) ~= backgroundArea
                imgCellKernelSegment(i,j) = 1;
        end %if1
    end %for2
end %for1

%clear close areas within kernels
clear amountAreas;
clear sizeAreas;
clear backgroundArea;
clear amountCellKernels;
clear sizeCellKernels;

%clear not longer matrices
clear imgHoechstSeg;
clear imgThresHoechst;
clear imgInvertThresHoechst;
clear imgLabelHoechst;
clear imgInvertLabelHoechst;
    
if boolPause == 1
    figure(4);
    imagesc(imgCellKernelSegment);
    colormap(gray);
    title('Nuclei segmentation');      
end

%Step3: Segment cells in gfp using nuclei results
%------------------------------------------------
%INPUT:  imgCellKernelSegment is given as result from first step
%        imgGFP
%OUTPUT: imgCellSegment
%First the GFP image is denoised using a median fiter. 
%Then the local minimum GFP value of each kernel is calculated. This value
%is used in a region growing algorithm which segments each cell. The main
%advantage is that dark and bright cells are segmented separatly.
%In addition, 

%use region growing to segment cells as long as there are still not 
%investigated pixels which can belong to cells.
%use a two phase approach: first mark neighborhood cells (8) and
%                          second decide whether pixel belongs to cell
%                          according to the gray value of the pixel
%optimization: each pixel can only be investigated once

if boolPause == 1
    if boolCherrySegmentation ~= 1
        fprintf('Segmeting cells in GFP...\n');
    else
        fprintf('Segmeting cells in Cherry...\n');
    end
end

%parameters
boolCond = 1;

%used matrices
%imgGFPSeg = zeros(imgSizeGFP);
%imgLabelCellKernels = zeros(imgSizeGFP);
%imgGrowingCells = zeros(imgSizeGFP);
%imgPixelInvestigated = zeros(imgSizeGFP);
%imgCellSegment = zeros(imgSizeGFP);
%imgGFPGradientX = zeros(imgSizeGFP);
%imgGFPGradientY = zeros(imgSizeGFP);
%imgGFPGradientTotal = zeros(imgSizeGFP);

%denoise the image and choose which image to take. GFP or Cherry
if boolCherrySegmentation == 1
    imgGFPSeg = medianFilter(imgCherry, medianSize);  %denoise the image  
else
    imgGFPSeg = medianFilter(imgGFP, medianSize);  %denoise the image
end

 maxGFP = max(max(imgGFPSeg));%calc maximum of denoised GFP image

%calc amount of cells
imgLabelCellKernels = bwlabel(imgCellKernelSegment, labelCon);
amountCellKernels = max(max(imgLabelCellKernels));

%calc local min of all cell kernels
minCells = ones(amountCellKernels, 1) * maxGFP;

%calculate minimum GFP value for each cell kernel
for i=1:imgSizeGFP(1)
    for j=1:imgSizeGFP(2)
        if imgLabelCellKernels(i,j) ~= 0
            if imgGFPSeg(i,j) < minCells(imgLabelCellKernels(i,j))
                minCells(imgLabelCellKernels(i,j)) = imgGFPSeg(i,j);       
            end %if2
        end %if1
    end %for2
end %for1

markGrowing = amountCellKernels + 20; %used to mark all surrounding pixels for a certain cell (additional).

imgGrowingCells = imgLabelCellKernels;
imgPixelInvestigated = imgCellKernelSegment;

count = 0;
while((boolCond) && (count < iterationRegionGrowing))
    
    change = 0;
    count = count +1;
    
    %mark all boundary cells
    for i=2:imgSizeGFP(1)-1
        for j=2:imgSizeGFP(2)-1
            if (imgGrowingCells(i,j) == 0) && (imgPixelInvestigated(i,j) == 0)
                if (imgGrowingCells(i-1,j-1) ~= 0) && (imgGrowingCells(i-1,j-1) < markGrowing) %1
                    imgGrowingCells(i,j) = imgGrowingCells(i-1,j-1)+ markGrowing;
                elseif (imgGrowingCells(i-1,j) ~= 0) && (imgGrowingCells(i-1,j) < markGrowing) %2
                    imgGrowingCells(i,j) = imgGrowingCells(i-1,j)+ markGrowing;
                elseif (imgGrowingCells(i-1,j+1) ~= 0) && (imgGrowingCells(i-1,j+1) < markGrowing) %3
                    imgGrowingCells(i,j) = imgGrowingCells(i-1,j+1)+ markGrowing;
                elseif (imgGrowingCells(i,j-1) ~= 0) && (imgGrowingCells(i,j-1) < markGrowing) %4
                    imgGrowingCells(i,j) = imgGrowingCells(i,j-1)+ markGrowing;
                elseif (imgGrowingCells(i,j+1) ~= 0) && (imgGrowingCells(i,j+1) < markGrowing) %5
                    imgGrowingCells(i,j) = imgGrowingCells(i,j+1)+ markGrowing;
                elseif (imgGrowingCells(i+1,j-1) ~= 0) && (imgGrowingCells(i+1,j-1) < markGrowing) %6
                    imgGrowingCells(i,j) = imgGrowingCells(i+1,j-1)+ markGrowing;  
                elseif (imgGrowingCells(i+1,j) ~= 0) && (imgGrowingCells(i+1,j) < markGrowing) %7
                    imgGrowingCells(i,j) = imgGrowingCells(i+1,j)+ markGrowing;
                elseif (imgGrowingCells(i+1,j+1) ~= 0) && (imgGrowingCells(i+1,j+1) < markGrowing) %8
                    imgGrowingCells(i,j) = imgGrowingCells(i+1,j+1)+ markGrowing; 
                end % if2
            end %if1
        end %for2
    end %for1
    
    %calculate gradient
    imgGFPGradientX = zeros(imgSizeGFP);
    imgGFPGradientY = zeros(imgSizeGFP);

    for i = 2:imgSizeGFP(1)-1
        for j = 2:imgSizeGFP(2)-1
            imgGFPGradientX(i,j) = imgGFPSeg(i+1,j) - imgGFPSeg(i-1,j);
            imgGFPGradientY(i,j) = imgGFPSeg(i,j+1) - imgGFPSeg(i,j-1);
        end
    end
    
    imgGFPGradientTotal = sqrt(imgGFPGradientX.*imgGFPGradientX+imgGFPGradientY.*imgGFPGradientY);
    
    %decide about growing - classifier enhanced strategy (local minimum and value ofgradient
    for i=2:imgSizeGFP(1)-1
        for j=2:imgSizeGFP(2)-1
            if (imgGrowingCells(i,j) > markGrowing)
                if ( imgGFPSeg(i,j) > minCells(imgGrowingCells(i,j) - markGrowing) || imgGFPGradientTotal(i,j) < maximumGradientValue ) 
                    imgGrowingCells(i,j) = imgGrowingCells(i,j) - markGrowing;
                    imgPixelInvestigated(i,j) = 2;
                    change = 1;
                else
                    imgGrowingCells(i,j) = 0;
                    imgPixelInvestigated(i,j) = 2;
                end %if2
            end %if1
        end %for2
    end %for1
    
    if change == 0
        boolCond = 0;
    end %if1
    
end %while

imgCellSegment = imgGrowingCells;

%clear unused matrices
clear imgGFPSeg;
clear imgLabelCellKernels;
clear imgGrowingCells;
clear imgPixelInvestigated;
clear imgGFPGradientX;
clear imgGFPGradientY;
clear imgGFPGradientTotal;

if boolPause == 1
    figure(5);
    imagesc(imgCellSegment);
    colormap(gray);
    title('Cell segmentation');            
end

%Step4: Cut cell out and classify it. Both GFP & Cherry  
%------------------------------------
%INPUT:  imgCellKernelSegment is given as result from first step
%        imgCellSegment(labeled)
%        imgGFP
%        imgCherry
%OUTPUT: Erg
%
%   First the centers of all cellnuclei are calculated.
%   Then each cell is cut out and analysed as follows:
%       -postprocess to improve cell quality
%       -decision whether cell is valid (size, location, amount nuclei)
%       -separate nucleus, interior and border of the cell
%       -extract values for GFP and Cherry channel
%       -classify cell 
if boolPause == 1
    fprintf('Classifying cells...\n');
end

%used matrices
%imgCellKernelCentre = zeros(imgSizeHoechst);
%imgLabelKernel = zeros(imgSizeHoechst);
%Image = zeros(sizeROIX, sizeROIY);
%ImageCherry = zeros(sizeROIX, sizeROIY);
%ImageHoechst = zeros(sizeROIX, sizeROIY);
%SegImage = zeros(sizeROIX, sizeROIY);
%SegKernel = zeros(sizeROIX, sizeROIY);
%CellKernel = zeros(sizeROIX, sizeROIY);
%CellInterior = zeros(sizeROIX, sizeROIY);
%CellBoundary = zeros(sizeROIX, sizeROIY);
%CellValues = zeros(sizeROIX, sizeROIY);
%CellCherryValues = zeros(sizeROIX, sizeROIY);
%CellLabeled = zeros(sizeROIX, sizeROIY);
%CellKernelLabeled = zeros(sizeROIX, sizeROIY);
%SaveImage = zeros(sizeROIX, 2*sizeROIY);

%first calculate the centres of the cell kernels
imgLabelKernel = bwlabel(imgCellKernelSegment, labelCon);
amountOfKernels = max(max(imgLabelKernel));

KernelsAmount = zeros(amountOfKernels, 1);
KernelsXKoord = zeros(amountOfKernels, 1);
KernelsYKoord = zeros(amountOfKernels, 1);

for i=1:imgSizeHoechst(1)
    for j=1:imgSizeHoechst(2)
        if imgLabelKernel(i,j)  ~= 0
            KernelsAmount(imgLabelKernel(i,j)) = KernelsAmount(imgLabelKernel(i,j)) + 1;
            KernelsXKoord(imgLabelKernel(i,j)) = KernelsXKoord(imgLabelKernel(i,j)) + i;
            KernelsYKoord(imgLabelKernel(i,j)) = KernelsYKoord(imgLabelKernel(i,j)) + j;
        end %if1
    end %for2
end %for1

for n=1:amountOfKernels
    KernelsXKoord(n) = round(KernelsXKoord(n)/KernelsAmount(n));
    KernelsYKoord(n) = round(KernelsYKoord(n)/KernelsAmount(n));
end %for1

imgCellKernelCentre = zeros(imgSizeHoechst);
for n=1:amountOfKernels
    imgCellKernelCentre(KernelsXKoord(n), KernelsYKoord(n)) = 1;
end %for1

%cut cell ROI and analyse each cell separately
Image = zeros(sizeROIX, sizeROIY);
ImageCherry = zeros(sizeROIX, sizeROIY);
ImageHoechst = zeros(sizeROIX, sizeROIY);
SegImage = zeros(sizeROIX, sizeROIY);
SegKernel = zeros(sizeROIX, sizeROIY);

amountCells = amountOfKernels;
amountGlogi = 0;
amountCyto = 0;
amountMembran = 0;
amountManuel = 0;
amountUndecided = 0;
amountInvalid = 0;

%-1 in case there is no cherry image
if boolCherryImage == 1 
    amountCherryCells = amountOfKernels;
    amountCherryGlogi = 0;
    amountCherryCyto = 0;
    amountCherryMembran = 0;
    amountCherryManuel = 0;
    amountCherryUndecided = 0;
    amountCherryInvalid = 0;
else
    amountCherryCells = -1;
    amountCherryGlogi = -1;
    amountCherryCyto = -1;
    amountCherryMembran = -1;
    amountCherryManuel = -1;
    amountCherryUndecided = -1;
    amountCherryInvalid = -1;
end

for n=1:amountOfKernels
    
    if boolPause == 1
        fprintf('Cell no %i\n',n);
    end
    
    cellValid = 1;
    
    %cut cell out - set values to 0 if image boundarys are exceeded
    for i=1:sizeROIX
        for j=1:sizeROIY
            if ((KernelsXKoord(n)-midROIX+i) < 1 || (KernelsXKoord(n)-midROIX+i) > imgSizeGFP(1) || (KernelsYKoord(n)-midROIY+j) < 1 || (KernelsYKoord(n)-midROIY+j) > imgSizeGFP(2))
                Image(i,j) = 0;
                ImageHoechst(i,j) = 0;
                SegImage(i,j) = 0;
                SegKernel(i,j) = 0;
                CellSeg(i,j) = 0;
            else %if1
                Image(i,j) = imgGFP((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                ImageCherry(i,j) = imgCherry((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                ImageHoechst(i,j) = imgHoechst((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                SegImage(i,j) = imgCellSegment((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j)) + n*imgCellKernelSegment((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                SegKernel(i,j) = imgCellKernelSegment((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                if imgCellSegment((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j)) == n
                    CellSeg(i,j) = 1;
                else %if2
                    CellSeg(i,j) = 0;
                end %if2
            end %if1
        end %for2
    end %for1
    
    %postprocessing of segmented cell
    %perform a closing and then an opening
    CellSeg = imdilate(CellSeg,se);
    CellSeg = imerode(CellSeg,se);

    CellSeg = imerode(CellSeg,se);
    CellSeg = imdilate(CellSeg,se);

    %remove not connected areas from the cell
    CellLabeled = bwlabel(CellSeg, labelCon);
    if( max(max(CellLabeled)) > 1)
        %more than one object found. Remove smaller ones
        amountObjects = max(max(CellLabeled));

        sizeObjects = zeros(amountObjects, 1);

        for i=1:sizeROIX 
            for j=1:sizeROIY 
                if CellLabeled(i,j) ~= 0
                    sizeObjects(CellLabeled(i,j)) = sizeObjects(CellLabeled(i,j)) + 1;
                end %if2
            end %for2
        end %for1
        
        maxSize = max(sizeObjects);

        for i=1:sizeROIX
            for j=1:sizeROIY
                if CellLabeled(i,j) ~= 0 
                    if sizeObjects(CellLabeled(i,j)) ~= maxSize
                        CellSeg(i,j) = 0;
                    end %if3
                end %if2
            end %for2
        end %for1       
    end %if1
    
    %quality control: cell size
    cellSize = sum(sum(CellSeg));
    
    if cellSize < minCellSize || cellSize > maxCellSize
        if boolPause == 1
            fprintf('Cell too big or too small!\n');
        end
        cellValid = 0;  %cell invalid 1
    end %if1
    
    %extract the cellkernel from the image
    CellKernel = CellSeg.*SegKernel;
    
    %deblurr the kernel - remove border area of nucleus
    CellKernel = imerode(CellKernel,se2);
    
    %if deblurred kernel contains two parts ->skip
    CellKernelLabel = bwlabel(CellKernel, labelCon);
    if( max(max(CellKernelLabel)) > 1)
        if boolPause == 1
            fprintf('More than one cell kernel!\n');
        end
        cellValid = 0; %cell invalid 2
    end %if1
    
    %detect the interior and border
    CellInterior = imerode(CellSeg,se3) - CellKernel;
    CellBoundary = CellSeg - CellInterior - CellKernel;  

    if( KernelsXKoord(n) < distanceCellKernelToBorder || imgSizeGFP(1)-distanceCellKernelToBorder < KernelsXKoord(n) || KernelsYKoord(n) < distanceCellKernelToBorder || imgSizeGFP(2)-distanceCellKernelToBorder < KernelsYKoord(n) )
        if boolPause == 1
            fprintf('Cellkernel too close to border!\n');
        end
        cellValid = 0; %cell invalid 3
    end
    
	%sizes of areas
    amountInt = sum(sum(CellInterior));
    amountBou = sum(sum(CellBoundary));
    amountKer = sum(sum(CellKernel));
    
    %vectors for values gfp    
    valuesCell = zeros(cellSize,1);
    valuesInt = zeros(amountInt,1);
    valuesBou = zeros(amountBou,1);
    valuesKer = zeros(amountKer,1);
        
    %vectors for values cherry
    valuesCellCherry = zeros(cellSize,1);
    valuesCherryInt = zeros(amountInt,1);
    valuesCherryBou = zeros(amountBou,1);
    valuesCherryKer = zeros(amountKer,1);
    
    %all considered cell values
    CellValues = zeros(sizeROIX, sizeROIY);
    CellCherryValues = zeros(sizeROIX, sizeROIY);
    
    %counts to store values
    countCell = 1;
    countInt = 1;
    countBou = 1;
    countKer = 1;
    
    %extract the values
    for i=1:sizeROIX 
        for j=1:sizeROIY 
            if ((KernelsXKoord(n)-midROIX+i) < 1 || (KernelsXKoord(n)-midROIX+i) > imgSizeGFP(1) || (KernelsYKoord(n)-midROIY+j) < 1 || (KernelsYKoord(n)-midROIY+j) > imgSizeGFP(2))
                CellValues(i,j) = 0;
                CellCherryValues(i,j) = 0;
            else %if1
                %Cell value
                if CellSeg(i,j) == 1;
                    valuesCell(countCell) = imgGFP((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                    valuesCellCherry(countCell) = imgCherry((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                    countCell = countCell+1;
                    CellValues(i,j) = imgGFP((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                    CellCherryValues(i,j) = imgCherry((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                else %if2
                    CellValues(i,j) = 0;
                    CellCherryValues(i,j) = 0;
                end %if2
                %Cell nucleus values
                if CellKernel(i,j) == 1;
                    valuesKer(countKer) = imgGFP((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                    valuesCherryKer(countKer) = imgCherry((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                    countKer = countKer+1;
                end %if2
                %Cell interior value
                if CellInterior(i,j) == 1;
                    valuesInt(countInt) = imgGFP((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                    valuesCherryInt(countInt) = imgCherry((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                    countInt = countInt+1;
                end %if2
                %Cell boundary values
                if CellBoundary(i,j) == 1;
                    valuesBou(countBou) = imgGFP((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                    valuesCherryBou(countBou) = imgCherry((KernelsXKoord(n)-midROIX+i),(KernelsYKoord(n)-midROIY+j));
                    countBou = countBou+1;
                end %if2
            end %if1
        end %for2
    end %for1    
    
    %calculate values used for decision GFP
    
    %Total average:meanTotal standard deviation:stabwTotal
    %Interior average:meanInt standard deviation:stabwInt
    %Nucleus average:meanKer standard deviation:stabwKer
    %Nucleus(corrected) average:meanKer2 standard deviation:stabwKer2
    %Boundary average:meanBou standard deviation:stabwBou
    
    %The corrected nucleus values are percentageKernel% of all nucleus values. In fact
    %the lower percentageKernel% are taken. This is due to the possibility that the golgi
    %can have some values inside the cell nucleus
    
    %For the boundary values the higher percentageBoundary% of all values are taken for the
    %analysis. Due to blurring of the thin cell membrans.
    
    %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    %average Total
    sumTotal = sum(valuesCell);  
    meanTotal = sumTotal / cellSize;
    
    %standard deviation Total
    sumDiffTotal = double(0);
    for i=1:cellSize
        sumDiffTotal = sumDiffTotal + double((meanTotal-valuesCell(i)))*double((meanTotal-valuesCell(i)));
    end %for1
    stabwTotal = sqrt(double(sumDiffTotal)/double(cellSize));
    
    %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    %average Interior
    sumInt = sum(valuesInt);    
    meanInt = sumInt / amountInt;
    
    %standard deviation Interior
    sumDiffInt = double(0);
    for i=1:amountInt
        sumDiffInt = sumDiffInt + double((meanInt-valuesInt(i)))*double((meanInt-valuesInt(i)));
    end %for1
    stabwInt = sqrt(double(sumDiffInt)/double(amountInt));
    
    %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    %average Kern
    sumKer = sum(valuesKer);   
    meanKer = sumKer / amountKer;
    
    %standard deviation kernel
    sumDiffKer = 0;
    for i=1:amountKer
        sumDiffKer = sumDiffKer + double((meanKer-valuesKer(i)))*double((meanKer-valuesKer(i)));
    end %for1 
    stabwKer = sqrt(double(sumDiffKer)/double(amountKer));
    
    %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    %average Kern(korrigiert)
    valuesKer_sort = sort(valuesKer);    
    amountKer_sort = round(amountKer*percentageKernel);
    valuesKer_sort = valuesKer_sort(1:amountKer_sort);
    
    sumValuesKer2 = sum(sum(valuesKer_sort));
    meanKer2 = sumValuesKer2/amountKer_sort;
    
    %standard deviation kernel(korrigiert)
    sumDiffKer2 = 0;
    for i=1:amountKer_sort
        sumDiffKer2 = sumDiffKer2 + double((meanKer2-valuesKer_sort(i)))*double((meanKer2-valuesKer_sort(i)));
    end %for1 
    stabwKer2 = sqrt(double(sumDiffKer2)/double(amountKer_sort)); 
    
    %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    %average Boundary

    %sumBou2 = sum(valuesBou);
    %meanBou2 = sumBou2 / amountBou;    %not used - use percentage
        
    valuesBou_sort = sort(valuesBou);   
    valuesBou_sort = valuesBou_sort(end-round(amountBou*percentageBoundary)+1:end);
    
    sumBou = sum(valuesBou_sort);  
    meanBou = sumBou / size(valuesBou_sort,1);

    %standard deviation Boundary
    sumDiffBou = 0; 
    for i=1:size(valuesBou_sort,1)
        sumDiffBou = sumDiffBou + double((meanBou-valuesBou_sort(i)))*double((meanBou-valuesBou_sort(i)));
    end %for1 
    stabwBou = sqrt(double(sumDiffBou)/double(amountBou)); 
    
    %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    %show all the possible values to create a decision function
    
    if boolPause == 1
        fprintf('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n');
        fprintf('Features for GFP classification\n');
        fprintf('The investigates cell has %i nucleus pixels, %i interior and %i border pixels.\n', amountKer, amountInt, amountBou);
        fprintf('Whole cell: average: %f and standard deviation %f.\n', meanTotal, stabwTotal);
        fprintf('Average Nuc: %i Nuc(corrected): %i Int: %i Bou: %i \n', meanKer, meanKer2, meanInt, meanBou);
        fprintf('Standard deviation Nuc: %i Nuc(corrected): %i Int: %i Bou: %i \n', stabwKer, stabwKer2, stabwInt, stabwBou);
    end
    
    %decision function GFP    
    if cellValid == 0 || amountInt < minAmountInteriorValues %quality control - invalid (previous test || formally manual)
        amountInvalid = amountInvalid +1;
        if boolPause == 1
            fprintf('***INVALID***\n');
        end 
    else
        if meanBou < weightInterior*meanInt && meanKer2 < meanInt && stabwTotal > minStAbwTotal %golgi strong 
            amountUndecided = amountUndecided +1;
            if boolPause == 1
                fprintf('GOLGI STRONG\n');
            end  
        elseif meanBou < meanInt && meanKer2 < meanInt %golgi
            amountGlogi = amountGlogi +1;
            if boolPause == 1
                fprintf('GOLGI\n');
            end             
        elseif meanBou > meanKer2 && meanBou > meanInt %membran
            amountMembran = amountMembran +1;
            if boolPause == 1
                fprintf('MEMBRAN\n');
            end 
        elseif meanKer2 > meanInt %cytoplasma
            amountCyto = amountCyto +1;
            if boolPause == 1
                fprintf('CYTOPLASMA\n');
            end
        else
            amountInvalid = amountInvalid +1; %Invalid
            if boolPause == 1
                fprintf('***INVALID - out of function***\n');
            end
        end %if2      
    end %if1

    if boolCherryImage == 1 
        %calculate values used for decision Cherry

        %Total average:meanTotalCherry standard deviation:stabwTotalCherry
        %Interior average:meanInt standard deviation:stabwInt
        %Nucleus average:meanKer standard deviation:stabwKer
        %Nucleus(corrected) average:meanKer2 standard deviation:stabwKer2
        %Boundary average:meanBou standard deviation:stabwBou

        %The corrected nucleus values are percentageKernel% of all nucleus values. In fact
        %the lower percentageKernel% are taken. This is due to the possibility that the golgi
        %can have some values inside the cell nucleus

        %For the boundary values the higher percentageBoundary% of all values are taken for the
        %analysis. This is because of blurr.

        %average Total
        %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        sumTotalCherry = sum(valuesCellCherry);   
        meanTotalCherry = sumTotalCherry / cellSize;

        %standard deviation Total
        sumDiffTotalCherry = double(0);
        for i=1:cellSize
            sumDiffTotalCherry = sumDiffTotalCherry + double((meanTotalCherry-valuesCellCherry(i)))*double((meanTotalCherry-valuesCellCherry(i)));
        end %for1
        stabwTotalCherry = sqrt(double(sumDiffTotalCherry)/double(cellSize));

        %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        %average interior
        sumInt = sum(valuesCherryInt);   
        meanInt = sumInt / amountInt;

        %standard deviation Interior
        sumDiffIntCherry = double(0);
        for i=1:amountInt
            sumDiffIntCherry = sumDiffIntCherry + double((meanInt-valuesCherryInt(i)))*double((meanInt-valuesCherryInt(i)));
        end %for1
        stabwIntCherry = sqrt(double(sumDiffIntCherry)/double(amountInt));

        %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        %average kern
        sumKer = sum(valuesCherryKer);  
        meanKer = sumKer / amountKer;

        %standard deviation kernel
        sumDiffKerCherry = double(0);
        for i=1:amountKer
            sumDiffKerCherry = sumDiffKerCherry + double((meanKer-valuesCherryKer(i)))*double((meanKer-valuesCherryKer(i)));
        end %for1 
        stabwKerCherry = sqrt(double(sumDiffKerCherry)/double(amountKer));

        %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        %average kern (korrigiert)
        valuesCherryKer_sort = sort(valuesCherryKer);
        amountKer_sort = round(amountKer*percentageKernel);
        valuesCherryKer_sort = valuesCherryKer_sort(1:amountKer_sort);

        sumValuesKer2 = sum(sum(valuesCherryKer_sort));
        meanKer2 = sumValuesKer2/amountKer_sort;

        %standard deviation kernel(korrigiert)
        sumDiffKer2Cherry = double(0);
        for i=1:amountKer_sort
            sumDiffKer2Cherry = sumDiffKer2Cherry + double((meanKer2-valuesCherryKer_sort(i)))*double((meanKer2-valuesCherryKer_sort(i)));
        end %for1 
        stabwKer2Cherry = sqrt(double(sumDiffKer2Cherry)/double(amountKer_sort)); 

        %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        %average Boundary
        %sumBou2 = sum(valuesCherryBou);
        %meanBou2 = sumBou2 / amountBou;   %not used - use percentage

        valuesCherryBou_sort = sort(valuesCherryBou);
        valuesCherryBou_sort = valuesCherryBou_sort(end-round(amountBou*percentageBoundary)+1:end);    

        sumBou = sum(valuesCherryBou_sort);    
        meanBou = sumBou / size(valuesCherryBou_sort,1);

        %standard deviation Boundary
        sumDiffBouCherry = double(0);
        for i=1:size(valuesCherryBou_sort,1)
            sumDiffBouCherry = sumDiffBouCherry + double((meanBou-valuesCherryBou_sort(i)))*double((meanBou-valuesCherryBou_sort(i)));
        end %for1 
        stabwBouCherry = sqrt(double(sumDiffBouCherry)/double(amountBou)); 

        %~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        %show all values
        if boolPause == 1
            fprintf('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n');
            fprintf('Features for Cherry classification\n');
            fprintf('The investigates cell has %i kernel pixel, %i interior and %i border pixels.\n', amountKer, amountInt, amountBou);
            fprintf('Whole cell: average: %f und standard deviation %f.\n', meanTotalCherry, stabwTotalCherry);
            fprintf('Average Nuc: %i Nuc(corrected): %i Int: %i Bou: %i \n', meanKer, meanKer2, meanInt, meanBou);
            fprintf('Standard deviation Nuc: %i Nuc(corrected): %i Int: %i Bou: %i \n', stabwKerCherry, stabwKer2Cherry, stabwIntCherry, stabwBouCherry);
        end %if1

        %decision function Cherry
        if cellValid == 0 || amountInt < minAmountInteriorValues %quality control - invalid (previous test || formally manual)
            amountCherryInvalid = amountCherryInvalid +1;
            if boolPause == 1
                fprintf('***INVALID***\n');
            end   
        else
            if meanBou < weightInteriorCherry*meanInt && meanKer2 < meanInt && stabwTotalCherry > minStAbwTotalCherry %golgi strong 
                amountCherryUndecided = amountCherryUndecided +1;
                if boolPause == 1
                    fprintf('GOLGI STRONG\n');
                end   
            elseif meanBou < meanInt && meanKer2 < meanInt %golgi
                amountCherryGlogi = amountCherryGlogi +1;
                if boolPause == 1
                    fprintf('GOLGI\n');
                end            
           elseif meanBou > meanKer2 && meanBou > meanInt %membran
                amountCherryMembran = amountCherryMembran +1;
                if boolPause == 1
                    fprintf('MEMBRAN\n');
                end 
            elseif meanKer2 > meanInt  %cytoplasma
                amountCherryCyto = amountCherryCyto +1;
                if boolPause == 1
                    fprintf('CYTOPLASMA\n');
                end
            else   %invalid
                amountCherryInvalid = amountCherryInvalid +1;
                if boolPause == 1
                    fprintf('***INVALID - out of function***\n');
                end
            end %if2
        end %if1
    end %cherry image

    if boolPause == 1
        fprintf('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n');

        %show result
        figure(6);
        
        subplot(3,2,1);
        imagesc(CellValues);
        colormap(gray);
        title('GFP cell'); 
        
        if boolCherryImage == 1 
            subplot(3,2,2);
            imagesc(CellCherryValues);
            colormap(gray);
            title('Cherry cell');   
        else
            subplot(3,2,2);
            imagesc(ones(sizeROIX, sizeROIY));
            colormap(gray);
            title('no Cherry cell');   
        end

        subplot(3,2,3);
        imagesc(Image);
        colormap(gray);
        title('GFP image'); 
        
        if boolCherryImage == 1 
            subplot(3,2,4);
            imagesc(ImageCherry);
            colormap(gray);
            title('Cherry image'); 
        else
            subplot(3,2,4);
            imagesc(ones(sizeROIX, sizeROIY));
            colormap(gray);
            title('no Cherry cell');   
        end

        subplot(3,2,5);
        imagesc(SegImage);
        colormap(gray);
        title('segmented cells'); 
        
        subplot(3,2,6);
        imagesc(CellKernel*3+CellInterior*2+CellBoundary*1);
        colormap(gray);
        title('Areas cell'); 

        pause;    
    end
   
end %for1

%set outpur value
Erg = [amountCells, amountCyto, amountGlogi, amountMembran, amountManuel, amountUndecided, amountInvalid, amountCherryCells, amountCherryCyto, amountCherryGlogi, amountCherryMembran, amountCherryManuel, amountCherryUndecided, amountCherryInvalid];

%other
clear KernelsAmount;
clear KernelsXKoord;
clear KernelsYKoord;

%vectors for values gfp    
clear valuesCell;
clear valuesInt;
clear valuesBou;
clear valuesBou_sort;
clear valuesKer;
clear valuesKer_sort;
        
%vectors for values cherry
clear valuesCellCherry;
clear valuesCherryInt;
clear valuesCherryBou;
clear valuesCherryBou_sort;
clear valuesCherryKer;
clear valuesCherryKer_sort;

%100*100 matrices
clear Image;
clear ImageCherry;
clear ImageHoechst;
clear SegKernel;
clear SegImage;

clear CellSeg;
clear CellKernel;
clear CellInterior;
clear CellBoundary;

clear CellValues;
clear CellCherryValues;

clear CellLabeled;
clear CellKernelLabeled;

%big matrices
clear imgLabelKernel;
clear imgCellKernelCentre;

%End
if boolPause == 1
    
    fprintf('Result amountCells %i, amountCyto %i, amountGlogi %i, amountMembran %i, amountManuel %i, amountGolgiStrong %i, amountInvalid %i\n', amountCells, amountCyto, amountGlogi, amountMembran, amountManuel, amountUndecided, amountInvalid);
    
    if boolCherryImage == 1
        fprintf('amountCherryCells %i, amountCherryCyto %i, amountCherryGlogi %i, amountCherryMembran %i, amountCherryManuel %i, amountCherryGolgiStrong %i, amountCherryInvalid %i\n', amountCherryCells, amountCherryCyto, amountCherryGlogi, amountCherryMembran, amountCherryManuel, amountCherryUndecided, amountCherryInvalid);
    end
    
    pause;
    figure(1);close(1);
    figure(2);close(2);
    figure(3);close(3);
    figure(4);close(4);
    figure(5);close(5);
    figure(6);close(6);
end %if1
end
