%==========================================================================
%
%   CellAnalyse.m designed to manage the batch analysis of cell image files
%
%   Copyright  2010, Steffen Remmele, Julia Ritzerfeld
%
%   This program is free software; you can redistribute it and/or modify it 
%   under the terms of the GNU General Public License as published by the 
%   Free Software Foundation; either version 3 of the License, 
%   or (at your option) any later version.
%
%   This program is distributed in the hope that it will be useful, but 
%   WITHOUT ANY WARRANTY; without even the implied warranty of 
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
%   General Public License for more details.
%
%   You should have received a copy of the GNU General Public License along 
%   with this program; if not, see <http://www.gnu.org/licenses/>.
%
%   CONTACT:
%   Steffen Remmele 
%   Experimental Radiation Oncology 
%   Mannheim Medical Center - University of Heidelberg 
%   Phone: 0621-383-6916
%   Mail: steffen.remmele@ziti.uni-heidelberg.de
%   HP: www.medical-physics.de
%
%   Dr. Julia Ritzerfeld
%   Biochemie-Zentrum der Universitt Heidelberg (BZH)
%   Phone: 06221-544180
%   Mail: julia.ritzerfeld@bzh.uni-heidelberg.de
%   HP: www.uni-heidelberg.de/zentral/bzh/
%
%==========================================================================

function varargout = CellAnalyse(varargin)
%==========================================================================
%
%   Function to do a batch run of the cell analysis. Hoechst, GFP and
%   Cherry files must be located in the same folder and a folder where the 
%   result of the analysis will be stored must be given. The filenames must 
%   fulfill the following requirement (example):
%   --W00015--P00001--Z00000--T00000--Hoechst.tif
%   --W00015--P00001--Z00000--T00000--eGFP.tif
%   --W00015--P00001--Z00000--T00000--Cherry.tif
%   It is possible to change the characters at the beginning of the file
%   names but 'Hoechst', 'eGFP' and 'Cherry' should not be altered and the
%   file name size must not be changed.
%
%   The cell analysis is done using analyseImages.m
%
%   In addition, an analysis of the image quality is done using the FFT of
%   the images. The quality mark (0 worst 3 best) is then determined using
%   a combination of FFT and amount of cells. The analysis is done
%   independently from the qualtity control. The quality rating is meant to
%   help with the interpretation of the results.
%
%   Last, the results are stored in two .txt files. One for the GFP results
%   and one for the Cherry results. The single results are separated by ";" 
%   and thus an import in a spreadsheet application like excel is possible.
%
%
%==========================================================================
% CELLANALYSE M-file for CellAnalyse.fig
%
%      CELLANALYSE, by itself, creates a new CELLANALYSE or raises the existing
%      singleton*.
%
%      H = CELLANALYSE returns the handle to a new CELLANALYSE or the handle to
%      the existing singleton*.
%
%      CELLANALYSE('CALLBACK',hObject,eventData,handles,...) calls the
%      local
%      function named CALLBACK in CELLANALYSE.M with the given input arguments.
%
%      CELLANALYSE('Property','Value',...) creates a new CELLANALYSE or raises the
%      existing singleton*.  Starting from the left, property value pairs are
%      applied to the GUI before CellAnalyse_OpeningFunction gets called.  An
%      unrecognized property name or invalid value makes property application
%      stop.  All inputs are passed to CellAnalyse_OpeningFcn via varargin.
%
%      *See GUI Options on GUIDE's Tools menu.  Choose "GUI allows only one
%      instance to run (singleton)".
%
% See also: GUIDE, GUIDATA, GUIHANDLES

% Edit the above text to modify the response to help CellAnalyse

% Last Modified by GUIDE v2.5 04-Mar-2010 16:05:44
%==========================================================================

    % Begin initialization code - DO NOT EDIT
    gui_Singleton = 1;
    gui_State = struct('gui_Name',       mfilename, ...
                       'gui_Singleton',  gui_Singleton, ...
                       'gui_OpeningFcn', @CellAnalyse_OpeningFcn, ...
                       'gui_OutputFcn',  @CellAnalyse_OutputFcn, ...
                       'gui_LayoutFcn',  [] , ...
                       'gui_Callback',   []);
    if nargin && ischar(varargin{1})
        gui_State.gui_Callback = str2func(varargin{1});
    end

    if nargout
        [varargout{1:nargout}] = gui_mainfcn(gui_State, varargin{:});
    else
        gui_mainfcn(gui_State, varargin{:});
    end
    % End initialization code - DO NOT EDIT
end

% --- Executes just before CellAnalyse is made visible.
function CellAnalyse_OpeningFcn(hObject, eventdata, handles, varargin)
% This function has no output args, see OutputFcn.
% hObject    handle to figure
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
% varargin   command line arguments to CellAnalyse (see VARARGIN)

    % Choose default command line output for CellAnalyse
    handles.output = hObject;

    % Update handles structure
    guidata(hObject, handles);

    % UIWAIT makes CellAnalyse wait for user response (see UIRESUME)
    % uiwait(handles.figure1);

    %%%ADDED GLOBAL VARIABLES%%%
    %global StrImagePath;
    %global StrResultPath;
    %global amountImgFiles;
    %global filesImages;

    global resultsAnalyses;
    resultsAnalyses = [];

    %initial settings for checkboxes
    set(handles.checkboxCherryImage,'value',1);
    set(handles.checkboxCherrySegmentation,'value',0);

    %=========================================================================
    %DEBUG TEST
    %global StrImagePath;
    %global StrResultPath;
    %StrImagePath = 'J:\TestData\MIAAB\BATCHdata\';
    %StrResultPath = 'J:\TestData\MIAAB\';
    %set(handles.pathImages,'String','J:\TestData\MIAAB\BATCHdata\');
    %set(handles.pathResult,'String','J:\TestData\MIAAB\');
    %=========================================================================
end

% --- Outputs from this function are returned to the command line.
function varargout = CellAnalyse_OutputFcn(hObject, eventdata, handles)  
    % varargout  cell array for returning output args (see VARARGOUT);
    % hObject    handle to figure
    % eventdata  reserved - to be defined in a future version of MATLAB
    % handles    structure with handles and user data (see GUIDATA)

    % Get default command line output from handles structure
    varargout{1} = handles.output;
end

function pathImages_Callback(hObject, eventdata, handles)   
% hObject    handle to pathImages (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of pathImages as text
%        str2double(get(hObject,'String')) returns contents of pathImages as a double

    global StrImagePath;

    %manual entry shall have no effect
    set(handles.pathImages,'String',StrImagePath);
end

% --- Executes during object creation, after setting all properties.
function pathImages_CreateFcn(hObject, eventdata, handles)   
% hObject    handle to pathImages (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
    if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
        set(hObject,'BackgroundColor','white');
    end
end

% --- Executes on button press in buttonBrowseImages.             
function buttonBrowseImages_Callback(hObject, eventdata, handles)   
% hObject    handle to buttonBrowseImages (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

    global StrImagePath;

    imagePath = uigetdir('.', 'Ordner mit den zu analysierenden Daten auswhlen');
    StrImagePath = imagePath;
    set(handles.pathImages,'String',imagePath);
end

function pathResult_Callback(hObject, eventdata, handles)        
% hObject    handle to pathResult (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

    global StrResultPath;

    set(handles.pathResult,'String',StrResultPath);
end

% --- Executes during object creation, after setting all properties.
function pathResult_CreateFcn(hObject, eventdata, handles)       
% hObject    handle to pathResult (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
    if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
        set(hObject,'BackgroundColor','white');
    end
end

% --- Executes on button press in buttonBrowseResult.
function buttonBrowseResult_Callback(hObject, eventdata, handles)   
% hObject    handle to buttonBrowseResult (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

    global StrResultPath;

    resultPath = uigetdir('.', 'Ordner in dem die Ergebnisse gespeichert werden sollen');
    StrResultPath = resultPath;
    set(handles.pathResult,'String',resultPath);
end

% --- Executes on button press in buttonCheckPath.
function buttonCheckPath_Callback(hObject, eventdata, handles)        
% hObject    handle to buttonCheckPath (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

    global StrImagePath;
    global StrResultPath;

    global amountImgFiles;
    global filesImages;

    %check whether all pathes are set
    imgSize = size(StrImagePath);
    resSize = size(StrResultPath);

    if ( imgSize(2) < 3 ||   resSize(2) < 3 )
        set(handles.textPathesNotSet,'Visible','on');
    else
        set(handles.textPathesNotSet,'Visible','off');
        set(handles.buttonCheckPath,'Visible','off');
        set(handles.buttonBrowseImages,'Visible','off');
        set(handles.buttonBrowseResult,'Visible','off');

        set(handles.checkboxCherryImage,'Visible','off');
        set(handles.checkboxCherrySegmentation,'Visible','off');
        
        boolCherryImage = get(handles.checkboxCherryImage,'Value');
        boolCherrySegmentation = get(handles.checkboxCherrySegmentation,'Value');
        
        if boolCherryImage == 1
            set(handles.textCherryImage,'Visible','on');
        end
        
        if boolCherrySegmentation == 1
            set(handles.textCherrySegmentation,'Visible','on');
        end
        
        set(handles.buttonStartAnalyses,'Visible','on');
        set(handles.amountFiles,'Visible','on');
        set(handles.textAmountFiles,'Visible','on');

        filesImages = dir(StrImagePath);
        amountImgFiles = size((filesImages),1) -2;

        set(handles.amountFiles,'String',num2str(amountImgFiles)); 
    end %if1
end

function amountFiles_Callback(hObject, eventdata, handles)            
% hObject    handle to amountFiles (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of amountFiles as text
%        str2double(get(hObject,'String')) returns contents of amountFiles as a double

    global amountImgFiles;

    set(handles.amountFiles,'String',num2str(amountImgFiles)); 
end

% --- Executes during object creation, after setting all properties.
function amountFiles_CreateFcn(hObject, eventdata, handles)           
% hObject    handle to amountFiles (see GCBO) 
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
    if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
        set(hObject,'BackgroundColor','white');
    end
end
% --- Executes on button press in buttonStartAnalyses.
function buttonStartAnalyses_Callback(hObject, eventdata, handles)   
% hObject    handle to buttonStartAnalyses (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

    %parameter for quality analysis
    threshold = 1000000;
    amountCellsAdaptive = 300;
    qualityMark1 = 3000;
    qualityMark2 = 9000;
    qualityMark3 = 15000;
        
    startTime = cputime;

    global resultsAnalyses; %will be an array with results
    
    global StrImagePath;
    global StrResultPath;

    global amountImgFiles;
    global filesImages;
    
    boolCherryImage = get(handles.checkboxCherryImage,'Value');
    boolCherrySegmentation = get(handles.checkboxCherrySegmentation,'Value');
    
    %start analyse
    set(handles.buttonStartAnalyses,'Visible','off');
    set(handles.textAnalyseStarted,'Visible','on');
    set(handles.textAnalyseStarted2,'Visible','on');  
    pause(0.1); %pause is necessary to alter figure before starting analyses 
    
    %save bad files
    saveBadFile = strcat(StrResultPath,'\','BadFiles.txt');
    fidBad = fopen(saveBadFile,'w');
    if fidBad == -1
        fprintf('PROBLEM: The BadFiles.txt could not be opened in %s. It will be stored in "C:\" \n',StrResultPath); 
        fidBad = fopen('c:\BadFiles.txt','w');
        if fidBad == -1
            fprintf('PROBLEM: The BadFiles.txt could not be opened at all!!\n'); 
        end
    end    
    fprintf(fidBad,'Bad files:\n');
    
    %Analyse
    boolExit = 1;
       
    bitvector = zeros(1,(amountImgFiles));  %amount files without . ..

    validFiles = [];
    
    %Analyse all file names in the folder and mark those which satisfy the
    %requirements for the analysis.

    for i=1:amountImgFiles+2
        if size(filesImages(i).name,2) > 5
            if size(filesImages(i).name,2) > 35
                if strcmp(filesImages(i).name(35),'e') == 1 || strcmp(filesImages(i).name(35),'H') == 1  %GFP & Hoechst
                    validFiles = [validFiles i];  %create list with all valid files - size previously unknown 
                elseif boolCherryImage == 1 && strcmp(filesImages(i).name(35),'C') == 1 %add Cherry if available
                    validFiles = [validFiles i];  %create list with all valid files - size previously unknown 
                else %if3
                     fprintf('WARNING: The following file name does not satisfy the requirements: %s\n', filesImages(i).name); 
                end %if3
            else %if2
                fprintf('WARNING: The following file name does not satisfy the requirements: %s\n', filesImages(i).name); 
            end %if2
        end %if1
    end %for1
    
    amountImgFilesValid = size(validFiles,2);
    
    %DEBUG check file names
    %for z = 1:amountImgFilesValid
    %    validFiles(z)
    %    filesImages(validFiles(z)).name
    %end % for1

    %Loop:
    %Search for a complete group of image files belonging together starting
    %with the GFP image. If the Hoechst or Cherry file is not found ->
    %message in bad files and in command window.
    %If the quality of one of the images is not good enought -> message in
    %bad files and in command window and no analysis
    %Analyse the image and store the result in the global variable
    %resultsAnalyses. This is an array with the results and each line
    %contains on result for either GFP or Cherry.
    %FileNr from filesImages, amount of: cells , cytoplasma, golgi, 
    %membram, manual, golgi(strong), invalid
    %Additionally, it is possible that there are no Cherry files.
    
    while boolExit == 1
        
        %GFP first, then search for Hoechst and Cherry
        HoechstFileNr = -1;  
        CherryFileNr = -1;
        
        boolExit = 1;
        j = 1;
        while boolExit == 1 && (strcmp(filesImages(validFiles(j)).name(35),'e') == 0 || bitvector(j) == 1)
            if j == amountImgFilesValid
                boolExit = 0;
            else
                j = j+1;
            end %if1
        end %while2

        if boolExit == 1
            str = filesImages(validFiles(j)).name;
            fileNameGFP = filesImages(validFiles(j)).name;
            GFPFileNr = validFiles(j);
            bitvector(j) = 1;
            for k=1:(amountImgFilesValid)
                if bitvector(k) == 0
                    if strncmp(str, filesImages(validFiles(k)).name, 34) == 1 && filesImages(validFiles(k)).name(35) == 'H'
                        bitvector(k) = 1;
                        fileGFP = strcat(StrImagePath,'\', str);
                        HoechstFileNr = validFiles(k);
                        fileNameHoechst = filesImages(validFiles(k)).name;
                        fileHoechst = strcat(StrImagePath,'\', filesImages(validFiles(k)).name);
                    elseif boolCherryImage == 1 && strncmp(str, filesImages(validFiles(k)).name, 34) == 1 && filesImages(validFiles(k)).name(35) == 'C'
                        bitvector(k) = 1;
                        CherryFileNr = validFiles(k);
                        fileNameCherry = filesImages(validFiles(k)).name;
                        fileCherry = strcat(StrImagePath,'\', filesImages(validFiles(k)).name);
                    end %if3
                end %if2
            end %for1
            
            if HoechstFileNr == -1  %check Hoechst
                fprintf('PROBLEM: A corresponding hoechst file was not found for the following GFP file %s\n', fileNameGFP); 
                fprintf(fidBad,strcat('Corresponding hoechst file missing for in:',str,'\n'));              
            elseif boolCherryImage == 1 && CherryFileNr == -1;  %check Cherry
                fprintf('PROBLEM: A corresponding cherry file was not found for the following GFP file %s\n', fileNameGFP); 
                fprintf(fidBad,strcat('Corresponding cherry file missing for in:',str,'\n'));
            else %if2
            
                %read files, check quality and start analyses
                imgGFP = imread(fileGFP);           
                imgHoechst = imread(fileHoechst);
                if boolCherryImage == 1
                    imgCherry = imread(fileCherry);
                else %if3
                    imgCherry = imgGFP;
                end %if3
                
                valueBadFileHoechst = CheckImageQuality(imgHoechst, threshold);
                valueBadFileGFP     = CheckImageQuality(imgGFP, threshold);
                valueBadFileCherry  = CheckImageQuality(imgCherry, threshold);
                
                if boolCherryImage == 1
                    fprintf('Analyse started for dataset GFP: %s Cherry: %s Hoechst: %s\n', fileNameGFP, fileNameCherry, fileNameHoechst); 
                else
                    fprintf('Analyse started for dataset GFP: %s Hoechst: %s (no Cherry file)\n', fileNameGFP, fileNameHoechst)
                end
                
                %Analyse images
                boolPause = 0;
                cellResult = analyseImages(imgGFP, imgCherry, imgHoechst, boolPause, boolCherryImage, boolCherrySegmentation);
                
                %hoechst quality
                if valueBadFileHoechst < qualityMark1 
                    badFileHoechst = 0;
                else
                    hoechstFactor = 1;
                    if(cellResult(1)< amountCellsAdaptive)
                        hoechstFactor = cellResult(1)/(-400)+1.75;
                    end %if4
                    
                    if((valueBadFileHoechst*hoechstFactor) >= qualityMark1 && (valueBadFileHoechst*hoechstFactor) < qualityMark2)
                        badFileHoechst = 1;
                    elseif((valueBadFileHoechst*hoechstFactor) >= qualityMark2 && (valueBadFileHoechst*hoechstFactor) < qualityMark3)
                        badFileHoechst = 2;    
                    elseif((valueBadFileHoechst*hoechstFactor) >= qualityMark3)
                        badFileHoechst = 3;    
                    end %if4
                end %if3
                
                %GFP quality
                if valueBadFileGFP < qualityMark1
                    badFileGFP = 0;
                else
                    GFPFactor = 1;
                    if(cellResult(1)<amountCellsAdaptive)
                        GFPFactor = cellResult(1)/(-400)+1.75;
                    end %if4
                    
                    if((valueBadFileGFP*GFPFactor) >= qualityMark1 && (valueBadFileGFP*GFPFactor) < qualityMark2)
                        badFileGFP = 1;
                    elseif((valueBadFileGFP*GFPFactor) >= qualityMark2 && (valueBadFileGFP*GFPFactor) < qualityMark3)
                        badFileGFP = 2;    
                    elseif((valueBadFileGFP*GFPFactor) >= qualityMark3)
                        badFileGFP = 3;    
                    end %if4
                end %if3

                %Cherry quality
                if valueBadFileCherry < qualityMark1
                    badFileCherry = 0;
                else
                    CherryFactor = 1;
                    if(cellResult(1)<amountCellsAdaptive)
                        CherryFactor = cellResult(8)/(-400)+1.75;
                    end %if4
                    
                    if((valueBadFileCherry*CherryFactor) >= qualityMark1 && (valueBadFileCherry*CherryFactor) < qualityMark2)
                        badFileCherry = 1;
                    elseif((valueBadFileCherry*CherryFactor) >= qualityMark2 && (valueBadFileCherry*CherryFactor) < qualityMark3)
                        badFileCherry = 2;    
                    elseif((valueBadFileCherry*CherryFactor) >= qualityMark3)
                        badFileCherry = 3;    
                    end %if4
                end %if3             
         
                if badFileHoechst == 0 || badFileGFP == 0 || badFileCherry == 0
                    if boolCherryImage == 1
                        fprintf('PROBLEM: Image quality for the following dataset was not good GFP: %s Cherry: %s Hoechst: %s\n', fileNameGFP, fileNameCherry, fileNameHoechst); 
                        fprintf(fidBad,strcat('Bad quality file found in:',fileNameHoechst,'; ',str,'; ',fileNameCherry,'\n'));
                    else
                        fprintf('PROBLEM: Image quality for the following dataset was not good GFP: %s Hoechst: %s\n', fileNameGFP, fileNameHoechst); 
                        fprintf(fidBad,strcat('Bad quality file found in:',fileNameHoechst,'; ',str,'\n'));
                    end
                end %if3
                
                resultsAnalyses = [resultsAnalyses;GFPFileNr, cellResult(1), cellResult(2), cellResult(3), cellResult(4), cellResult(5), cellResult(6), cellResult(7), badFileHoechst, badFileGFP; CherryFileNr, cellResult(8), cellResult(9), cellResult(10), cellResult(11), cellResult(12), cellResult(13), cellResult(14), badFileHoechst, badFileCherry];
       
            end %if2
        end %if1
    end %while1
    
    fclose(fidBad);   
    
    endTime = cputime;
    fprintf('*****************************\n'); 
    fprintf('**Cell analysis successful!**\n'); 
    fprintf('*****************************\n'); 
    fprintf('**Time for execution in seconds: %g\n', endTime - startTime); 
    
    %analyse finished
    set(handles.textAnalyseStarted,'Visible','off');
    set(handles.textAnalyseStarted2,'Visible','off');  
    set(handles.buttonViewResults,'Visible','on');  
end

% --- Executes on button press in buttonViewResults.
function buttonViewResults_Callback(hObject, eventdata, handles)   
% hObject    handle to buttonViewResults (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

    global resultsAnalyses; 
    
    boolCherryImage = get(handles.checkboxCherryImage,'Value');
    
    set(handles.buttonViewResults,'Visible','off'); 
    set(handles.buttonDone,'Visible','On');
    
    %here just an overview of all results is shown by displaying a sum for
    %all types of cells
    if boolCherryImage == 1
        resultsTotal = sum(resultsAnalyses);
    else
        resultsHelp = resultsAnalyses;
        resultsHelp(resultsHelp == -1) = 0;
        resultsTotal = sum(resultsHelp);
    end
    
    set(handles.totalCells,'String',num2str(resultsTotal(2))); 
    set(handles.totalCyto,'String',num2str(resultsTotal(3))); 
    set(handles.totalGolgi,'String',num2str(resultsTotal(4))); 
    set(handles.totalMem,'String',num2str(resultsTotal(5))); 
    %set(handles.totalManual,'String',num2str(resultsTotal(6))); 
    set(handles.totalUndec,'String',num2str(resultsTotal(7))); 
    set(handles.totalInvalid,'String',num2str(resultsTotal(8))); 

    set(handles.textcells,'Visible','on'); 
    set(handles.textCyto,'Visible','on'); 
    set(handles.textGolgi,'Visible','on'); 
    set(handles.textMem,'Visible','on'); 
    %set(handles.textMan,'Visible','on'); 
    set(handles.textUndec,'Visible','on'); 
    set(handles.textInval,'Visible','on'); 
    set(handles.totalCells,'Visible','on'); 
    set(handles.totalCyto,'Visible','on'); 
    set(handles.totalGolgi,'Visible','on'); 
    set(handles.totalMem,'Visible','on'); 
    %set(handles.totalManual,'Visible','on'); 
    set(handles.totalUndec,'Visible','on'); 
    set(handles.totalInvalid,'Visible','on'); 
  
end

function totalCyto_Callback(hObject, eventdata, handles)             
% hObject    handle to totalCyto (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of totalCyto as text
%        str2double(get(hObject,'String')) returns contents of totalCyto as a double
end

% --- Executes during object creation, after setting all properties.
function totalCyto_CreateFcn(hObject, eventdata, handles)             
% hObject    handle to totalCyto (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
    if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
        set(hObject,'BackgroundColor','white');
    end
end

function totalGolgi_Callback(hObject, eventdata, handles)             
% hObject    handle to totalGolgi (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of totalGolgi as text
%        str2double(get(hObject,'String')) returns contents of totalGolgi as a double
end

% --- Executes during object creation, after setting all properties.
function totalGolgi_CreateFcn(hObject, eventdata, handles)             
% hObject    handle to totalGolgi (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
    if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
        set(hObject,'BackgroundColor','white');
    end
end

function totalMem_Callback(hObject, eventdata, handles)             
% hObject    handle to totalMem (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of totalMem as text
%        str2double(get(hObject,'String')) returns contents of totalMem as a double
end

% --- Executes during object creation, after setting all properties.
function totalMem_CreateFcn(hObject, eventdata, handles)             
% hObject    handle to totalMem (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
    if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
        set(hObject,'BackgroundColor','white');
    end
end

function totalUndec_Callback(hObject, eventdata, handles)             
% hObject    handle to totalUndec (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of totalUndec as text
%        str2double(get(hObject,'String')) returns contents of totalUndec as a double
end

% --- Executes during object creation, after setting all properties.
function totalUndec_CreateFcn(hObject, eventdata, handles)             
% hObject    handle to totalUndec (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
    if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
        set(hObject,'BackgroundColor','white');
    end
end

function totalInvalid_Callback(hObject, eventdata, handles)             
% hObject    handle to totalInvalid (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of totalInvalid as text
%        str2double(get(hObject,'String')) returns contents of totalInvalid as a double
end

% --- Executes during object creation, after setting all properties.
function totalInvalid_CreateFcn(hObject, eventdata, handles)             
% hObject    handle to totalInvalid (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
    if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
        set(hObject,'BackgroundColor','white');
    end
end

function totalCells_Callback(hObject, eventdata, handles)             
% hObject    handle to totalCells (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of totalCells as text
%        str2double(get(hObject,'String')) returns contents of totalCells as a double
end

% --- Executes during object creation, after setting all properties.
function totalCells_CreateFcn(hObject, eventdata, handles)             
% hObject    handle to totalCells (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
    if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
        set(hObject,'BackgroundColor','white');
    end
end

% --- Executes on button press in buttonDone.
function buttonDone_Callback(hObject, eventdata, handles)
% hObject    handle to buttonDone (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

    %%%ADDED GLOBAL VARIABLES%%%
    global StrResultPath;  
    global filesImages;
    global resultsAnalyses;

    set(handles.buttonDone,'Visible','Off');    

    boolCherryImage = get(handles.checkboxCherryImage,'Value');

    %The results will be stored in two txt files. One for the GFP and one for
    %the Cherry results. So for each row in the result array the name will be
    %analysed and then the result will be stored in the corresponding txt file.

    amountResults = size(resultsAnalyses,1);

    saveGFPFile = strcat(StrResultPath,'\','GFPAnalyse.txt');

    fidGFP = fopen(saveGFPFile,'w');
    if fidGFP == -1
        fprintf('PROBLEM: The GFPAnalyse.txt could not be opened in %s. It will be stored in "C:\" \n',StrResultPath); 
        fidGFP = fopen('c:\GFPAnalyse.txt','w');
        if fidGFP == -1
            fprintf('PROBLEM: The GFPAnalyse.txt could not be opened at all!!\n'); 
        end
    end

    fprintf(fidGFP,'Filename;Anzahl Zellen;Anzahl Cytoplasma;Anzahl Golgi; Anzahl Golgi (strong); Anzahl Membran; Anzahl Invalid; Qualitt Hoechst; Qualitt GFP;\n');

    if boolCherryImage == 1

        saveCherryFile = strcat(StrResultPath,'\','CherryAnalyse.txt');

        fidCherry = fopen(saveCherryFile,'w');
        if fidCherry == -1
            fprintf('PROBLEM: The CherryAnalyse.txt could not be opened in %s. It will be stored in "C:\" \n',StrResultPath); 
            fidCherry = fopen('c:\CherryAnalyse.txt','w');
            if fidCherry == -1
                fprintf('PROBLEM: The CherryAnalyse.txt could not be opened at all!!\n'); 
            end
        end

        fprintf(fidCherry,'Filename;Anzahl Zellen;Anzahl Cytoplasma;Anzahl Golgi; Anzahl Golgi (strong); Anzahl Membran; Anzahl Invalid; Qualitt Hoechst; Qualitt Cherry;\n');
    end %if cherry


    for i=1:(amountResults)

        fileNumber = resultsAnalyses(i,1);

        if  fileNumber ~= -1
            if boolCherryImage == 1 && filesImages(fileNumber).name(35) == 'C'
                fprintf(fidCherry,strcat(filesImages(fileNumber).name,';%4.0f;%4.0f;%4.0f;%4.0f;%4.0f;%4.0f;%4.0f;%4.0f\n'),resultsAnalyses(i,2),resultsAnalyses(i,3),resultsAnalyses(i,4),resultsAnalyses(i,7),resultsAnalyses(i,5),resultsAnalyses(i,8),resultsAnalyses(i,9),resultsAnalyses(i,10));
            elseif filesImages(fileNumber).name(35) == 'e'
                fprintf(fidGFP,strcat(filesImages(fileNumber).name,';%4.0f;%4.0f;%4.0f;%4.0f;%4.0f;%4.0f;%4.0f;%4.0f\n'),resultsAnalyses(i,2),resultsAnalyses(i,3),resultsAnalyses(i,4),resultsAnalyses(i,7),resultsAnalyses(i,5),resultsAnalyses(i,8),resultsAnalyses(i,9),resultsAnalyses(i,10));
            end   
        end
         %fprintf(strcat(filesImages(fileNumber).name,';%4.0f;%4.0f;%4.0f;%4.0f;%4.0f;%4.0f\n'),resultsAnalyses(i,2),resultsAnalyses(i,3),resultsAnalyses(i,4),resultsAnalyses(i,5),resultsAnalyses(i,7),resultsAnalyses(i,8),resultsAnalyses(i,9),resultsAnalyses(i,10));
    end
   
    fclose(fidGFP);
    if boolCherryImage == 1
        fclose(fidCherry);
    end
    
    set(handles.buttonEnde,'Visible','On');
end
% --- Executes on button press in buttonEnde.
function buttonEnde_Callback(hObject, eventdata, handles)
% hObject    handle to buttonEnde (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

    close(handles.figure1);
end

% --- Executes on button press in checkboxCherryImage.
function checkboxCherryImage_Callback(hObject, eventdata, handles)
% hObject    handle to checkboxCherryImage (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of checkboxCherryImage
    boolCherryImage = get(handles.checkboxCherryImage,'Value');

    if boolCherryImage == 1
        set(handles.checkboxCherrySegmentation,'Visible','on');
    else
        set(handles.checkboxCherrySegmentation,'Visible','off');
        set(handles.checkboxCherrySegmentation,'value',0);
    end
end
% --- Executes on button press in checkboxCherrySegmentation.
function checkboxCherrySegmentation_Callback(hObject, eventdata, handles)
% hObject    handle to checkboxCherrySegmentation (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of checkboxCherrySegmentation
end

