#!/usr/bin/perl -w

use DBI;
use vars qw/ %opt /;
use strict vars;
use Data::Dumper;

# process the command-line args
sub init() {
        use Getopt::Std;
        my $opt_string = 'ht:r:p:s:l:a:';
        getopts("$opt_string", \%opt) or usage();
        usage() if $opt{h} or !$opt{t} or !$opt{r} or !$opt{p} or !$opt{l} or !$opt{a};
}

# print if requested or insufficent args
sub usage() {
        print STDERR << "EOF";
        
usage: $0 [-h] [-t tablename] [-r tablename] [-p peaksize] [-l xref list]
        
 -h     : this message
 -t     : 'pairedpeaks' table name
 -r	: 'readpairs' table name
 -l	: table to crossreference for known insertion sites (i.e. paperlist, masterlist, etc.)
 -p	: minimum peak size
 -s	: database hostname (optional)
 -a	: 'abinitio' table, links tables specified by -t and -l and stores output of this script

example: $0 -t pairedpeaks -r readpairs -p 20 -l paperlist 

EOF
        exit;
}


# converts an integer to binary, returns rightmost two bits
sub twobits {
	return substr(unpack("B32", pack("N", shift)), -2);
}

# converts a binary string to an integer
sub toint {
	return unpack("N", pack("B32", substr("0" x 32 . shift, -32)));
}

# average function, takes an array
sub avg {
	my $sum = 0;
	for (@_) {
		$sum += $_;
	}
	return $sum/(scalar @_);
}

# square root of the unbiased variance
sub sd {
        my @d;
        my $avg = avg(@_);
        for (@_) {
                push @d, ($_ - $avg)**2;
        }

        my $sum = 0;
        for(@d) {
                $sum += $_;
        }
        return sqrt($sum/((scalar @d) - 1));
}


# checks a pair of sets (arrayrefs) of numbers for overlap
# returns the total number of overlapping locations in both sets
sub overlap {

        my @s1 = sort {$a <=> $b} @{$_[0]};
        my @s2 = sort {$a <=> $b} @{$_[1]};

        my $s1min = $s1[0];
        my $s1max = $s1[(scalar @s1) - 1];
	my $s2min = $s2[0];
	my $s2max = $s2[(scalar @s2) - 1];

        my $s1ol = 0;
	my $s2ol = 0;

        for (@s2) {
		$s1ol ++ if ($_ >= $s1min && $_ <= $s1max);
        }
	for (@s1) {
		$s2ol ++ if ($_ >= $s2min && $_ <= $s2max);
	}

	return $s1ol+$s2ol;
}

# if the range of one set is completely contained by the range of the other set
# this function will try to 'uncover' them and salvage the detectable insertion

sub uncover {
        my @s1 = sort {$a <=> $b} @{$_[0]};
        my @s2 = sort {$a <=> $b} @{$_[1]};

        my $s1min = $s1[0];
        my $s1max = $s1[(scalar @s1) - 1];
	my $s2min = $s2[0];
	my $s2max = $s2[(scalar @s2) - 1];

	# need to control cases where one inverval completely contains the other
	# because this is often do to one or two outlier values

	while ($s1min < $s2min && $s1max > $s2max) { # while s1 contains s2 ...
		if (avg(@s1) >= avg(@s2)) { # s1 tends to be larger than s2
			shift @s1; # remove the smallest value if s1 tends to be larger
			$s1min = $s1[0];
		} else {
			pop @s1; # or remove the larger value if s1 tends to be smaller
			$s1max = $s1[(scalar @s1) - 1];
		}
	}

	while ($s2min < $s1min && $s2max > $s1max) { # while s2 contains s1 ...
		if (avg(@s2) >= avg(@s1)) {
			shift @s2; # remove the smallest value if s2 tends to be larger
			$s2min = $s2[0];
		} else {
			pop @s2; # or remove the larger value if s2 tends to be smaller
			$s2max = $s2[(scalar @s2) - 1];
		}
	}

	return (\@s1,\@s2);
}


# finds the minimum distance between two non-overlapping sets
sub minsetdist {
	my @s1 = sort {$a <=> $b} split(/,/,$_[0]);
	my @s2 = sort {$a <=> $b} split(/,/,$_[1]);
	if (avg(@s1) < avg(@s2)) {
		return $s2[0] - $s1[(scalar @s1)-1];
	} else {
		return $s1[0] - $s2[(scalar @s2)-1];
	}
}

# returns an array of (start,stop) corresponding to the l1/genome junction
# i.e. the endpoints for the distance returned by minsetdist
sub inspoint {
	my @s1 = sort {$a <=> $b} split(/,/,$_[0]);
	my @s2 = sort {$a <=> $b} split(/,/,$_[1]);
	if (avg(@s1) < avg(@s2)) {
		return ($s1[(scalar @s1)-1], $s2[0]);
	} else {
		return ($s2[(scalar @s2)-1], $s1[0]);
	}
}

# finds the width of a pair end
sub endwidth {
	my @reads = sort {$a <=> $b} @_;
	return $reads[(scalar @reads)-1] - $reads[0];
}

# if a few locs are outliers (i.e. no more than 25% of the whole peak)
# remove them and return the 'cleaned' array. Done by considering interquartile
# ranges. 

sub cleanlocs {
        my @locs = sort {$a <=> $b} @_;
        my $Q1pos = int(0.25*(scalar @locs));
        my $Q3pos = int(0.75*(scalar @locs));
        my $IQR = $locs[$Q3pos] - $locs[$Q1pos];

	# adjusting these lower yields higher stringency and vice versa

        my $low = $locs[$Q1pos] - 1.5*$IQR;
        my $high = $locs[$Q3pos] + 1.5*$IQR;

	#print "low: $low\thigh: $high\n";

        my @cleaned;
        for (@_) {
                push @cleaned, $_ if ($_ > $low && $_ < $high);
        }

        my $pctremoved = ((scalar @_ - scalar @cleaned)/(scalar @_));

	# if more than 20% of the locations were outliers, this set was not salvageable
	# can be adjusted down for greater stringency
        if ($pctremoved < 0.25) {
		print "cleaning excluded $pctremoved * 100% of reads\n";
                return @cleaned;
        } else {
		print "***too many outliers! (exc: $pctremoved)\n";
                return @_;
        }
}

# function for anonymous hash composer to init %pairs
sub hashinit {
	return ('3p' => 0, 'mindist' => 0, 'tight' => 0, 'size' => 0, 'inv' => 0, 'length' => 0, 'insstart' => 0, 'insstop' => 0, 'orient' => 0, 'overlap' => 0);
}

# cross reference with table given by $opt{l} (-l flag)
sub xref {
	my ($chr,$start,$stop,$server,$table) = @_;
	$chr =~ s/chr//;
	my $database = 'l1hsgeno';                                                                                                                    
	my $user = 'l1';                                                                                                                              
	my $passwd = 'l1';                                                                                                                            
                                                                                                                                              
	my $dbh = DBI->connect("dbi:mysql:database=$database;host=$server;port=8081;mysql_socket=/gpfs/fs0/u/ewingad/mysql/mysql.sock", $user, $passwd);
	my $approxins = int((($start + $stop)/2));

	my $query = "SELECT `id`i,`v_pcr`,`family` FROM `$table` WHERE `chr`='$chr' and $approxins between loc-600 and loc+600";
	my $sth = $dbh->prepare($query);
	$sth->execute();
	my ($id,$v_pcr,$family);
	$sth->bind_columns(\$id,\$v_pcr,\$family);
	if ($sth->fetch()) {
		return ($id,$v_pcr,$family);
	} else {
		return (0,0,0);
	}
}

# make a table to store data
sub maketable {
	my ($table,$server) = @_;
        my $database = 'l1hsgeno';
        my $user = 'l1';
        my $passwd = 'l1';

        my $dbh = DBI->connect("dbi:mysql:database=$database;host=$server;port=8081;mysql_socket=/gpfs/fs0/u/ewingad/mysql/mysql.sock", $user, $passwd);


        my $maketable_query = "
        CREATE TABLE IF NOT EXISTS `$table` (
        `id` INT NOT NULL AUTO_INCREMENT ,
        `chr` CHAR ( 2 )  NOT NULL ,
        `start` INT NOT NULL ,
        `stop` INT NOT NULL ,
	`strand` CHAR ( 1 ) NOT NULL ,
        `inv` BOOL DEFAULT 0 ,
        `length` INT NOT NULL ,
	`genomes` INT NOT NULL ,
	`genomelist` TEXT NOT NULL ,
        `peakid` INT NOT NULL ,
        `insid` INT NOT NULL , 
        PRIMARY KEY ( `id` )
        ) TYPE = MYISAM ;
        ";
        my $sth = $dbh->prepare($maketable_query);
        $sth->execute();

        my $index_query = "ALTER TABLE `$table` ADD INDEX ( `peakid` )";
        $sth = $dbh->prepare($index_query);
        $sth->execute();

        $index_query = "ALTER TABLE `$table` ADD INDEX ( `insid` )";
        $sth = $dbh->prepare($index_query);
        $sth->execute();
}

# return the number of LINEs that overlap and terminate within an interval
sub findline {
	my ($dbh,$chr,$windowstart,$windowstop) = @_;
	my $table = "LINE";
	my $query = "SELECT * FROM `$table` WHERE chr='$chr' AND ";
	$query .= "(family='L1HS' OR family like 'L1PA%') AND (";
	$query .= "(start between $windowstart and $windowstop) OR ";
	$query .= "(stop between $windowstart and $windowstop)";
	$query .= ")";
	my $sth = $dbh->prepare($query);
	$sth->execute();
	
	return $sth->rows;
}

sub uniqlist {
	my %seen;
	for (@_) { $seen{$_} ++; }
	return keys %seen;
}

init();
maketable($opt{a},$opt{s});

my $database = 'l1hsgeno';
my $l1database = 'hg18';
my $server = 'localhost';
$server = $opt{s} if ($opt{s});
my $user = 'l1';
my $passwd = 'l1';

my $dbh = DBI->connect("dbi:mysql:database=$database;host=$server;port=8081;mysql_socket=/gpfs/fs0/u/ewingad/mysql/mysql.sock", $user, $passwd);
my $l1dbh = DBI->connect("dbi:mysql:database=$l1database;host=$server;port=8081;mysql_socket=/gpfs/fs0/u/ewingad/mysql/mysql.sock", $user, $passwd);

my $minpeak = $opt{p};
my $t_table = $opt{t};
my $r_table = $opt{r};

my $maxfp = 600; # max size for peak 'footprint' 

# read peaks (collections of paired reads)
my $t_query = "SELECT $t_table.id,$t_table.chr,minloc,maxloc,mintnloc,maxtnloc,family,v_pcr FROM $t_table left join paperlist on $t_table.insid=paperlist.id WHERE n >= $minpeak and maxtnloc > 5900";
my $t_sth = $dbh->prepare($t_query);
$t_sth->execute();
my ($t_peakid,$t_chr,$t_minloc,$t_maxloc,$t_mintnloc,$t_maxtnloc,$t_family,$t_pcr);
$t_sth->bind_columns(\$t_peakid,\$t_chr,\$t_minloc,\$t_maxloc,\$t_mintnloc,\$t_maxtnloc,\$t_family,\$t_pcr);

while($t_sth->fetch()) {
	my $r_query = "SELECT id,loc,strand,tnloc,tnstrand,genome FROM $r_table WHERE peakid=$t_peakid";
	my $r_sth = $dbh->prepare($r_query);
	$r_sth->execute();
	my ($id,$loc,$strand,$tnloc,$tnstrand,$genome);
	$r_sth->bind_columns(\$id,\$loc,\$strand,\$tnloc,\$tnstrand,\$genome);

	# loop through each read pair

	my @p; # holds read pairs in hashes
	my $n = 0;
	while($r_sth->fetch()) {
		$p[$n]{'id'} = $id;
		$p[$n]{'loc'} = $loc;
		$p[$n]{'strand'} = $strand;
		$p[$n]{'tnloc'} = $tnloc;
		$p[$n]{'tnstrand'} = $tnstrand;
		$p[$n]{'genome'} = $genome;
		$n ++;
	}

	#
	# Split the read pairs into sets based on their orientation
	# Binary encoding:
	# "+-" --> genome is + tn is -
	# 00 = -- = 0
	# 01 = -+ = 1
	# 10 = +- = 2
	# 11 = ++ = 3
	#

	my $ostring = $strand . $tnstrand;
	$ostring =~ tr/-+/01/; 
	my @sets; # stores sets of paired reads based on orientations
	my %pairs; # possible pairs of orientation sets, hash keeps it unique

	for (0 .. (scalar @p)-1) {
		my $ostring = $p[$_]{'strand'} . $p[$_]{'tnstrand'};
		$ostring =~ tr/-+/01/;
		push @{$sets[(toint $ostring)]}, $_;
	}

	# for each set there will be one or two reasonable partners
	# - 01 cannot be paired with 00 and 11 can't be paired with 10 (genomic strands must differ)
	# - partner can't be null
	# - genome location 'footprints' cannot overlap	

	# loop over sets 0 and 1, once we hit set 2 we know there is no valid partner since
	# sets 2 and 3 have the same genomic strand (10 and 11 = +- and ++)
	for (0 .. 3) { 
		if ($sets[$_]) {
			my $fb = substr(twobits($_),0,1); # first bit
			my $sb = substr(twobits($_),1,1); # second bit
			# so possible partners are ~$fb . "0" and ~$fb . "1"
			$fb =~ tr/01/10/; # "complement"
			my $p1 = toint($fb . "0");
			my $p2 = toint($fb . "1");
			
			if (!$sets[$p1] && !$sets[$p2]) {
			} elsif ($sets[$p1] && !$sets[$p2]) {
				my @pair = sort {$a <=> $b} ($_, $p1);
				$pairs{join(',',@pair)} = {&hashinit};
			} elsif (!$sets[$p1] && $sets[$p2]) {
				my @pair = sort {$a <=> $b} ($_, $p2);
				$pairs{join(',',@pair)} = {&hashinit};
			} else {
				my @pair = sort {$a <=> $b} ($_, $p1);
				$pairs{join(',',@pair)} = {&hashinit};
				@pair = sort {$a <=> $b} ($_, $p2);
				$pairs{join(',',@pair)} = {&hashinit};
			}
		}
	}

	# consider all valid set pairs and eliminate those that overlap
	# for those that aren't eliminated, find the min. distance between the pair
	# also:
	# consider all set pairs that are still valid and identify those which have at least one 3' set
	# two 3' sets is valid because the insertion may be very short (< 100 bp), but zero 3' sets is
	# obviously not valid.
	for(keys %pairs) {
		my ($s1n, $s2n) = split(/,/,$_);
		my @s1 = @{$sets[$s1n]};
		my @s2 = @{$sets[$s2n]};
		my @s1genome;
		my @s2genome;
		my @s1tn;
		my @s2tn;
	
		# get genomic locations for all pairs in the two sets
		for (@s1) {
			push @s1genome, $p[$_]{'loc'};
			push @s1tn, $p[$_]{'tnloc'};
		}
		for (@s2) {
			push @s2genome, $p[$_]{'loc'};
			push @s2tn, $p[$_]{'tnloc'};
		}

		# remove outliers
		@s1tn = cleanlocs(@s1tn);
		@s2tn = cleanlocs(@s2tn);
		@s1genome = cleanlocs(@s1genome);
		@s2genome = cleanlocs(@s2genome);

		# try to rectify situations where one set of genomic coords
		# completely overlaps the other
		my ($a1, $a2) = uncover(\@s1genome,\@s2genome);
		@s1genome = @{$a1};
		@s2genome = @{$a2};

		# Need to remove genome/tn pairs that no longer exist because one end has been removed

		my @s1genomenew;
		my @s1tnnew;
		my %s1genomelookup;
		my %s1tnlookup;

		for(@s1tn) { $s1tnlookup{$_} = 1; }
		for(@s1genome) { $s1genomelookup{$_} = 1; }

		for(@s1) {
			my $idx = $_;
			for(@s1genome) {
				# locs are represented in both tn and genome lists, keep tnloc
				if ($p[$idx]{'loc'} == $_  && $s1tnlookup{$p[$idx]{'tnloc'}}) {
					push @s1tnnew, $p[$idx]{'tnloc'};
				}
			}
			for(@s1tn) {
				# same as above, keep genome loc
				if ($p[$idx]{'tnloc'} == $_  && $s1genomelookup{$p[$idx]{'loc'}}) {
					push @s1genomenew, $p[$idx]{'loc'};
				}
			}
		}
		
		my @s2genomenew;
		my @s2tnnew;
		my %s2genomelookup;
		my %s2tnlookup;

		for(@s2tn) { $s2tnlookup{$_} = 1; }
		for(@s2genome) { $s2genomelookup{$_} = 1; }

		for(@s2) {
			my $idx = $_;
			for(@s2genome) {
				# locs are represented in both tn and genome lists, keep tnloc
				if ($p[$idx]{'loc'} == $_  && $s2tnlookup{$p[$idx]{'tnloc'}}) {
					push @s2tnnew, $p[$idx]{'tnloc'};
				}
			}
			for(@s2tn) {
				# same as above, keep genome loc
				if ($p[$idx]{'tnloc'} == $_  && $s2genomelookup{$p[$idx]{'loc'}}) {
					push @s2genomenew, $p[$idx]{'loc'};
				}
			}
		}

		@s1genome = uniqlist(@s1genomenew);
		@s2genome = uniqlist(@s2genomenew);
		@s1tn = uniqlist(@s1tnnew);
		@s2tn = uniqlist(@s2tnnew);

		# build a list of the remaining genomes, if locations were removed, the corresponding
		# genome(s) won't be in this list. Put another way, genomes corresponding to removed
		# locations will not be added to the list. It doesn't matter than duplicate locations
		# have been removed as genomes are included based on corresponding locations, not tuples. 

		my @genomelist;

		for (@s1,@s2) {
			my $idx = $_;
			foreach (@s1genome,@s2genome) {
				push @genomelist, $p[$idx]{'genome'} if ($p[$idx]{'loc'} == $_);
			}
			#foreach (@s1tn,@s2tn) {
			#	push @genomelist, $p[$idx]{'genome'} if ($p[$idx]{'tnloc'} == $_);
			#}
		}


		# needs to remove duplicates from the genome list:

		@genomelist = uniqlist(@genomelist);

		print "genomelist: " . join(',',@genomelist) . "\n";
		print "s1tn: " . join(',',@s1tn) . "\n";
		print "s2tn: " . join(',',@s2tn) . "\n";
		print "s1genome: " . join(',',@s1genome) . "\n";
		print "s2genome: " . join(',',@s2genome) . "\n";
	
		# do the locations of the sets overlap w.r.t. genomic coords?
		my $ol = overlap(\@s1genome,\@s2genome) if ((scalar @s1genome) >= 2 && (scalar @s2genome) >= 2);

		# make sure all coord lists have something in them (2 is arbitrary)
		if ((scalar @s1tn) < 2 || (scalar @s2tn) < 2 || (scalar @s1genome) < 2 || (scalar @s2genome) < 2) {
			print "deleting (to small!) \n";
			print Dumper $pairs{$_};
			print "\n";
			delete $pairs{$_};
		} elsif ($ol > 0) { # overlap cutoff, could adjust this later
			print "deleting (overlap = $ol): \n";
			print Dumper $pairs{$_};
			print "\n";
			delete $pairs{$_};
		} else {
			# collecting various stats
			$pairs{$_}{'mindist'} = minsetdist(join(',',@s1genome), join(',',@s2genome));
			($pairs{$_}{'insstart'},$pairs{$_}{'insstop'}) = inspoint(join(',',@s1genome), join(',',@s2genome));
			# number of genomes, list of genomes...
			$pairs{$_}{'genomes'} = scalar @genomelist;
			$pairs{$_}{'genomelist'} = join(',',@genomelist);
			# maximum and minimum positions of peaks in transposons
			my $p1tnmax = (sort {$a <=> $b} @s1tn)[(scalar @s1tn)-1]; 
			my $p2tnmax = (sort {$a <=> $b} @s2tn)[(scalar @s2tn)-1];
			my $p1tnmin = (sort {$a <=> $b} @s1tn)[0];
			my $p2tnmin = (sort {$a <=> $b} @s2tn)[0];

			# detect inversions (same second bit in the orientation at both tn ends)
			my $sb1 = substr(twobits($s1n),1,1);
			my $sb2 = substr(twobits($s2n),1,1);
			$pairs{$_}{'inv'} = 1 if ($sb1 == $sb2);

			# figure out the orientation of the insertion, it's - (0) by default so just
			# change it if it's + (3p end is genomically downstream of 5p end)
			if ( ((avg(@s1genome) < avg(@s2genome)) && (avg(@s1tn) < avg(@s2tn))) ||
			     ((avg(@s1genome) > avg(@s2genome)) && (avg(@s1tn) > avg(@s2tn))) ) {
				$pairs{$_}{'orient'} = 1;
			}

			# approx length of the insertion, 6019 = length of L1RP reference elt.
			if ($p1tnmin < $p2tnmin) {
				$pairs{$_}{'length'} = 6019 - $p1tnmin;
			} else {
				$pairs{$_}{'length'} = 6019 - $p2tnmin;
			}

			# figure out if one or both ends are possible 3' ends
			if (($p1tnmax > 5900 && endwidth(@s1tn) <= $maxfp) || ($p2tnmax > 5900 && endwidth(@s2tn) <= $maxfp)) {
				$pairs{$_}{'3p'} = 1;
			} else {
				$pairs{$_}{'3p'} = 0;
			}

			# check if end 'footprints' are <= max bp
			if (endwidth(@s1tn) <= $maxfp && endwidth(@s2tn) <= $maxfp && endwidth(@s1genome) <= $maxfp & endwidth(@s2genome) <= $maxfp) {
				$pairs{$_}{'tight'} = 1;
			} else {
				$pairs{$_}{'tight'} = 0;
			}
			
			# check end sizes
			if ((scalar @{$sets[$s1n]}) >= 3 && (scalar @{$sets[$s2n]}) >= 3) {
				$pairs{$_}{'size'} = 1;
			} else {
				$pairs{$_}{'size'} = 0;
			}


			# both ends beed to be bigger than 3 reads and one end needs to be a 3' end and the chosen pair
			# needs to comprise > 33% of the whole peak (set of sets) - these are arbitrary/empirically determined
			# and maybe should be changed
			# also, both ends need to have 'tight' widths (footprint <= 300bp)
			# also, minimum distance btwn peaks must be < 500 bp

			if (defined $pairs{$_}) {
				if ($pairs{$_}{'3p'} == 0 || $pairs{$_}{'tight'} == 0 || $pairs{$_}{'size'} == 0 || $pairs{$_}{'mindist'} > 100) {
					print "deleting :\n";
					print Dumper $pairs{$_};
					print "\n";
					delete $pairs{$_};
				} 
			}

		}
	}

	# output
	for (keys %pairs) {
		my ($s1n, $s2n) = split(/,/,$_);
		my ($insid,$verify,$family) = xref($t_chr,$pairs{$_}{'insstart'},$pairs{$_}{'insstop'},$server,$opt{l});

		if (!$insid) {
			print "NO INSID\n";
		}

		print "$t_peakid\tchr$t_chr:$t_minloc-$t_maxloc\tpairs: " . (scalar (keys %pairs));
		print "\t" . $_ . "\t" . $pairs{$_}{'3p'} . $pairs{$_}{'tight'} . $pairs{$_}{'size'} . "\t" . $pairs{$_}{'overlap'} . "\t" . $pairs{$_}{'mindist'} . "\t";
		print $pairs{$_}{'insstart'} . "\t" . $pairs{$_}{'insstop'} . "\t" . $pairs{$_}{'inv'} . "\t" . $pairs{$_}{'length'};
		print "\t" . $pairs{$_}{'orient'};
		print "\t$insid\t$verify\t$family\n";

		my ($start,$stop,$strand,$inv,$length,$tname,$genomes,$genomelist) = ($pairs{$_}{'insstart'},$pairs{$_}{'insstop'},$pairs{$_}{'orient'},$pairs{$_}{'inv'},$pairs{$_}{'length'},$opt{a},$pairs{$_}{'genomes'},$pairs{$_}{'genomelist'});
		$strand =~ tr/01/-+/;

		print "searching against known LINEs... ";
		if (findline($l1dbh,$t_chr,$start,$stop) == 0 && $length > 150) {
			print "\nno LINE found, inserting...\n";
			my $insquery = "INSERT INTO `$tname` (`chr`,`start`,`stop`,`strand`,`inv`,`length`,`genomes`,`genomelist`,`peakid`,`insid`) VALUES (\'$t_chr\',\'$start\',\'$stop\',\'$strand\',\'$inv\',\'$length\',\'$genomes\',\'$genomelist\',\'$t_peakid\',\'$insid\')";
			print "$insquery\n\n";
			my $inssth = $dbh->prepare($insquery);
			$inssth->execute();
		} else {
			print "LINE found.\n";
		}
	}
}

