/*
 * Copyright (c) 2010 The Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package org.broadinstitute.sting.utils.help;

import org.broadinstitute.sting.utils.classloader.JVMUtils;
import org.broadinstitute.sting.utils.StingException;
import org.broadinstitute.sting.commandline.CommandLineProgram;

import java.util.List;
import java.util.Collections;
import java.io.IOException;

/**
 * Contains details additional details that the program can
 * supply about itself.
 *
 * @author hanna
 * @version 0.1
 */

public class ApplicationDetails {
    /**
     * Retrieve key information about the application (name, who to contact for support, etc.).
     */
    final List<String> applicationHeader;

    /**
     * Extract details covering exactly how to run this executable.
     */
    final String runningInstructions;

    /**
     * Additional help particular to this command-line application.
     */
    final String additionalHelp;

    public ApplicationDetails( List<String> applicationHeader, String runningInstructions ) {
        this(applicationHeader,runningInstructions,null);    
    }

    public ApplicationDetails( List<String> applicationHeader, String runningInstructions, String additionalHelp ) {
        this.applicationHeader = applicationHeader;
        this.runningInstructions = runningInstructions;
        this.additionalHelp = additionalHelp;
    }

    public static List<String> createDefaultHeader(Class<? extends CommandLineProgram> application) {
        return Collections.singletonList("Program Name: " + application.getName());
    }

    public static String createDefaultRunningInstructions(Class<? extends CommandLineProgram> application) {
        // Default implementation to find a command line that makes sense.
        // If the user is running from a jar, return '-jar <jarname>'; otherwise
        // return the full class name.
        String runningInstructions = null;
        try {
            runningInstructions = JVMUtils.getLocationFor( application ).getName();
        }
        catch( IOException ex ) {
            throw new StingException("Unable to determine running instructions", ex);
        }

        if( runningInstructions.endsWith(".jar") )
            runningInstructions = String.format("-jar %s", runningInstructions);
        else
            runningInstructions = application.getName();

        return runningInstructions;
    }
}
