/*
 * Copyright (c) 2010 The Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package org.broadinstitute.sting.utils.fasta;

import net.sf.picard.PicardException;
import net.sf.picard.io.IoUtil;

import java.util.Scanner;
import java.util.Map;
import java.util.LinkedHashMap;
import java.util.Iterator;
import java.util.regex.MatchResult;
import java.io.File;
import java.io.FileNotFoundException;

/**
 * Reads a fasta index file (.fai), as generated by `samtools faidx`.
 */
public class FastaSequenceIndex implements Iterable {
    /**
     * Store the entries.  Use a LinkedHashMap for consistent iteration in insertion order.
     */
    private final Map<String,FastaSequenceIndexEntry> sequenceEntries = new LinkedHashMap<String,FastaSequenceIndexEntry>();

    /**
     * Build a sequence index from the specified file.
     * @param indexFile File to open.
     * @throws FileNotFoundException if the index file cannot be found.
     */
    protected FastaSequenceIndex( File indexFile ) throws FileNotFoundException {
        if(!indexFile.exists())
            throw new FileNotFoundException(String.format("Fasta index file is missing: %s",indexFile.getAbsolutePath()));

        IoUtil.assertFileIsReadable(indexFile);
        parseIndexFile(indexFile);
    }

     /**
     * Build an empty sequence index. Entries can be added later.
     */
    protected FastaSequenceIndex() {

    }

    /**
     * Parse the contents of an index file, caching the results internally.
     * @param indexFile File to parse.
     * @throws FileNotFoundException Thrown if file could not be opened.
     */
    private void parseIndexFile(File indexFile) throws FileNotFoundException {
        Scanner scanner = new Scanner(indexFile);

        while( scanner.hasNext() ) {
            // Tokenize and validate the index line.                     
            String result = scanner.findInLine("(.+)\\t+(\\d+)\\s+(\\d+)\\s+(\\d+)\\s+(\\d+)");
            if( result == null )
                throw new PicardException("Found invalid line in index file:" + scanner.nextLine());
            MatchResult tokens = scanner.match();
            if( tokens.groupCount() != 5 )
                throw new PicardException("Found invalid line in index file:" + scanner.nextLine());

            // Skip past the line separator
            scanner.nextLine();

            // Parse the index line.
            String contig = tokens.group(1);
            long size = Long.valueOf(tokens.group(2));
            long location = Long.valueOf(tokens.group(3));
            int basesPerLine = Integer.valueOf(tokens.group(4));
            int bytesPerLine = Integer.valueOf(tokens.group(5));

            // Build sequence structure
            sequenceEntries.put( contig,new FastaSequenceIndexEntry(contig,location,size,basesPerLine,bytesPerLine) );
        }
    }

    /**
     * Does the given contig name have a corresponding entry?
     * @param contigName The contig name for which to search.
     * @return True if contig name is present; false otherwise.
     */
    public boolean hasIndexEntry( String contigName ) {
        return sequenceEntries.containsKey(contigName);
    }

    /**
     * Retrieve the index entry associated with the given contig.
     * @param contigName Name of the contig for which to search.
     * @return Index entry associated with the given contig.
     * @throws PicardException if the associated index entry can't be found.
     */
    public FastaSequenceIndexEntry getIndexEntry( String contigName ) {
        if( !hasIndexEntry(contigName) )
            throw new PicardException("Unable to find entry for contig: " + contigName);

        return sequenceEntries.get(contigName);
    }

    /**
     * Creates an iterator which can iterate through all entries in a fasta index.
     * @return iterator over all fasta index entries.
     */
    public Iterator<FastaSequenceIndexEntry> iterator() {
        return sequenceEntries.values().iterator();
    }

    /**
     * Returns the number of elements in the index.
     * @return Number of elements in the index.
     */
    public int size() {
        return sequenceEntries.size();
    }

    /**
     * Adds entry to index. Used by Fai file generator to create index entry on the fly.
     * @param contig The name of the contig
     * @param location Byte-referenced location of contig in file
     * @param size Number of bases in contig
     * @param basesPerLine Number of bases in each line. Must be uniform.
     * @param bytesPerLine Number of bytes in each line. Must be uniform.
     */
    public void addIndexEntry(String contig, long location, long size, int basesPerLine, int bytesPerLine) {
        sequenceEntries.put( contig,new FastaSequenceIndexEntry(contig,location,size,basesPerLine,bytesPerLine) );
    }

    /**
     * Compare two FastaSequenceIndex objects. Built for use in testing. No hash function has been created.
     * @param other Another FastaSequenceIndex to compare
     * @return True if index has the same entries as other instance, in the same order
     */
    public boolean equals(FastaSequenceIndex other) {
        Iterator<FastaSequenceIndexEntry> iter = this.iterator();
        Iterator<FastaSequenceIndexEntry> otherIter = other.iterator();
        while (iter.hasNext()) {
            if (!otherIter.hasNext())
                return false;
            if (!iter.next().equals(otherIter.next()))
                return false;
        }
        return true;
    }
}

/**
 * Hold an individual entry in a fasta sequence index file.
 */
class FastaSequenceIndexEntry {
    private final String contig;
    private final long location;
    private final long size;
    private final int basesPerLine;
    private final int bytesPerLine;

    /**
     * Create a new entry with the given parameters.
     * @param contig Contig this entry represents.
     * @param location Location (byte coordinate) in the fasta file.
     * @param size The number of bases in the contig.
     * @param basesPerLine How many bases are on each line.
     * @param bytesPerLine How many bytes are on each line (includes newline characters).
     */
    public FastaSequenceIndexEntry( String contig,
                                    long location,
                                    long size,
                                    int basesPerLine,
                                    int bytesPerLine ) {
        this.contig = contig;
        this.location = location;
        this.size = size;
        this.basesPerLine = basesPerLine;
        this.bytesPerLine = bytesPerLine;
    }

    /**
     * Gets the contig associated with this entry.
     * @return String representation of the contig.
     */
    public String getContig() {
        return contig;
    }

    /**
     * Gets the location of this contig within the fasta.
     * @return seek position within the fasta.
     */
    public long getLocation() {
        return location;
    }

    /**
     * Gets the size, in bytes, of the data in the contig.
     * @return size of the contig bases in bytes.
     */
    public long getSize() {
        return size;
    }

    /**
     * Gets the number of bases in a given line.
     * @return Number of bases in the fasta line.
     */
    public int getBasesPerLine() {
        return basesPerLine;
    }

    /**
     * How many bytes (bases + whitespace) are consumed by the
     * given line?
     * @return Number of bytes in a line.
     */
    public int getBytesPerLine() {
        return bytesPerLine;
    }

    /**
     * For debugging.  Emit the contents of each contig line.
     * @return A string representation of the contig line.
     */
    public String toString() {
        return String.format("contig %s; location %d; size %d; basesPerLine %d; bytesPerLine %d", contig,
                                                                                                  location,
                                                                                                  size,
                                                                                                  basesPerLine,
                                                                                                  bytesPerLine );
    }

    /**
     * Print string in format of fai file line
     * @return Contig as one line in a fai file
     */
    public String toIndexFileLine() {
        return String.format("%s\t%d\t%d\t%d\t%d", contig, size, location, basesPerLine, bytesPerLine);
    }

    /**
     * Compare entry to another instance
     * @param other another FastaSequenceIndexEntry
     * @return True if each has the same name, location, size, basesPerLine and bytesPerLine
     */
    public boolean equals(FastaSequenceIndexEntry other) {
        return (contig.equals(other.contig) && size == other.size && location == other.location
        && basesPerLine == other.basesPerLine && bytesPerLine == other.bytesPerLine);
    }
}
