package org.broadinstitute.sting.gatk.walkers.concordance;

import org.broad.tribble.vcf.VCFGenotypeRecord;
import org.broad.tribble.vcf.VCFInfoHeaderLine;
import org.broad.tribble.vcf.VCFRecord;
import org.broadinstitute.sting.gatk.contexts.ReferenceContext;
import org.broadinstitute.sting.utils.StingException;

import java.util.*;

/**
 * Split up two call sets into their various "Venn diagram" sets
 */
public class SimpleVenn implements ConcordanceType {

    private String sample1, sample2;

    public SimpleVenn() {}

    public void initialize(Map<String, String> args, Set<String> samples) {
        if ( samples.size() != 2 )
            throw new StingException("SimpleVenn concordance test cannot handle anything other than 2 VCF records");        

        Iterator<String> iter = samples.iterator();
        sample1 = iter.next();
        sample2 = iter.next();
    }

    public String computeConcordance(Map<String, VCFGenotypeRecord> samplesToRecords, ReferenceContext ref ) {

        VCFGenotypeRecord call1 = samplesToRecords.get(sample1);
        if ( call1 != null && call1.isNoCall() )
            call1 = null;
        VCFGenotypeRecord call2 = samplesToRecords.get(sample2);
        if ( call2 != null && call2.isNoCall() )
            call2 = null;

        if ( call1 == null && call2 == null )
            return null;

        // set 1 only
        if ( call2 == null )
            return sample1 + "_only";

        // set 2 only
        else if ( call1 == null )
            return sample2 + "_only";

        // at this point we know that neither is null, so now we need to test for alternate allele concordance
        VCFRecord callV1 = call1.getRecord();
        VCFRecord callV2 = call2.getRecord();

        // we can't really deal with multi-allelic variants
        if ( callV1.isBiallelic() && callV2.isBiallelic() ) {
            // intersection (concordant)
            if ( callV1.getAlternativeBaseForSNP() == callV2.getAlternativeBaseForSNP() )
                return "concordant";
            // intersection (discordant)
            else
                return "discordant";
        }

        return "concordant";
    }

    public String getInfoName() { return "Venn"; }
    public VCFInfoHeaderLine getInfoDescription() { return new VCFInfoHeaderLine(getInfoName(), 1, VCFInfoHeaderLine.INFO_TYPE.String, "2-way Venn split"); }
}
