/*
 * Copyright (c) 2010 The Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package org.broadinstitute.sting.alignment;

import org.broadinstitute.sting.gatk.refdata.ReadMetaDataTracker;
import org.broadinstitute.sting.commandline.Argument;
import org.broadinstitute.sting.gatk.walkers.ReadWalker;
import org.broadinstitute.sting.gatk.walkers.WalkerName;
import org.broadinstitute.sting.gatk.contexts.ReferenceContext;
import org.broadinstitute.sting.alignment.bwa.c.BWACAligner;
import org.broadinstitute.sting.alignment.bwa.BWAConfiguration;
import org.broadinstitute.sting.alignment.bwa.BWTFiles;
import net.sf.samtools.*;
import net.sf.picard.reference.ReferenceSequenceFileFactory;

import java.io.File;

/**
 * Aligns reads to a given reference using Heng Li's BWA aligner, presenting the resulting alignments in SAM or BAM format.
 * Mimics the steps 'bwa aln' followed by 'bwa samse' using the BWA/C implementation.
 *
 * @author mhanna
 * @version 0.1
 */
@WalkerName("Align")
public class AlignmentWalker extends ReadWalker<Integer,Integer> {
    @Argument(fullName="BWTPrefix",shortName="BWT",doc="Index files generated by bwa index -d bwtsw",required=false)
    private String prefix = null;

    @Argument(fullName = "outputBam", shortName = "ob", doc = "Write output to this BAM filename instead of STDOUT", required = false)
    private String outputBamFile = null;

    @Argument(fullName = "bam_compression", shortName = "compress", doc = "Compression level to use for writing BAM files", required = false)
    private Integer bamCompression = 5;

    /**
     * The actual aligner.
     */
    private BWACAligner aligner = null;

    /**
     * Target for reads to output.
     */
    private SAMFileWriter outputBam = null;

    /**
     * New header to use, if desired.
     */
    private SAMFileHeader header;

    /**
     * Create an aligner object.  The aligner object will load and hold the BWT until close() is called.
     */    
    @Override
    public void initialize() {
        if(prefix == null)
            prefix = getToolkit().getArguments().referenceFile.getAbsolutePath();        
        BWTFiles bwtFiles = new BWTFiles(prefix);
        BWAConfiguration configuration = new BWAConfiguration();
        aligner = new BWACAligner(bwtFiles,configuration);

        // Take the header of the SAM file, tweak it by adding in the reference dictionary and specifying that the target file is unsorted.
        header = getToolkit().getSAMFileHeader().clone();
        SAMSequenceDictionary referenceDictionary =
                ReferenceSequenceFileFactory.getReferenceSequenceFile(getToolkit().getArguments().referenceFile).getSequenceDictionary();
        header.setSequenceDictionary(referenceDictionary);
        header.setSortOrder(SAMFileHeader.SortOrder.unsorted);

        if (outputBamFile != null) {
            SAMFileWriterFactory factory = new SAMFileWriterFactory();
            outputBam = factory.makeBAMWriter(header, false, new File(outputBamFile), bamCompression);
        }
    }

    /**
     * Aligns a read to the given reference.
     * @param ref Reference over the read.  Read will most likely be unmapped, so ref will be null.
     * @param read Read to align.
     * @return Number of alignments found for this read.
     */
    @Override
    public Integer map(ReferenceContext ref, SAMRecord read, ReadMetaDataTracker metaDataTracker) {
        SAMRecord alignedRead = aligner.align(read,header);
        if (outputBam != null) {
            outputBam.addAlignment(alignedRead);
        } else {
            out.println(alignedRead.format());
        }
        return 1;
    }

    /**
     * Initial value for reduce.  In this case, alignments will be counted.
     * @return 0, indicating no alignments yet found.
     */
    @Override
    public Integer reduceInit() { return 0; }

    /**
     * Calculates the number of alignments found.
     * @param value Number of alignments found by this map.
     * @param sum Number of alignments found before this map.
     * @return Number of alignments found up to and including this map.
     */    
    @Override
    public Integer reduce(Integer value, Integer sum) {
        return value + sum;
    }

    /**
     * Cleanup.
     * @param result Number of reads processed.
     */    
    @Override
    public void onTraversalDone(Integer result) {
        aligner.close();
        if (outputBam != null)
            outputBam.close();
        super.onTraversalDone(result);
    }

}
