/*
 * Decompiled with CFR 0.152.
 */
package net.sf.samtools;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.List;
import java.util.TreeMap;
import net.sf.samtools.BAMFileConstants;
import net.sf.samtools.BAMIndex;
import net.sf.samtools.BAMIndexContent;
import net.sf.samtools.Bin;
import net.sf.samtools.Chunk;
import net.sf.samtools.LinearIndex;
import net.sf.samtools.SAMException;
import net.sf.samtools.util.RuntimeIOException;

abstract class AbstractBAMFileIndex
implements BAMIndex {
    private static final int MAX_BINS = 37450;
    protected static final int BIN_SPAN = 0x20000000;
    private static final int[] LEVEL_STARTS = new int[]{0, 1, 9, 73, 585, 4681};
    private final File mFile;
    private MappedByteBuffer mFileBuffer;

    protected AbstractBAMFileIndex(File file) {
        this.mFile = file;
        if (file != null) {
            this.open();
        }
    }

    public int getNumIndexLevels() {
        return LEVEL_STARTS.length;
    }

    public int getFirstBinInLevel(int levelNumber) {
        return LEVEL_STARTS[levelNumber];
    }

    public int getLevelSize(int levelNumber) {
        if (levelNumber == this.getNumIndexLevels()) {
            return 37451 - LEVEL_STARTS[levelNumber];
        }
        return LEVEL_STARTS[levelNumber + 1] - LEVEL_STARTS[levelNumber];
    }

    public int getLevelForBin(Bin bin) {
        if (bin.getBinNumber() >= 37450) {
            throw new SAMException("Tried to get level for invalid bin.");
        }
        for (int i = this.getNumIndexLevels() - 1; i >= 0; --i) {
            if (bin.getBinNumber() < LEVEL_STARTS[i]) continue;
            return i;
        }
        throw new SAMException("Unable to find correct bin for bin " + bin);
    }

    public int getFirstLocusInBin(Bin bin) {
        int level = this.getLevelForBin(bin);
        int levelStart = LEVEL_STARTS[level];
        int levelSize = (level == this.getNumIndexLevels() - 1 ? 37449 : LEVEL_STARTS[level + 1]) - levelStart;
        return (bin.getBinNumber() - levelStart) * (0x20000000 / levelSize) + 1;
    }

    public int getLastLocusInBin(Bin bin) {
        int level = this.getLevelForBin(bin);
        int levelStart = LEVEL_STARTS[level];
        int levelSize = (level == this.getNumIndexLevels() - 1 ? 37449 : LEVEL_STARTS[level + 1]) - levelStart;
        return (bin.getBinNumber() - levelStart + 1) * (0x20000000 / levelSize);
    }

    @Override
    public void open() {
        if (this.mFileBuffer != null) {
            return;
        }
        try {
            FileInputStream fileStream = new FileInputStream(this.mFile);
            FileChannel fileChannel = fileStream.getChannel();
            this.mFileBuffer = fileChannel.map(FileChannel.MapMode.READ_ONLY, 0L, fileChannel.size());
            this.mFileBuffer.order(ByteOrder.LITTLE_ENDIAN);
            fileChannel.close();
            fileStream.close();
        }
        catch (IOException exc) {
            throw new RuntimeIOException(exc.getMessage(), exc);
        }
        this.seek(0);
        byte[] buffer = new byte[4];
        this.readBytes(buffer);
        if (!Arrays.equals(buffer, BAMFileConstants.BAM_INDEX_MAGIC)) {
            this.close();
            throw new RuntimeException("Invalid file header in BAM index " + this.mFile + ": " + new String(buffer));
        }
    }

    public int getNumberOfReferences() {
        if (this.mFileBuffer == null) {
            throw new SAMException("Cannot query a closed index file");
        }
        this.seek(4);
        return this.readInteger();
    }

    @Override
    public void close() {
        this.mFileBuffer = null;
    }

    public void writeText(int n_ref, File OUTPUT, boolean sortBins) throws FileNotFoundException {
        PrintWriter pw = new PrintWriter(OUTPUT);
        pw.println("n_ref=" + n_ref);
        for (int i = 0; i < n_ref; ++i) {
            try {
                if (this.getQueryResults(i) == null) {
                    BAMIndexContent.writeNullTextContent(pw, i);
                    continue;
                }
                this.getQueryResults(i).writeText(pw, sortBins);
                continue;
            }
            catch (Exception e) {
                System.err.println(e.getMessage() + " Exception writing text for reference " + i);
            }
        }
        pw.close();
    }

    public void writeBinary(int n_ref, File OUTPUT, boolean sortBins, long bamFileSize) throws IOException {
        int defaultBufferSize = 1000000;
        int bufferSize = bamFileSize < 1000000L && bamFileSize != 0L ? (int)bamFileSize : 1000000;
        FileOutputStream stream = new FileOutputStream(OUTPUT, true);
        FileChannel fileChannel = stream.getChannel();
        ByteBuffer bb = ByteBuffer.allocateDirect(bufferSize);
        bb.order(ByteOrder.LITTLE_ENDIAN);
        byte[] magic = BAMFileConstants.BAM_INDEX_MAGIC;
        bb.put(magic);
        bb.putInt(n_ref);
        for (int i = 0; i < n_ref; ++i) {
            if (this.getQueryResults(i) == null) {
                BAMIndexContent.writeNullBinaryContent(bb);
                continue;
            }
            this.getQueryResults(i).writeBinary(bb, sortBins);
            bb.flip();
            fileChannel.write(bb);
            bb.position(0);
            bb.limit(bufferSize);
        }
        bb.flip();
        fileChannel.write(bb);
        fileChannel.close();
        stream.close();
    }

    @Override
    public long getStartOfLastLinearBin() {
        if (this.mFileBuffer == null) {
            throw new SAMException("Cannot query a closed index file");
        }
        this.seek(4);
        int sequenceCount = this.readInteger();
        long lastLinearIndexPointer = -1L;
        for (int i = 0; i < sequenceCount; ++i) {
            int nBins = this.readInteger();
            for (int j1 = 0; j1 < nBins; ++j1) {
                this.skipBytes(4);
                int nChunks = this.readInteger();
                this.skipBytes(16 * nChunks);
            }
            int nLinearBins = this.readInteger();
            if (nLinearBins <= 0) continue;
            this.skipBytes(8 * (nLinearBins - 1));
            lastLinearIndexPointer = this.readLong();
        }
        return lastLinearIndexPointer;
    }

    protected BAMIndexContent query(int referenceSequence, int startPos, int endPos) {
        if (this.mFileBuffer == null) {
            throw new SAMException("Cannot query a closed index file");
        }
        this.seek(4);
        ArrayList<Bin> bins = null;
        TreeMap<Bin, List<Chunk>> binToChunks = new TreeMap<Bin, List<Chunk>>();
        LinearIndex linearIndex = null;
        int sequenceCount = this.readInteger();
        if (referenceSequence >= sequenceCount) {
            return null;
        }
        BitSet regionBins = this.regionToBins(startPos, endPos);
        if (regionBins == null) {
            return null;
        }
        this.skipToSequence(referenceSequence);
        int binCount = this.readInteger();
        bins = new ArrayList<Bin>(binCount);
        for (int binNumber = 0; binNumber < binCount; ++binNumber) {
            ArrayList<Chunk> chunks = new ArrayList<Chunk>();
            int indexBin = this.readInteger();
            int nChunks = this.readInteger();
            if (regionBins.get(indexBin)) {
                for (int ci = 0; ci < nChunks; ++ci) {
                    long chunkBegin = this.readLong();
                    long chunkEnd = this.readLong();
                    chunks.add(new Chunk(chunkBegin, chunkEnd));
                }
            } else {
                this.skipBytes(16 * nChunks);
            }
            Bin bin = new Bin(referenceSequence, indexBin);
            bins.add(bin);
            binToChunks.put(bin, chunks);
        }
        Collections.sort(bins);
        int nLinearBins = this.readInteger();
        int regionLinearBinStart = LinearIndex.convertToLinearIndexOffset(startPos);
        int regionLinearBinStop = LinearIndex.convertToLinearIndexOffset(endPos) > 0 ? LinearIndex.convertToLinearIndexOffset(endPos) : nLinearBins - 1;
        long[] linearIndexEntries = new long[]{};
        if (regionLinearBinStart < nLinearBins) {
            linearIndexEntries = new long[regionLinearBinStop - regionLinearBinStart + 1];
            this.skipBytes(8 * regionLinearBinStart);
            for (int linearBin = regionLinearBinStart; linearBin <= regionLinearBinStop; ++linearBin) {
                linearIndexEntries[linearBin - regionLinearBinStart] = this.readLong();
            }
        }
        linearIndex = new LinearIndex(referenceSequence, regionLinearBinStart, linearIndexEntries);
        return new BAMIndexContent(referenceSequence, bins, binToChunks, linearIndex);
    }

    protected abstract BAMIndexContent getQueryResults(int var1);

    protected int getMaxAddressibleGenomicLocation() {
        return 0x20000000;
    }

    protected BitSet regionToBins(int startPos, int endPos) {
        int k;
        int end;
        int maxPos = 0x1FFFFFFF;
        int start = startPos <= 0 ? 0 : startPos - 1 & 0x1FFFFFFF;
        int n = end = endPos <= 0 ? 0x1FFFFFFF : endPos - 1 & 0x1FFFFFFF;
        if (start > end) {
            return null;
        }
        BitSet bitSet = new BitSet(37450);
        bitSet.set(0);
        for (k = 1 + (start >> 26); k <= 1 + (end >> 26); ++k) {
            bitSet.set(k);
        }
        for (k = 9 + (start >> 23); k <= 9 + (end >> 23); ++k) {
            bitSet.set(k);
        }
        for (k = 73 + (start >> 20); k <= 73 + (end >> 20); ++k) {
            bitSet.set(k);
        }
        for (k = 585 + (start >> 17); k <= 585 + (end >> 17); ++k) {
            bitSet.set(k);
        }
        for (k = 4681 + (start >> 14); k <= 4681 + (end >> 14); ++k) {
            bitSet.set(k);
        }
        return bitSet;
    }

    protected List<Chunk> optimizeChunkList(List<Chunk> chunks, long minimumOffset) {
        Chunk lastChunk = null;
        Collections.sort(chunks);
        ArrayList<Chunk> result = new ArrayList<Chunk>();
        for (Chunk chunk : chunks) {
            if (chunk.getChunkEnd() <= minimumOffset) continue;
            if (result.isEmpty()) {
                result.add(chunk);
                lastChunk = chunk;
                continue;
            }
            long lastFileBlock = this.getFileBlock(lastChunk.getChunkEnd());
            long chunkFileBlock = this.getFileBlock(chunk.getChunkStart());
            if (chunkFileBlock - lastFileBlock > 1L) {
                result.add(chunk);
                lastChunk = chunk;
                continue;
            }
            if (chunk.getChunkEnd() <= lastChunk.getChunkEnd()) continue;
            lastChunk.setChunkEnd(chunk.getChunkEnd());
        }
        return result;
    }

    private void skipToSequence(int sequenceIndex) {
        for (int i = 0; i < sequenceIndex; ++i) {
            int nBins = this.readInteger();
            for (int j = 0; j < nBins; ++j) {
                int bin = this.readInteger();
                int nChunks = this.readInteger();
                this.skipBytes(16 * nChunks);
            }
            int nLinearBins = this.readInteger();
            this.skipBytes(8 * nLinearBins);
        }
    }

    private long getFileBlock(long bgzfOffset) {
        return bgzfOffset >> 16 & 0xFFFFFFFFFFFFL;
    }

    private void readBytes(byte[] bytes) {
        this.mFileBuffer.get(bytes);
    }

    private int readInteger() {
        return this.mFileBuffer.getInt();
    }

    private long readLong() {
        return this.mFileBuffer.getLong();
    }

    private void skipBytes(int count) {
        this.mFileBuffer.position(this.mFileBuffer.position() + count);
    }

    private void seek(int position) {
        this.mFileBuffer.position(position);
    }
}

