/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.reference;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Iterator;
import net.sf.picard.PicardException;
import net.sf.picard.io.IoUtil;
import net.sf.picard.reference.AbstractFastaSequenceFile;
import net.sf.picard.reference.FastaSequenceIndex;
import net.sf.picard.reference.FastaSequenceIndexEntry;
import net.sf.picard.reference.ReferenceSequence;
import net.sf.samtools.SAMSequenceDictionary;
import net.sf.samtools.SAMSequenceRecord;

public class IndexedFastaSequenceFile
extends AbstractFastaSequenceFile {
    private final FileChannel channel;
    private final FastaSequenceIndex index;
    private Iterator<FastaSequenceIndexEntry> indexIterator;

    public IndexedFastaSequenceFile(File file, FastaSequenceIndex index) {
        super(file);
        FileInputStream in;
        if (index == null) {
            throw new IllegalArgumentException("Null index for fasta " + file);
        }
        this.index = index;
        IoUtil.assertFileIsReadable(file);
        try {
            in = new FileInputStream(file);
        }
        catch (FileNotFoundException e) {
            throw new PicardException("Fasta file should be readable but is not: " + file, e);
        }
        this.channel = in.getChannel();
        if (this.getSequenceDictionary() == null) {
            throw new IllegalArgumentException("No sequence dictionary exists for fasta " + file + ".  Cannot create indexed reader");
        }
        this.reset();
        IndexedFastaSequenceFile.sanityCheckDictionaryAgainstIndex(file.getAbsolutePath(), this.sequenceDictionary, index);
    }

    public IndexedFastaSequenceFile(File file) {
        this(file, new FastaSequenceIndex(IndexedFastaSequenceFile.findFastaIndex(file)));
    }

    @Override
    public boolean isIndexed() {
        return true;
    }

    private static File findFastaIndex(File fastaFile) {
        File indexFile = new File(fastaFile.getAbsolutePath() + ".fai");
        if (!indexFile.exists()) {
            return null;
        }
        return indexFile;
    }

    public static boolean canCreateIndexedFastaReader(File fastaFile) {
        return fastaFile.exists() && IndexedFastaSequenceFile.findSequenceDictionary(fastaFile) != null && IndexedFastaSequenceFile.findFastaIndex(fastaFile) != null;
    }

    protected static void sanityCheckDictionaryAgainstIndex(String fastaFile, SAMSequenceDictionary sequenceDictionary, FastaSequenceIndex index) {
        if (sequenceDictionary.getSequences().size() != index.size()) {
            throw new PicardException("Sequence dictionary and index contain different numbers of contigs");
        }
        Iterator sequenceIterator = sequenceDictionary.getSequences().iterator();
        Iterator<FastaSequenceIndexEntry> indexIterator = index.iterator();
        while (sequenceIterator.hasNext() && indexIterator.hasNext()) {
            SAMSequenceRecord sequenceEntry = (SAMSequenceRecord)sequenceIterator.next();
            FastaSequenceIndexEntry indexEntry = indexIterator.next();
            if (!sequenceEntry.getSequenceName().equals(indexEntry.getContig())) {
                throw new PicardException(String.format("Mismatch between sequence dictionary fasta index for %s, sequence '%s' != '%s'.", fastaFile, sequenceEntry.getSequenceName(), indexEntry.getContig()));
            }
            if ((long)sequenceEntry.getSequenceLength() == indexEntry.getSize()) continue;
            throw new PicardException("Index length does not match dictionary length for contig: " + sequenceEntry.getSequenceName());
        }
    }

    @Override
    public SAMSequenceDictionary getSequenceDictionary() {
        return this.sequenceDictionary;
    }

    @Override
    public ReferenceSequence getSequence(String contig) {
        return this.getSubsequenceAt(contig, 1L, (int)this.index.getIndexEntry(contig).getSize());
    }

    @Override
    public ReferenceSequence getSubsequenceAt(String contig, long start, long stop) {
        if (start > stop) {
            throw new PicardException(String.format("Malformed query; start point %d lies after end point %d", start, stop));
        }
        FastaSequenceIndexEntry indexEntry = this.index.getIndexEntry(contig);
        if (stop > indexEntry.getSize()) {
            throw new PicardException("Query asks for data past end of contig");
        }
        int length = (int)(stop - start + 1L);
        byte[] target = new byte[length];
        ByteBuffer targetBuffer = ByteBuffer.wrap(target);
        int basesPerLine = indexEntry.getBasesPerLine();
        int bytesPerLine = indexEntry.getBytesPerLine();
        long startOffset = (start - 1L) / (long)basesPerLine * (long)bytesPerLine + (start - 1L) % (long)basesPerLine;
        long stopOffset = (stop - 1L) / (long)basesPerLine * (long)bytesPerLine + (stop - 1L) % (long)basesPerLine;
        int size = (int)(stopOffset - startOffset) + 1;
        ByteBuffer channelBuffer = ByteBuffer.allocate(size);
        try {
            this.channel.read(channelBuffer, indexEntry.getLocation() + startOffset);
        }
        catch (IOException ex) {
            throw new PicardException("Unable to load specified portion of FASTA into memory.");
        }
        channelBuffer.position(0);
        channelBuffer.limit(Math.min(basesPerLine - (int)startOffset % bytesPerLine, size));
        while (channelBuffer.hasRemaining()) {
            targetBuffer.put(channelBuffer);
            channelBuffer.limit(Math.min(channelBuffer.limit() + bytesPerLine, size));
            channelBuffer.position(Math.min(channelBuffer.position() + bytesPerLine - basesPerLine, size));
        }
        return new ReferenceSequence(contig, this.sequenceDictionary.getSequenceIndex(contig), target);
    }

    @Override
    public ReferenceSequence nextSequence() {
        if (!this.indexIterator.hasNext()) {
            return null;
        }
        return this.getSequence(this.indexIterator.next().getContig());
    }

    @Override
    public void reset() {
        this.indexIterator = this.index.iterator();
    }

    @Override
    public String toString() {
        return this.file.getAbsolutePath();
    }
}

