/*------------------------------------------------------------*
 *
 * File : varia.h
 * Author: NTM
 * Created : 30/09/03
 *
 *
 * Copyright (C) Nicolas Thierry-Mieg, 2006.
 *
 *
 * This file is part of InterPool, written by 
 * Nicolas Thierry-Mieg (CNRS, France) Nicolas.Thierry-Mieg@imag.fr
 *
 * InterPool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InterPool is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with InterPool; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *-------------------------------------------------------------*/

#ifndef _varia_h_
#define _varia_h_

#include "types.h" /* MOT */

/*!
  \file varia.h
  \brief This folds holds all methods relevant to VVs.

  Data structures used:

  //////////// Variables'Values Vector (VV) ///////////////<br>
  the VV, used to represent the interactions (ie variables'values) deduced 
  from the observed pool values, is stored as a vector of MOTs (size 2*n/(sizeof(MOT)*CHAR_BIT)).
  each variable takes 2 bits, meaning:
  - 00 -> variable is negative
  - 01 -> variable is ambiguous (or not yet known)
  - 11 -> variable is true
  
  Memory for VVs is allocated in buildInitialVV.
  
  NOTE: the same structure is used for VVs and for Pools, in order
  to allow efficient updates of VVs (using bitwise operations on MOTs).<br>
  Therefore, it seems natural that pools and VVs share a common get/set 
  interface. In fact, this is the case: to get and set the value of a variable
  in a VV, use functions getpoolvalue and setpoolvalue from pools.h.
  
  //////////// Positive Variables Vector ///////////////<br>
  For simulations, a set of variables is randomly chosen to be positive,
  using simulateVars/2.<br>
  These positive variables are not stored using the VV structure,
  because typically few variables are positive, and it would be a waste
  to store their values in a full VV.<br>
  Instead, they are stored in a vector of ints, where the first element is
  the number of positive variables, followed by the sorted variables themselves.
*/

//////////////////////////////////////////////////////////////////
// For solving a (noisy) signature:

/*!
  \brief Allocates memory for a VV vector, and fills it
  with inibyte. 
  Useful value: 85 (==01010101, meaning "all variables are unknown").
  I also use 0 occasionally.<br>
  NOTE: n should be 1 (to get a single MOT) or it should be a constant
  value across the life of an execution (ie, the number of vars). Otherwise,
  you will nullify the "static result storage" optimization of nbmotparpool.
*/
MOT* buildInitialVV(int n, unsigned char inibyte) ;


/*!
  \brief Update the VV VVtab, to reflect the fact that pool is 
  negative.
*/
void updateVVNegPool(MOT* VVtab, MOT* pool, int n);


/*!
  \brief Update the VV VVtab, to reflect the fact that pool 
  is positive.
  Returns 1 if pool being positive causes a conflict with current VVtab,
  O otherwise.
*/
int updateVVPosPool(MOT* VVtab, MOT* pool, int n);


/*!
  \brief Return TRUE if pool contains a variable that is not 
  marked 00 in VVtab.
*/
bool hasANonNegVar(MOT* pool, MOT *VVtab, int n);


/*!
  \brief This function updates the VVtab to VVtab | pool (bitwise).
*/
void naivePosUpdate(MOT* VVtab, MOT* pool, int n);


//////////////////////////////////////////////////////////////////
// For simulations:

/*!
  \brief Generate a set of simulated values for the n Boolean variables {0,..,n-1}.
  nbPosVars is the number of positive variables (there used to be avposvars
  and varposvars but this has been removed 25/06/04).
  contraint: nbPosVars must be >=0 (this is checked for).
  This function returns a vector of ints of size nbPosVars+1,
  in a memory area allocated in the function itself.
  The vector holds: nbPosVars at index 0, followed by the SORTED positive 
  variables starting at index 1.
*/
int* simulateVars(int n, int nbPosVars);


#endif

