/*-----------------------------------------------------------*
 *
 * File : pools.h
 * Author : NTM
 * Created : 30/09/03
 *
 *
 * Copyright (C) Nicolas Thierry-Mieg, 2006.
 *
 *
 * This file is part of InterPool, written by 
 * Nicolas Thierry-Mieg (CNRS, France) Nicolas.Thierry-Mieg@imag.fr
 *
 * InterPool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InterPool is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with InterPool; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *------------------------------------------------------------*/

#ifndef _pools_h_
#define _pools_h_

#include "types.h" /* MOT */


/*!
  \file pools.h
  \brief This file holds all methods relevant to pools.

  Pool Data Structure:<br>
  A pool is stored as a vector of MOTs of size nbmotparpool(n)~=n/(sizeof(MOT)*4).
  each variable takes 2 bits, whose meanings are:
  - 00 -> variable is absent from pool
  - 11 -> variable is present in pool
  
  The other 2 possibilities (01 and 10) are currently not used, but using 
  2 bits is useful for the decoding (in solveSigSimple.c and the various 
  solvexp*.c), where bitwise operations take place between pools and 
  the variables'values vector.

  In the current implementation, in fact all pools are stored in a large
  block of MOTs (called tabpool); but this is transparent: all access to 
  pools is done through pool/3.
*/


/*!
  \brief read a pooling design in a file and return a pool data structure.
  designFile must be readable and contain a set of pools in the correct format.
  format is: one line per pool, each line is a list of variables
  (ie, ints in {0,...,(n-1)}) separated by ':'
*/
MOT* buildPools(char* designFile, int n, int nbPools) ;


/*!
  \brief Returns the number of MOTs used to store one pool.
*/
int nbmotparpool(int n);


/*!
  \brief Return the pool (ie, pointer to the first MOT where it is stored)
  in the global table tabpool, for pool of absolute number absPoolNum.
*/
MOT* pool(MOT* tabpool, int absPoolNum, int n);

//////////////////////////////////////////////////////////////////
// the following getpoolvalue and setpoolvalue are also used to
// interface with Variables'Values Vectors (VVs). See varia.h for details.
//////////////////////////////////////////////////////////////////

/*!
  \brief Returns the value (0,1,10 or 11 currently) of variable var in the pool/VV.
*/
int getpoolvalue(MOT* pool, int var);

/*! 
  \brief Set the value of variable var in pool/VV to value.
  value should be 0, 1, 10 or 11 (stored on 2 bits in pool).
*/
void setpoolvalue(MOT* pool, int var, int value);


#endif
