#!/usr/bin/env perl

#
# Copyright (C) Nicolas Thierry-Mieg, 2009.
#
#
# This is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This script is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this script; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


# read coords2pools.WASP2x4, and build coords2pools.WASP2x16
# holding the coords to pools mapping of the WASP2x16
# plates obtained by copying the WASP2x4 plates into 
# 1536 format plates.

use strict ;
use warnings ;

# We just have to modify the plate numbers and coordinates.
# However we will save the data and only print in the
# end, so the order is good
# See makeBatchMappings.WASP2.pl for details, it inspired most
# of this code.

# first fill plate2pool (so we can fill it out of order),
# content will then be printed in the correct order to a file.
# "$batch,$poolnum" is stored in $plate2pool[($P-1)*384 + ($C-1)*16 + ($R-1)]
my @plate2pool ;


my $infile = "coords2pools.WASP2x4" ;
open(INFILE, "$infile") ||
    die "cannot open $infile for reading\n" ;

my $outfile = "coords2pools.WASP2x16" ;
(-e $outfile) && die "$outfile already exists, dying\n" ;
open(OUTFILE, ">$outfile") ||
    die "cannot open $outfile for writing\n" ;

# first line, just change the title
my $line = <INFILE> ;
$line =~ s/WASP2x4/WASP2x16/ ;
print OUTFILE $line ;
# second line: headers, identical
$line = <INFILE> ;
print OUTFILE $line ;
# now the data
while($line = <INFILE>)
{
    chomp $line ;
    ($line =~ /^(\d+),(\d+),(\d+),(\d+,\d+)$/) ||
	die "cannot parse line:\n$line\n" ;

    my ($P,$C,$R,$rest) = ($1,$2,$3,$4) ;

    # As explained by Xiaofeng in an email from 23/10/2006,
    # he combined the WASP2x4 plates similarly to what
    # we did for the 384 plates, ie he takes them in
    # order from 1 to 20 and shifts them in a cycle to
    # .A, .B, .C, .D (taking a new 1536 plate after .D).

    # use the index of the plate to simplify notations
    my $ip = $P - 1 ;

    my $Pnew = int($ip / 4) + 1 ;

    # set $rowDec = -1 for .A and .B (odd rows), 0 otherwise
    # similarly, set $colDec = -1 for .A and .C, 0 otherwise
    my ($rowDec,$colDec) = (0,0) ;
    ((($ip % 4) == 0) || (($ip % 4) == 1)) && ($rowDec= -1) ;
    (($ip % 2) == 0) && ($colDec= -1) ;

    my $Cnew = $C * 2 + $colDec ;
    my $Rnew = $R * 2 + $rowDec ;

    # OK, save (we will print later to get the order right)
    $plate2pool[($Pnew-1)*1536 + ($Cnew-1)*32 + ($Rnew-1)] = "$rest" ;
}
close(INFILE) ;


# sanity: plate2pool should have 169*40 + 94*4 + 19*2 + 75*2 == 7324
# non-empty entries (empty entries occur for empty wells, eg bottom 
# right corner of every plate)
{
    my $cnt = 0 ;
    foreach my $i (0..$#plate2pool)
    {
	(defined $plate2pool[$i]) && ($cnt++) ;
    }
    ($cnt != 7324) && die "plate2pool has $cnt elements, should be 7324\n" ;
}


# now print
foreach my $P (1..5)
{
    foreach my $C (1..48)
    {
	foreach my $R (1..32)
	{
	    (defined $plate2pool[($P-1)*1536 + ($C-1)*32 + ($R-1)]) && 
		(print OUTFILE "$P,$C,$R,", $plate2pool[($P-1)*1536 + ($C-1)*32 + ($R-1)], "\n") ;
	}
    }
}



