/**
*	SIGRS - Identifying genomic regions of contrasting composition using a partial sum process
*	Copyright (C) 2008 Pontus Larsson
*	 
*	This file is part of SIGRS.
*	  
*	SIGRS is free software: you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation, either version 3 of the License, or
*	(at your option) any later version.
*
*	SIGRS is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with SIGRS. If not, see <http://www.gnu.org/licenses/>.
*/
package SIGRS;

import java.io.File;
import java.util.Date;
       
/**
*	SIGRS is a collection of routines used in searching for regions of contrasting composition (CCRs) in sequence files using a partial sum process.
*	Significance of segments is evaluated using Karlin-Altschul statistics and specifically an extension by Karlin-Dembo allowing
*	for nucleotides to have a Markov-dependence (see e.g. <a href="http://www.pnas.org/cgi/reprint/90/12/5873" target="_blank">Karlin & Altschul (1993)</a> and
*	<a href="http://www.jstor.org/view/00018678/ap050087/05a00070/0" target="_blank">Karlin & Dembo (1992)</a>
*	<P>
*	The routines are provided as is and no guarantee regarding stability etc. is given so use at your own risk!
*	<P>
*	<P>
*	See publication <b>Larsson, P., Hinas, A., Ardell, D.H., Kirsebom, L.A., Virtanen, A. and Sderbom, F.</b> <i>De novo search for non-coding RNA genes in the AT-rich genome of Dictyostelium discoideum: performance of
*	Markov-dependent genome feature scoring</i>
*	<P>
*	Questions and comments can be directed to <a href="mailto:Pontus.Larsson@icm.uu.se">Pontus.Larsson@icm.uu.se</a>
*	@author Pontus Larsson
*
*/


public class SIGRSScoreObject {
	private double[][] M0ScoreMatrix;
	private double[][] M1ScoreMatrix;
	private double[][][] backgroundFrequencies;
	private double[][][] targetFrequencies;
	private double K0;
	private double L0;
	private double H0;
	private double K1;
	private double L1;
	private double H1;

	public SIGRSScoreObject() {
		M0ScoreMatrix = new double[0][];
		M1ScoreMatrix = new double[0][];
		backgroundFrequencies = new double[0][][];
		targetFrequencies = new double[0][][];
		K0 = 0;
		L0 = 0;
		H0 = 0;
		K1 = 0;
		L1 = 0;
		H1 = 0;
	}

	/**
	 * Returns the value of M0ScoreMatrix.
	 */
	public double[][] getM0ScoreMatrix() {return M0ScoreMatrix;}
	/**
	 * Sets the value of M0ScoreMatrix.
	 * @param M0ScoreMatrix The value to assign M0ScoreMatrix.
	 */
	public void setM0ScoreMatrix(double[][] M0ScoreMatrix) {this.M0ScoreMatrix = M0ScoreMatrix;}
	/**
	 * Returns the value of M1ScoreMatrix.
	 */
	public double[][] getM1ScoreMatrix() {return M1ScoreMatrix;}
	/**
	 * Sets the value of M1ScoreMatrix.
	 * @param M1ScoreMatrix The value to assign M1ScoreMatrix.
	 */
	public void setM1ScoreMatrix(double[][] M1ScoreMatrix) {this.M1ScoreMatrix = M1ScoreMatrix;}
	/**
	 * Returns the value of backgroundFrequencies.
	 */
	public double[][][] getBackgroundFrequencies() {return backgroundFrequencies;}
	/**
	 * Sets the value of backgroundFrequencies.
	 * @param backgroundFrequencies The value to assign backgroundFrequencies.
	 */
	public void setBackgroundFrequencies(double[][][] backgroundFrequencies) {this.backgroundFrequencies = backgroundFrequencies;}
	/**
	 * Returns the value of targetFrequencies.
	 */
	public double[][][] getTargetFrequencies() {return targetFrequencies;}
	/**
	 * Sets the value of targetFrequencies.
	 * @param targetFrequencies The value to assign targetFrequencies.
	 */
	public void setTargetFrequencies(double[][][] targetFrequencies) {this.targetFrequencies = targetFrequencies;}
	/**
	 * Returns the value of K0.
	 */
	public double getK0() {return K0;}
	/**
	 * Sets the value of K0.
	 * @param K0 The value to assign K0.
	 */
	public void setK0(double K0) {this.K0 = K0;}
	/**
	 * Returns the value of L0.
	 */
	public double getL0() {return L0;}
	/**
	 * Sets the value of L0.
	 * @param L0 The value to assign L0.
	 */
	public void setL0(double L0) {this.L0 = L0;}
	/**
	 * Returns the value of H0.
	 */
	public double getH0() {return H0;}
	/**
	 * Sets the value of H0.
	 * @param H0 The value to assign H0.
	 */
	public void setH0(double H0) {this.H0 = H0;}
	/**
	 * Returns the value of K1.
	 */
	public double getK1() {return K1;}
	/**
	 * Sets the value of K1.
	 * @param K1 The value to assign K1.
	 */
	public void setK1(double K1) {this.K1 = K1;}
	/**
	 * Returns the value of L1.
	 */
	public double getL1() {return L1;}
	/**
	 * Sets the value of L1.
	 * @param L1 The value to assign L1.
	 */
	public void setL1(double L1) {this.L1 = L1;}
	/**
	 * Returns the value of H1.
	 */
	public double getH1() {return H1;}
	/**
	 * Sets the value of H1.
	 * @param H1 The value to assign H1.
	 */
	public void setH1(double H1) {this.H1 = H1;}

	/**
	*	Calculates the frequencies, scores and associated parameters for the M0 and M1 model
	*	@param backgroundSeqs Array with encoded background sequences
	*	@param targetSeqs Array with encoded target sequences
	*/
	public void calculateScores(byte[][] backgroundSeqs, byte[][] targetSeqs, LogWriter lw) throws Exception {
		calculateBackgroundFrequencies(backgroundSeqs,lw);
		calculateTargetFrequencies(targetSeqs,lw);
		calculateScores(lw);
	}
	public void calculateScores(File backgroundSeqs, File targetSeqs, LogWriter lw) throws Exception {
		this.setBackgroundFrequencies(calculateFrequencies(backgroundSeqs));
		this.setTargetFrequencies(calculateFrequencies(targetSeqs));
		calculateScores(lw);
	}
	public void calculateScores(LogWriter lw) {
		lw.println("# "+(new Date()).toString()+"\tCalculates M0 score matrix");
		this.M0ScoreMatrix = SIGRSScoreObject.calculateScores(this.backgroundFrequencies[0],this.targetFrequencies[0]);
		for (int i=0; i<this.M0ScoreMatrix.length; i++)
			lw.println("\t"+String.valueOf((int) this.M0ScoreMatrix[i][0]));
		lw.println("# "+(new Date()).toString()+"\tCalculates M1 score matrix");
		this.M1ScoreMatrix = SIGRSScoreObject.calculateScores(this.backgroundFrequencies[2],this.targetFrequencies[2]);
		for (int i=0; i<this.M1ScoreMatrix.length; i++) {
			for (int j=0; j<this.M1ScoreMatrix[i].length; j++)
				lw.print("\t"+Methods.pad(String.valueOf((int) this.M1ScoreMatrix[i][j]),3));
			lw.println();
		}
		lw.print("# "+(new Date()).toString()+"\tCalculates M0 lambda = ");
		this.L0 = KAStatistics.lambda(this.M0ScoreMatrix,this.backgroundFrequencies[0]);
		lw.println(String.valueOf(this.L0));
		lw.print("# "+(new Date()).toString()+"\tCalculates M0 entropy = ");
		this.H0 = KAStatistics.entropy(this.M0ScoreMatrix,this.backgroundFrequencies[0],this.L0);
		lw.println(String.valueOf(this.H0));
		lw.print("# "+(new Date()).toString()+"\tCalculates M0 K = ");
		this.K0 = KAStatistics.estimateK(this.M0ScoreMatrix,this.backgroundFrequencies[0],this.L0,this.H0);
		lw.println(String.valueOf(this.K0));
		lw.print("# "+(new Date()).toString()+"\tCalculates M1 theta = ");
		double[] t = KDStatistics.theta(this.M1ScoreMatrix,this.backgroundFrequencies[2]);
		this.L1 = t[0];
		lw.println(String.valueOf(this.L1));
		lw.print("# "+(new Date()).toString()+"\tCalculates M1 entropy = ");
		this.H1 = KAStatistics.entropy(this.M1ScoreMatrix,this.backgroundFrequencies[2],this.L1);
		lw.println(String.valueOf(this.H1));
		lw.print("# "+(new Date()).toString()+"\tCalculates M1 K = ");
		this.K1 = KDStatistics.K(this.L1,Methods.subarray(1,t.length,t),this.M1ScoreMatrix,this.backgroundFrequencies[2]);
		lw.println(String.valueOf(this.K1));
	}

	public void calculateBackgroundFrequencies(byte[][] backgroundSeqs, LogWriter lw) {
		this.setBackgroundFrequencies(SIGRSScoreObject.calculateFrequencies(backgroundSeqs));

		lw.println("# "+(new Date()).toString()+"\tCalculates background frequencies");
		lw.println("#\tMononucleotides:");
		for (int i=0; i<this.backgroundFrequencies[0].length; i++)
			lw.println("#\t\t"+String.valueOf(this.backgroundFrequencies[0][i][0]));
		lw.println("#\tDinucleotides:");
		for (int i=0; i<this.backgroundFrequencies[1].length; i++) {
			lw.print("#\t");
			for (int j=0; j<this.backgroundFrequencies[1][i].length; j++)
				lw.print("\t"+Methods.pad(String.valueOf(this.backgroundFrequencies[1][i][j]),10));
			lw.println();
		}
		lw.println("#\tConditional dinucleotides:");
		for (int i=0; i<this.backgroundFrequencies[2].length; i++) {
			lw.print("#\t");
			for (int j=0; j<this.backgroundFrequencies[2][i].length; j++)
				lw.print("\t"+Methods.pad(String.valueOf(this.backgroundFrequencies[2][i][j]),10));
			lw.println();
		}
	}
	public void calculateTargetFrequencies(byte[][] targetSeqs, LogWriter lw) {
		this.setTargetFrequencies(SIGRSScoreObject.calculateFrequencies(targetSeqs));

		lw.println("# "+(new Date()).toString()+"\tCalculates target frequencies");
		lw.println("#\tMononucleotides:");
		for (int i=0; i<this.targetFrequencies[0].length; i++)
			lw.println("#\t\t"+String.valueOf(this.targetFrequencies[0][i][0]));
		lw.println("#\tDinucleotides:");
		for (int i=0; i<this.targetFrequencies[1].length; i++) {
			lw.print("#\t");
			for (int j=0; j<this.targetFrequencies[1][i].length; j++)
				lw.print("\t"+Methods.pad(String.valueOf(this.targetFrequencies[1][i][j]),10));
			lw.println();
		}
		lw.println("#\tConditional dinucleotides:");
		for (int i=0; i<this.targetFrequencies[2].length; i++) {
			lw.print("#\t");
			for (int j=0; j<this.targetFrequencies[2][i].length; j++)
				lw.print("\t"+Methods.pad(String.valueOf(this.targetFrequencies[2][i][j]),10));
			lw.println();
		}
	}
	/**
	*	Calculates a set of frequencies from a set of input sequences
	*	@param seq An array of encoded input sequences
	*	@return An array with the different sets of frequencies:
	*	[0] -> Mononucleotide frequencies (4x1 array)
	*	[1] -> Dinucleotide frequencies (4x4 array)
	*	[2] -> Conditional dinucleotide frequencies (4x4 array)
	*/
	public static double[][][] calculateFrequencies(File seqFile) throws Exception {
		int[][] monoCounts = new int[][] {Methods.countMonoNucleotides(seqFile)};
		int[][][] diCounts = new int[][][] {Methods.countDiNucleotides(seqFile)};

		return calculateFrequencies(monoCounts,diCounts);
	}
	public static double[][][] calculateFrequencies(byte[][] seq) {
		int[][] monoCounts = new int[seq.length][];
		int[][][] diCounts = new int[seq.length][][];

		for (int i=0; i<seq.length; i++) {
			monoCounts[i] = Methods.countMonoNucleotides(seq[i]);
			diCounts[i] = Methods.countDiNucleotides(seq[i]);
		}

		return calculateFrequencies(monoCounts,diCounts);
	}
	public static double[][][] calculateFrequencies(int[][] monoCounts, int[][][] diCounts) {
		int[] mC;
		int[][] dC;

		int nMono;
		int nDi;

		double[][][] frequencies = new double[3][][];
		frequencies[0] = new double[4][1];
		frequencies[1] = new double[4][4];
		frequencies[2] = new double[4][4];

		for (int i=0; i<monoCounts.length; i++) {
			mC = monoCounts[i];
			dC = diCounts[i];

			// For the calculation of K to converge, the rowsum of the conditional dinucleotides
			// must add up to 1. The mononucleotide counts are compensated for nucleotides that are
			// not included in the dinucleotide count

			nMono = 0;
			nDi = 0;

			for (int k=1; k<5; k++) {
				// Compensate for [ACGT]N dinucleotides
				mC[k] = mC[k]-dC[k][0]-dC[k][5];
				nMono += mC[k];
				for (int j=1; j<5; j++)
					nDi += dC[k][j];
			}

			for (int k=0; k<4; k++)
				frequencies[0][k][0] += (1.*mC[k+1])/nMono;
			for (int k=0; k<4; k++)
				for (int j=0; j<4; j++)
					frequencies[1][k][j] += (1.*dC[k+1][j+1])/nDi;

			for (int k=0; k<4; k++)
				for (int j=0; j<4; j++)
					frequencies[2][k][j] += ((1.*dC[k+1][j+1])/nDi)/((1.*mC[k+1])/nMono);

			String n = new String("-ACGT");
			/*
			System.out.println("\nClass "+String.valueOf(i+1));
			System.out.println("__DATA__");
			for (int j=1; j<5; j++)
				System.out.println("t\t\t"+n.charAt(j)+"\t"+mC[j]);
			for (int j=1; j<5; j++)
				for (int k=1; k<5; k++)
					System.out.println("t\t\t"+n.charAt(j)+n.charAt(k)+"\t"+dC[j][k]);
			*/

		}

		frequencies[0] = Methods.vectorDivide(frequencies[0],1.*monoCounts.length);
		frequencies[1] = Methods.vectorDivide(frequencies[1],1.*monoCounts.length);
		frequencies[2] = Methods.vectorDivide(frequencies[2],1.*monoCounts.length);

		return frequencies;
	}

	/**
	*	Calculates score matrix based on target and background frequencies
	*	according to s = C*log(tg/bg) and rounded to the closest integer
	*	Equivalent to calling calculateScores(background,target,10.,2.)
	*	@param background Matrix holding the frequencies of letters in the background
	*	@param target Matrix holding the frequencies of letters in the target
	*	@return A score matrix with the same dimensions as the input frequency matrices
	*/
	public static double[][] calculateScores(double[][] background, double[][] target) {return calculateScores(background,target,10.,2.);}
	/**
	*	Calculates score matrix based on target and background frequencies
	*	according to s = C*log(tg/bg) and rounded to the closest integer
	*	@param background Matrix holding the frequencies of letters in the background
	*	@param target Matrix holding the frequencies of letters in the target
	*	@param C A scaling constant
	*	@param base The logarithm base
	*	@return A score matrix with the same dimensions as the input frequency matrices
	*/
	public static double[][] calculateScores(double[][] background, double[][] target, double C, double base) {
		if (background.length == 0 || background.length != target.length)
			return new double[0][0];
		double[][] scores = new double[background.length][background[0].length];
		for (int i=0; i<scores.length; i++)
			for (int j=0; j<scores[i].length; j++)
				scores[i][j] = Math.rint(C*Methods.logN(target[i][j]/background[i][j],base));
		return scores;
	}

	public static SIGRSScoreObject getScoreObject(File scoreFile, LogWriter lw) throws Exception {
		String contents = Methods.getFileContents(scoreFile);
		String[] rows = contents.split("\\n+");
		String[] cols;
		double[][][] backgroundFrequencies = new double[3][][];
		double[][][] targetFrequencies = new double[3][][];
		double[][] M0ScoreMatrix = new double[4][1];
		double K0;
		double H0;
		double L0;
		double[][] M1ScoreMatrix = new double[4][4];
		double K1;
		double H1;
		double L1;
		int i = 0;

		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse background mononucleotide frequencies
		backgroundFrequencies[0] = new double[4][1];
		for (int k=i; i<k+4; i++) {
			cols = rows[i].split("\\t");
			backgroundFrequencies[0][i-k][0] = Double.parseDouble(cols[1]);
		}

		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse background dinucleotide frequencies
		backgroundFrequencies[1] = new double[4][4];
		for (int k=i; i<k+4; i++) {
			cols = rows[i].split("\\t");
			for (int j=1; j<5; j++)
				backgroundFrequencies[1][i-k][j-1] = Double.parseDouble(cols[j]);
		}

		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse background conditional dinucleotide frequencies
		backgroundFrequencies[2] = new double[4][4];
		for (int k=i; i<k+4; i++) {
			cols = rows[i].split("\\t");
			for (int j=1; j<5; j++)
				backgroundFrequencies[2][i-k][j-1] = Double.parseDouble(cols[j]);
		}

		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse target mononucleotide frequencies
		targetFrequencies[0] = new double[4][1];
		for (int k=i; i<k+4; i++) {
			cols = rows[i].split("\\t");
			targetFrequencies[0][i-k][0] = Double.parseDouble(cols[1]);
		}

		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse target dinucleotide frequencies
		targetFrequencies[1] = new double[4][4];
		for (int k=i; i<k+4; i++) {
			cols = rows[i].split("\\t");
			for (int j=1; j<5; j++)
				targetFrequencies[1][i-k][j-1] = Double.parseDouble(cols[j]);
		}

		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse target conditional dinucleotide frequencies
		targetFrequencies[2] = new double[4][4];
		for (int k=i; i<k+4; i++) {
			cols = rows[i].split("\\t");
			for (int j=1; j<5; j++)
				targetFrequencies[2][i-k][j-1] = Double.parseDouble(cols[j]);
		}

		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse M0-model score matrix
		for (int k=i; i<k+4; i++) {
			cols = rows[i].split("\\t");
			M0ScoreMatrix[i-k][0] = Double.parseDouble(cols[1]);
		}
		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse M0-model parameter K
		K0 = Double.parseDouble(rows[i]);
		i++;
		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse M0-model entropy
		H0 = Double.parseDouble(rows[i]);
		i++;
		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse M0-model lambda
		L0 = Double.parseDouble(rows[i]);
		i++;

		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse M1-model score matrix
		for (int k=i; i<k+4; i++) {
			cols = rows[i].split("\\t");
			for (int j=1; j<5; j++)
				M1ScoreMatrix[i-k][j-1] = Double.parseDouble(cols[j]);
		}
		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse M1-model parameter K
		K1 = Double.parseDouble(rows[i]);
		i++;
		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse M1-model entropy
		H1 = Double.parseDouble(rows[i]);
		i++;
		while (rows[i].length() == 0 || rows[i].charAt(0) == '#')
			i++;
		// Parse M1-model lambda
		L1 = Double.parseDouble(rows[i]);
		i++;

		SIGRSScoreObject sso = new SIGRSScoreObject();
		sso.setBackgroundFrequencies(backgroundFrequencies);
		sso.setTargetFrequencies(targetFrequencies);
		sso.setM0ScoreMatrix(M0ScoreMatrix);
		sso.setK0(K0);
		sso.setH0(H0);
		sso.setL0(L0);
		sso.setM1ScoreMatrix(M1ScoreMatrix);
		sso.setK1(K1);
		sso.setH1(H1);
		sso.setL1(L1);

		lw.println(sso.toString());

		return sso;
	}

	/**
	*	Returns a string with the contents of the score object
	*/
	public String toString() {
		String output = new String();
		String nts = new String("ACGT");
		int pad = 40;

		output += "# This is the mononucleotide background frequencies\n";
		for (int i=0; i<this.backgroundFrequencies[0].length; i++)
			output += nts.charAt(i)+"\t"+String.valueOf(this.backgroundFrequencies[0][i][0])+"\n";
		output += "# This is the dinucleotide background frequencies\n#";
		for (int i=0; i<nts.length(); i++)
			output += "\t"+Methods.pad(nts.substring(i,i+1),pad);
		output += "\n";
		for (int i=0; i<this.backgroundFrequencies[1].length; i++) {
			output += nts.charAt(i);
			for (int j=0; j<this.backgroundFrequencies[1][i].length; j++)
				output += "\t"+Methods.pad(String.valueOf(this.backgroundFrequencies[1][i][j]),pad);
			output += "\n";
		}
		output += "# This is the conditional dinucleotide background frequencies\n#";
		for (int i=0; i<nts.length(); i++)
			output += "\t"+Methods.pad(nts.substring(i,i+1),pad);
		output += "\n";
		for (int i=0; i<this.backgroundFrequencies[2].length; i++) {
			output += nts.charAt(i);
			for (int j=0; j<this.backgroundFrequencies[2][i].length; j++)
				output += "\t"+Methods.pad(String.valueOf(this.backgroundFrequencies[2][i][j]),pad);
			output += "\n";
		}
		output += "# This is the mononucleotide target frequencies\n";
		for (int i=0; i<this.targetFrequencies[0].length; i++)
			output += nts.charAt(i)+"\t"+String.valueOf(this.targetFrequencies[0][i][0])+"\n";
		output += "# This is the dinucleotide target frequencies\n#";
		for (int i=0; i<nts.length(); i++)
			output += "\t"+Methods.pad(nts.substring(i,i+1),pad);
		output += "\n";
		for (int i=0; i<this.targetFrequencies[1].length; i++) {
			output += nts.charAt(i);
			for (int j=0; j<this.targetFrequencies[1][i].length; j++)
				output += "\t"+Methods.pad(String.valueOf(this.targetFrequencies[1][i][j]),pad);
			output += "\n";
		}
		output += "# This is the conditional dinucleotide target frequencies\n#";
		for (int i=0; i<nts.length(); i++)
			output += "\t"+Methods.pad(nts.substring(i,i+1),pad);
		output += "\n";
		for (int i=0; i<this.targetFrequencies[2].length; i++) {
			output += nts.charAt(i);
			for (int j=0; j<this.targetFrequencies[2][i].length; j++)
				output += "\t"+Methods.pad(String.valueOf(this.targetFrequencies[2][i][j]),pad);
			output += "\n";
		}
		output += "# This is the M0-model scores\n";
		for (int i=0; i<this.M0ScoreMatrix.length; i++)
			output += nts.charAt(i)+"\t"+String.valueOf((int) this.M0ScoreMatrix[i][0])+"\n";
		output += "# This is the M0-model K parameter\n";
			output += String.valueOf(this.K0)+"\n";
		output += "# This is the M0-model entropy\n";
			output += String.valueOf(this.H0)+"\n";
		output += "# This is the M0-model lambda parameter\n";
			output += String.valueOf(this.L0)+"\n";
		output += "# This is the M1-model scores\n#";
		for (int i=0; i<nts.length(); i++)
			output += "\t"+Methods.pad(nts.substring(i,i+1),pad);
		output += "\n";
		for (int i=0; i<this.M1ScoreMatrix.length; i++) {
			output += nts.charAt(i);
			for (int j=0; j<this.M1ScoreMatrix[i].length; j++)
				output += "\t"+Methods.pad(String.valueOf((int) this.M1ScoreMatrix[i][j]),pad);
			output += "\n";
		}
		output += "# This is the M1-model K parameter\n";
			output += String.valueOf(this.K1)+"\n";
		output += "# This is the M1-model entropy\n";
			output += String.valueOf(this.H1)+"\n";
		output += "# This is the M1-model lambda parameter\n";
			output += String.valueOf(this.L1)+"\n";
		return output;
	}

}

