/**
*	SIGRS - Identifying genomic regions of contrasting composition using a partial sum process
*	Copyright (C) 2008 Pontus Larsson
*	 
*	This file is part of SIGRS.
*	  
*	SIGRS is free software: you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation, either version 3 of the License, or
*	(at your option) any later version.
*
*	SIGRS is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with SIGRS. If not, see <http://www.gnu.org/licenses/>.
*/ 

package SIGRS;

import java.util.Vector;
import Jama.Matrix;

      
/**
*	SIGRS is a collection of routines used in searching for regions of contrasting composition (CCRs) in sequence files using a partial sum process.
*	Significance of segments is evaluated using Karlin-Altschul statistics and specifically an extension by Karlin-Dembo allowing
*	for nucleotides to have a Markov-dependence (see e.g. <a href="http://www.pnas.org/cgi/reprint/90/12/5873" target="_blank">Karlin & Altschul (1993)</a> and
*	<a href="http://www.jstor.org/view/00018678/ap050087/05a00070/0" target="_blank">Karlin & Dembo (1992)</a>
*	<P>
*	The routines are provided as is and no guarantee regarding stability etc. is given so use at your own risk!
*	<P>
*	<P>
*	See publication <b>Larsson, P., Hinas, A., Ardell, D.H., Kirsebom, L.A., Virtanen, A. and Sderbom, F.</b> <i>De novo search for non-coding RNA genes in the AT-rich genome of Dictyostelium discoideum: performance of
*	Markov-dependent genome feature scoring</i>
*	<P>
*	Questions and comments can be directed to <a href="mailto:Pontus.Larsson@icm.uu.se">Pontus.Larsson@icm.uu.se</a>
*	@author Pontus Larsson
*
*/

public class KDTreeStructure {

	private int depth;
	private int nodes;
	private int leaves;
	private KDTreeNode rootNode;

	public KDTreeStructure() 	{
		this.depth = 0;
		this.nodes = 0;
		this.leaves = 0;
		this.rootNode = null;
	}


	public void buildTree(int[] steps, int maxValue, KDTreeStructure other) {
		if (other == null)
			this.rootNode.buildTree(steps,maxValue,this.rootNode);
		else
			this.rootNode.buildTree(steps,maxValue,other.getRoot());
	}

	public Matrix getPath(int destination, int target, int maxValue) {
		if (this.rootNode.getLabel() == target)
			return this.rootNode.getPath(destination,maxValue);
		return null;
	}

	public KDTreeNode getRoot() {return this.rootNode;}

	public void reset(Vector visited) {this.rootNode.reset(visited);}

	public void setRoot(KDTreeNode rootNode) {this.rootNode = rootNode;}

	public void updateTree(int[] labels, Matrix[] q) {this.rootNode.updateTree(labels,q,this.rootNode.getLabel());}
}

