/**
*	SIGRS - Identifying genomic regions of contrasting composition using a partial sum process
*	Copyright (C) 2008 Pontus Larsson
*	 
*	This file is part of SIGRS.
*	  
*	SIGRS is free software: you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation, either version 3 of the License, or
*	(at your option) any later version.
*
*	SIGRS is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with SIGRS. If not, see <http://www.gnu.org/licenses/>.
*/  

package SIGRS;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;

import java.util.Date;
     
/**
*	SIGRS is a collection of routines used in searching for regions of contrasting composition (CCRs) in sequence files using a partial sum process.
*	Significance of segments is evaluated using Karlin-Altschul statistics and specifically an extension by Karlin-Dembo allowing
*	for nucleotides to have a Markov-dependence (see e.g. <a href="http://www.pnas.org/cgi/reprint/90/12/5873" target="_blank">Karlin & Altschul (1993)</a> and
*	<a href="http://www.jstor.org/view/00018678/ap050087/05a00070/0" target="_blank">Karlin & Dembo (1992)</a>
*	<P>
*	The routines are provided as is and no guarantee regarding stability etc. is given so use at your own risk!
*	<P>
*	<P>
*	See publication <b>Larsson, P., Hinas, A., Ardell, D.H., Kirsebom, L.A., Virtanen, A. and Sderbom, F.</b> <i>De novo search for non-coding RNA genes in the AT-rich genome of Dictyostelium discoideum: performance of
*	Markov-dependent genome feature scoring</i>
*	<P>
*	Questions and comments can be directed to <a href="mailto:Pontus.Larsson@icm.uu.se">Pontus.Larsson@icm.uu.se</a>
*	@author Pontus Larsson
*
*/

public class Methods {

	public static String nucleotideCode = new String("NACGTX");
	public static char[] nucleotideCharCode = new char[] {78,65,67,71,84,88};

	/**
	*	Adds an element to the end of an array
	*	@param n The new element to add
	*	@param arr The array to add the new element to
	*	@return A new array with the new element added to the input array
	*/
	public static byte[] addElement(byte n, byte[] arr) {
		if (arr == null || arr.length == 0)
			return new byte[] {n};
		int sz = arr.length;
		byte[] newArr = new byte[sz+1];
		for (int i=0; i<sz; i++)
			newArr[i] = arr[i];
		newArr[sz] = n;
		return newArr;
	}
	/**
	*	Adds an element to the end of an array
	*	@param n The new element to add
	*	@param arr The array to add the new element to
	*	@return A new array with the new element added to the input array
	*/
	public static byte[][] addElement(byte[] n, byte[][] arr) {
		if (arr == null || arr.length == 0)
			return new byte[][] {n};
		int sz = arr.length;
		byte[][] newArr = new byte[sz+1][];
		for (int i=0; i<sz; i++)
			newArr[i] = arr[i];
		newArr[sz] = n;
		return newArr;
	}
	/**
	*	Adds an element to the end of an array
	*	@param n The new element to add
	*	@param arr The array to add the new element to
	*	@return A new array with the new element added to the input array
	*/
	public static double[][] addElement(double[] n, double[][] arr) {
		if (arr == null || arr.length == 0)
			return new double[][] {n};
		int sz = arr.length;
		double[][] newArr = new double[sz+1][];
		for (int i=0; i<sz; i++)
			newArr[i] = arr[i];
		newArr[sz] = n;
		return newArr;
	}
	/**
	*	Adds an element to the end of an array
	*	@param n The new element to add
	*	@param arr The array to add the new element to
	*	@return A new array with the new element added to the input array
	*/
	public static int[] addElement(int n, int[] arr) {
		if (arr == null || arr.length == 0)
			return new int[] {n};
		int sz = arr.length;
		int[] newArr = new int[sz+1];
		for (int i=0; i<sz; i++)
			newArr[i] = arr[i];
		newArr[sz] = n;
		return newArr;
	}
	/**
	*	Adds an element to the end of an array
	*	@param n The new element to add
	*	@param arr The array to add the new element to
	*	@return A new array with the new element added to the input array
	*/
	public static File[] addElement(File nO, File[] arr) {
		if (arr == null || arr.length == 0)
			return new File[] {nO};
		int sz = arr.length;
		File[] newArr = new File[sz+1];
		for (int i=0; i<sz; i++)
			newArr[i] = arr[i];
		newArr[sz] = nO;
		return newArr;
	}
	/**
	*	Adds an element to the end of an array
	*	@param n The new element to add
	*	@param arr The array to add the new element to
	*	@return A new array with the new element added to the input array
	*/
	public static Object[] addElement(Object nO, Object[] arr) {
		if (arr == null || arr.length == 0)
			return new Object[] {nO};
		int sz = arr.length;
		Object[] newArr = new Object[sz+1];
		for (int i=0; i<sz; i++)
			newArr[i] = arr[i];
		newArr[sz] = nO;
		return newArr;
	}
	/**
	*	Adds an element to the end of an array
	*	@param n The new element to add
	*	@param arr The array to add the new element to
	*	@return A new array with the new element added to the input array
	*/
	public static String[] addElement(String nO, String[] arr) {
		if (arr == null || arr.length == 0)
			return new String[] {nO};
		int sz = arr.length;
		String[] newArr = new String[sz+1];
		for (int i=0; i<sz; i++)
			newArr[i] = arr[i];
		newArr[sz] = nO;
		return newArr;
	}

	/**
	*	Appends an array to the end of another array
	*	@param nE The array to append
	*	@param src The new array gets appended to the end of this one
	*	@return An array with the first array appended to the end of the second
	*/
	public static byte[] append(byte[] nE, byte[] src)	{return append(0,nE.length,nE,src);}
	public static byte[] append(int start, int stop, byte[] nE, byte[] src) {
		if (src == null)
			src = new byte[0];
		int sLt = src.length;
		int nLt = stop-start;
		byte[] newArr = new byte[sLt+nLt];
		for (int i=0; i<sLt; i++)
			newArr[i] = src[i];
		for (int i=start; i<stop; i++)
			newArr[sLt+i] = nE[i];
		return newArr;
	}
	/**
	*	Appends an array to the end of another array
	*	@param nE The array to append
	*	@param src The new array gets appended to the end of this one
	*	@return An array with the first array appended to the end of the second
	*/
	public static char[] append(char[] nE, char[] src)	{return append(0,nE.length,nE,src);}
	public static char[] append(int start, int stop, char[] nE, char[] src) {
		if (src == null)
			src = new char[0];
		int sLt = src.length;
		int nLt = stop-start;
		char[] newArr = new char[sLt+nLt];
		for (int i=0; i<sLt; i++)
			newArr[i] = src[i];
		for (int i=start; i<stop; i++)
			newArr[sLt+i] = nE[i];
		return newArr;
	}
	/**
	*	Appends an array to the end of another array
	*	@param nE The array to append
	*	@param src The new array gets appended to the end of this one
	*	@return An array with the first array appended to the end of the second
	*/
	public static double[] append(double[] nE, double[] src) {return append(0,nE.length,nE,src);}
	private static double[] append(int start, int stop, double[] nE, double[] src) {
		if (src == null)
			src = new double[0];
		int sLt = src.length;
		int nLt = stop-start;
		double[] newArr = new double[sLt+nLt];
		for (int i=0; i<sLt; i++)
			newArr[i] = src[i];
		for (int i=start; i<stop; i++)
			newArr[sLt+i] = nE[i];
		return newArr;
	}
	/**
	*	Appends an array to the end of another array
	*	@param nE The array to append
	*	@param src The new array gets appended to the end of this one
	*	@return An array with the first array appended to the end of the second
	*/
	public static double[][] append(double[][] nE, double[][] src)	{return append(0,nE.length,nE,src);}
	private static double[][] append(int start, int stop, double[][] nE, double[][] src) {
		if (src == null)
			src = new double[0][];
		int sLt = src.length;
		int nLt = stop-start;
		double[][] newArr = new double[sLt+nLt][];
		for (int i=0; i<sLt; i++)
			newArr[i] = src[i];
		for (int i=start; i<stop; i++)
			newArr[sLt+i] = nE[i];
		return newArr;
	}

	/**
	*	Appends the contents of one file to the other
	*	@param f1 The file which contents will be appended
	*	@param f2 The file that will be appended
	*/
	public static void append(File f1, File f2) throws Exception {
		BufferedReader br = new BufferedReader(new FileReader(f1));
		BufferedWriter bw = new BufferedWriter(new FileWriter(f2,true));
		String buff;
		while ((buff = br.readLine()) != null) {
			bw.write(buff);
			bw.newLine();
		}
		br.close();
		bw.close();
	}

	/**
	*	Appends an array to the end of another array
	*	@param nE The array to append
	*	@param src The new array gets appended to the end of this one
	*	@return An array with the first array appended to the end of the second
	*/
	public static int[] append(int[] nE, int[] src)	{return append(0,nE.length,nE,src);}
	private static int[] append(int start, int stop, int[] nE, int[] src)
	{
		if (src == null)
			src = new int[0];
		int sLt = src.length;
		int nLt = stop-start;
		int[] newArr = new int[sLt+nLt];
		for (int i=0; i<sLt; i++)
			newArr[i] = src[i];
		for (int i=start; i<stop; i++)
			newArr[sLt+i] = nE[i];
		return newArr;
	}
	/**
	*	Appends an array to the end of another array
	*	@param nE The array to append
	*	@param src The new array gets appended to the end of this one
	*	@return An array with the first array appended to the end of the second
	*/
	public static String[] append(String[] nE, String[] src) {
		if (src == null || src.length == 0)
			return nE;
		int sLt = src.length;
		int nLt = nE.length;
		String[] newArr = new String[sLt+nLt];
		for (int i=0; i<sLt; i++)
			newArr[i] = src[i];
		for (int i=0; i<nLt; i++)
			newArr[sLt+i] = nE[i];
		return newArr;
	}
	/**
	*	Appends an array to the end of another array
	*	@param nE The array to append
	*	@param src The new array gets appended to the end of this one
	*	@return An array with the first array appended to the end of the second
	*/
	public static String[][] append(String[][] nE, String[][] src) {
		if (src == null || src.length == 0)
			return nE;
		int sLt = src.length;
		int nLt = nE.length;
		String[][] newArr = new String[sLt+nLt][];
		for (int i=0; i<sLt; i++)
			newArr[i] = src[i];
		for (int i=0; i<nLt; i++)
			newArr[sLt+i] = nE[i];
		return newArr;
	}

	public static File concatenateFile(File src, File concat) throws Exception {
		src.createNewFile();
		BufferedWriter bw = new BufferedWriter(new FileWriter(src,true));
		BufferedReader br = new BufferedReader(new FileReader(concat));
		String buff;
		while ((buff = br.readLine()) != null) {
			bw.write(buff);
			bw.newLine();
		}
		bw.close();
		br.close();
		return src;
	}

	/**
	*	Counts the number of each nucleotide in a sequence
	*	Only A,C,G and T is counted. Any other nucleotide is counted as N
	*	@return 1x6array with nucleotide counts: [0] -> N, [1] -> A, [2] -> C, [3] -> G, [4] -> T, [5] -> N
	*/
	public static int[] countMonoNucleotides(byte[] seq) {
		int[] counts = new int[6];
		for (int i=0; i<seq.length; i++)
			counts[Math.max(0,Math.min(5,seq[i]))]++;

		return counts;
	}
	/**
	*	Counts the number of each nucleotide in a sequence
	*	Only A,C,G and T is counted. Any other nucleotide is counted as N
	*	@return 1x6array with nucleotide counts: [0] -> N, [1] -> A, [2] -> C, [3] -> G, [4] -> T, [5] -> N
	*/
	public static int[] countMonoNucleotides(File seqFile) throws Exception {
		// Buffersize (50% of available memory or 64 MB, whichever is the least)
		int chunkSize = Math.min((int) (0.5 * ((int) Runtime.getRuntime().totalMemory())),64*1024*1024);

		BufferedReader br = new BufferedReader(new FileReader(seqFile));

		int rd;
		char[] cBuff;
		char last = 62;
		String name;
		int j = -1;
		int i;
		int[] counts = new int[6];
		while (true) {
			cBuff = new char[chunkSize];
			rd = br.read(cBuff,0,chunkSize);
			if (rd < 0)
				break;
			i = 0;
			while (i < rd) {
				// Make upper case
				if (cBuff[i] >= 97 && cBuff[i] <= 122)
					cBuff[i] -= 32;
				// Change U to T
				if (cBuff[i] == 85)
					cBuff[i] = 84;
				// A valid letter?
				if (cBuff[i] != 62 && cBuff[i] != 10 && cBuff[i] != 13) {
					// Code
					j = Math.max(nucleotideCode.indexOf(cBuff[i]),0);
					counts[j]++;
				}
				else if (cBuff[i] == 62) {
					i++;
					while (i < rd && cBuff[i] != 10 && cBuff[i] != 13)
						i++;
					while (i < rd && (cBuff[i] == 10 || cBuff[i] == 13))
						i++;
					i--;
				}
				i++;
			}
		}
		br.close();

		return counts;
	}
	/**
	*	Counts the number of each dinucleotide in a sequence
	*	Only A,C,G and T is counted. Any other nucleotide is counted as N
	*	@return 6x6-array with dinucleotide counts: [0] -> N, [1] -> A, [2] -> C, [3] -> G, [4] -> T, [5] -> N
	*/
	public static int[][] countDiNucleotides(byte[] seq) {
		int[][] counts = new int[6][6];
		for (int i=1; i<seq.length; i++)
			counts[Math.max(0,Math.min(5,seq[i-1]))][Math.max(0,Math.min(5,seq[i]))]++;
		return counts;
	}
	/**
	*	Counts the number of each dinucleotide in a sequence
	*	Only A,C,G and T is counted. Any other nucleotide is counted as N
	*	@return 6x6-array with dinucleotide counts: [0] -> N, [1] -> A, [2] -> C, [3] -> G, [4] -> T, [5] -> X
	*/
	public static int[][] countDiNucleotides(File seqFile) throws Exception {
		// Buffersize (50% of available memory or 64 MB, whichever is the least)
		int chunkSize = Math.min((int) (0.5 * ((int) Runtime.getRuntime().totalMemory())),64*1024*1024);

		BufferedReader br = new BufferedReader(new FileReader(seqFile));

		int rd;
		char[] cBuff;
		char last = 62;
		String name;
		int j;
		int k;
		int[][] counts = new int[6][6];
		while (true) {
			cBuff = new char[chunkSize];
			// Save the last character read (or a '>' if it's the start of the file)
			cBuff[0] = last;
			rd = br.read(cBuff,1,chunkSize-1);
			if (rd < 0) {
				// The last transition should be counted as going to a masked segment
				counts[Math.max(nucleotideCode.indexOf(last),0)][5]++;
				break;
			}
			int i = 1;
			// Add 1 to rd since we put an extra character at the beginning
			rd++;
			while (i < rd) {
				// Change to upper case
				if (cBuff[i] >= 97 && cBuff[i] <= 122)
					cBuff[i] -= 32;
				// Change U to T
				if (cBuff[i] == 85)
					cBuff[i] = 84;
				// No special character ('\n', '\r' or '>') is encountered. The previous char is always ok
				if (cBuff[i] != 10 && cBuff[i] != 13 && cBuff[i] != 62) {
					j = Math.max(nucleotideCode.indexOf(cBuff[i-1]),0);
					k = Math.max(nucleotideCode.indexOf(cBuff[i]),0);
					counts[j][k]++;

					// Save the last character that was ok
					last = cBuff[i];
				}
				// A line break is encountered, 10 = '\n', 13 = '\r'
				else if (cBuff[i] == 10 || cBuff[i] == 13) {
					j = Math.max(nucleotideCode.indexOf(cBuff[i-1]),0);
					// Save the last character that was ok
					last = cBuff[i-1];

					// Jump over the line break
					i = skipLineBreak(cBuff,i,rd);
					if (i == rd)
						break;
					// If a new sequence begins it shouldn't be counted as a dinucleotide
					// and the last dinucleotide is counted as making a transition into a masked segment
					if (cBuff[i] == 62) {
						k = 5;
						counts[j][k]++;
						j = 5;
						last = nucleotideCharCode[j];
						i = skipIdLine(cBuff,i,rd);
					}
					if (i == rd)
						break;

					k = Math.max(nucleotideCode.indexOf(cBuff[i]),0);
					counts[j][k]++;

					// Save the last character that was ok
					last = cBuff[i];
				}
				// A new sequence begins, 62 = '>'
				else if (cBuff[i] == 62) {
					i = skipIdLine(cBuff,i,rd);
					// Save the last character that was ok
					if (i < rd)
						last = cBuff[i];
				}
				i++;
			}
		}
		br.close();

		return counts;
	}

	public static int skipLineBreak(char[] cBuff, int i, int limit) {
		while (i < limit && (cBuff[i] == 10 || cBuff[i] == 13))
			i++;
		return i;
	}
	public static int skipIdLine(char[] cBuff, int i, int limit) {
		// Skip the id line
		while (i < limit && cBuff[i] != 10 && cBuff[i] != 13)
			i++;
		i = skipLineBreak(cBuff,i,limit);
		return i;
	}

	public static String decode(byte[] byteSeq) {
		char[] seq = new char[byteSeq.length];
		for (int i=0; i<byteSeq.length; i++)
			seq[i] = nucleotideCharCode[byteSeq[i]];
		return new String(seq);
	}

	/**
	*	Encodes a nucleotide sequence into bytes where
	*	1=A, 2=C, 3=G, 4=T, 5=X (Masked), any other nucleotide is encoded as 0=N
	*	@param seq The input sequence
	*	@return The input sequence encoded to a byte array
	*/
	public static byte[] encode(String seq)
	{
		seq = seq.toUpperCase();
		seq = seq.replace('U','T');
		char[] s = seq.toCharArray();
		byte[] encoded = new byte[s.length];
		for (int i=0; i<s.length; i++)
			encoded[i] = (byte) Math.max(nucleotideCode.indexOf(s[i]),0);
		return encoded;
	}

	/**
	*	Returns the desired column from a matrix array
	*	@param c Index of the column to return
	*	@param arr The array
	*	@return An array corresponding to the desired column from the input matrix
	*/
	public static double[] getColumn(int c, double[][] arr)	{
		int sz = arr.length;
		double[] nC = new double[sz];
		for (int i=0; i<sz; i++)
			if (arr[i].length >= c)
				nC[i] = arr[i][c];
		return nC;
	}
	/**
	*	Returns the desired column from a matrix array
	*	@param c Index of the column to return
	*	@param arr The array
	*	@return An array corresponding to the desired column from the input matrix
	*/
	public static String[] getColumn(int c, String[][] arr) {
		int sz = arr.length;
		String[] nC = new String[sz];
		for (int i=0; i<sz; i++)
			if (arr[i].length >= c)
				nC[i] = arr[i][c];
		return nC;
	}

	/**
	*	Finds the index of an element within an array
	*	@param obj The element to search for
	*	@param arr The array to search within
	*	@return The index of the first occurance of the element within the array or -1 if the element was not found
	*/
	public static int indexOf(int obj, int[] arr) {
		if (arr == null || arr.length == 0)
			return -1;
		int sz = arr.length;
		for (int i=0; i<sz; i++)
			if (arr[i] == obj)
				return i;
		return -1;
	}
	/**
	*	Finds the index of an element within an array
	*	@param obj The element to search for
	*	@param arr The array to search within
	*	@return The index of the first occurance of the element within the array or -1 if the element was not found
	*/
	public static int indexOf(String obj, String[] arr) {
		if (arr == null || arr.length == 0)
			return -1;
		int sz = arr.length;
		for (int i=0; i<sz; i++)
			if (arr[i] != null && arr[i].compareTo(obj) == 0)
				return i;
		return -1;
	}

	/**
	*	Calculates the logarithm in an arbitrary base
	*	@param x Value to take logarithm of
	*	@param N Base of logarithm
	*	@return The logarithm of x in base N
	*/
	public static double logN(double x, double N) {return Math.log(x)/Math.log(N);}

	/**
	*	Finds the greatest element in the input array
	*	@param arr The input array to search
	*	@return The maximum value in the input array
	*/
	public static double max(double[] arr) {
		int sz = arr.length;
		if (sz == 0)
			return 0;
		double max = arr[0];
		for (int i=1; i<sz; i++)
			if (arr[i] > max)
				max = arr[i];
		return max;
	}
	/**
	*	Finds the greatest element in the input array
	*	@param arr The input array to search
	*	@return The maximum value in the input array
	*/
	public static double max(double[][] arr) {
		int sz = arr.length;
		double max = max(arr[0]), t;
		for (int i=1; i<sz; i++)
			if ((t = max(arr[i])) > max)
				max = t;
		return max;
	}

	/**
	*	Finds the minimum element in the input array
	*	@param arr The input array to search
	*	@return The minimum value in the input array
	*/
	public static double min(double[] arr) {return -1.*max(vectorMultiply(arr,-1.));}
	/**
	*	Finds the minimum element in the input array
	*	@param arr The input array to search
	*	@return The minimum value in the input array
	*/
	public static double min(double[][] arr) {return -1.*max(vectorMultiply(arr,-1.));}

	/**
	*	Creates a new array of specified length and with all elements set to a specified value
	*	@param len The length of the array
	*	@param fill The value all elements will have
	*	@return A new double array of specified length and containing specified values
	*/
	public static double[] newDoubleArray(int len, double fill) {
		return Methods.vectorAdd(new double[len],fill);
	}

	/**
	*	Pads a string with whitespaces to a specified length. If input string is longer it is
	*	truncated instead
	*	@param str The string to be padded
	*	@param padLength The desired length of the string
	*	@return The input string with whitespaces added at the end so the total length is equal to padLength. If the input string was longer it is truncated instead.
	*/
	public static String pad(String str, int padLength) {
		char padChar = ' ';
		int lt = str.length();
		if (padLength <= lt)
			return str.substring(0,padLength);
		int tLt = padLength-lt;
		char[] ch = new char[tLt];
		for (int i=0; i<tLt; i++)
			ch[i] = padChar;
		return str + new String(ch);
	}

	/**
	*	Parses a file in FASTA format and returns an array with the
	*	identifiers and sequences. For files with few long sequences, use parseBigFasta() instead.
	*	@param fastaFile Sequence file in FASTA format
	*	@return Nx2-array holding the identifiers and sequences. [i][0] -> Identifier, [i][1] -> Sequence
	*/
	public static String[][] parseFasta(File fastaFile) throws Exception {
		BufferedReader br = new BufferedReader(new FileReader(fastaFile));
		int LINES = 10000;
		String[][] seqs = new String[LINES][2];
		char[] tSeq;
		int nextSeq = 0;
		String buff = br.readLine();
		while (buff != null) {
			if (buff.length() > 0 && buff.charAt(0) == '>')	{
				seqs[nextSeq][0] = buff.substring(1);
				tSeq = new char[0];
				while ((buff = br.readLine()) != null && buff.length() > 0 && buff.charAt(0) != '>')
					tSeq = append(buff.toCharArray(),tSeq);
				seqs[nextSeq][1] = new String(tSeq);
				nextSeq++;
				if (nextSeq == LINES) {
					seqs = append(new String[LINES][2],seqs);
					LINES += LINES;
				}
			}
			else
				buff = br.readLine();
		}
		br.close();
		return subarray(0,nextSeq,seqs);
	}

	public static String[][] parseBigFasta(File f) throws Exception	{
		int INITIAL_CAPACITY = 100;
		int INCREMENT = 50;
		int nextEntry = -1;
		String[][] entries = new String[INITIAL_CAPACITY][2];
		BufferedReader br = new BufferedReader(new FileReader(f));
		// Buffersize (50% of available memory or 64 MB, whichever is the least)
		int chunkSize = Math.min((int) (0.5 * ((int) Runtime.getRuntime().totalMemory())),64*1024*1024);

		int rd;
		char[] cBuff;
		char[] entry = new char[chunkSize];
		String name;
		int j = 0;
		while (true) {
			cBuff = new char[chunkSize];
			rd = br.read(cBuff,0,chunkSize);
			if (rd < 0) {
				if (nextEntry >= 0)
					entries[nextEntry][1] = new String(subarray(0,j,entry));
				nextEntry++;
				break;
			}
			cBuff = subarray(0,rd,cBuff);
			for (int i=0; i<rd; i++) {
				if (j >= entry.length)
					entry = append(new char[chunkSize],entry);
				if (cBuff[i] != 10 && cBuff[i] != 13 && cBuff[i] != 62 && cBuff[i] != 32)
					entry[j] = cBuff[i];
				else if (cBuff[i] == 10 || cBuff[i] == 13 || cBuff[i] == 32)
					j--;
				else {
					if (nextEntry >= 0)
						entries[nextEntry][1] = new String(subarray(0,j,entry));
					entry = new char[chunkSize];
					j = -1;
					i++;
					name = new String();
					while (i < rd && cBuff[i] != 10 && cBuff[i] != 13) {
						name += cBuff[i];
						i++;
					}
					nextEntry++;
					if (nextEntry == entries.length)
						entries = append(new String[INCREMENT][2],entries);
					entries[nextEntry][0] = name;
				}
				j++;
			}
		}
		br.close();
		return subarray(0,nextEntry,entries);
	}

	/**
	*	Sorts an array in ascending order
	*	@param src The array to sort
	*	@return A new array with the input array elements sorted in ascending order
	*/
	public static int[] quickSort(int[] src) {
		int sz = src.length, nL=0,nR=0;
		if (sz == 0 || sz == 1)
			return src;
		int k = src[sz/2];
		int[] left = new int[sz-1], right = new int[sz-1];
		for (int i=0; i<sz; i++) {
			if (i != sz/2) {
				if (src[i] <= k) {
					left[nL] = src[i];
					nL++;
				}
				else {
					right[nR] = src[i];
					nR++;
				}
			}
		}
		return append(append(quickSort(subarray(0,nR,right)),new int[] {k}),quickSort(subarray(0,nL,left)));
	}

	/**
	*	Reads the entire content of a file into a string
	*	@param f The file to read
	*	@return The contents of the input file in a string
	*/
	public static final String getFileContents(File f) throws Exception {
		char[] buff = new char[(int) f.length()];
		FileReader fr = new FileReader(f);
		fr.read(buff,0,buff.length);
		return new String(buff);
	}

	/**
	*	Removes an element from an array and shifts the remaining
	*	elements to the left
	*	@param n The index of the element to be removed
	*	@param arr The array to remove the element from
	*	@return An array with the element removed
	*/
	public static byte[][] removeElementAt(int n, byte[][] arr) {
		if (arr == null || n < 0 || n >= arr.length)
			return arr;
		int sz = arr.length;
		byte[][] newArr = new byte[sz-1][];
		for (int i=0; i<n; i++)
			newArr[i] = arr[i];
		for (int i=n+1; i<sz; i++)
			newArr[i-1] = arr[i];
		return newArr;
	}
	/**
	*	Reverse the order of the elements in an array
	*	@param arr The array to reverse
	*	@return A copy of the array with the elements in reverse order
	*/
	public static int[] reverse(int[] arr) {return reverse(arr,0,arr.length);}
	private static int[] reverse(int[] seq, int start, int stop)
	{
		int sz = stop-start;
		int[] newArr = new int[sz];
		for (int i=stop; i>start; i--)
			newArr[sz-i+start] = seq[i-1];
		return newArr;
	}

	/**
	*	Gets the reverse complement of a byte encoded sequence
	*	@param seq The sequence to reverse complement
	*	@return The reverse complement of the input sequence
	*/
	public static byte[] reverseComplement(byte[] seq) {
		int sz = seq.length;
		byte[] rC = new byte[sz];
		byte[] subst = new byte[] {0,4,3,2,1,5};
		for (int i=sz; i>0; i--)
			rC[i-1] = subst[Math.max(0,Math.min(5,seq[sz-i]))];
		return rC;
	}
	public static File reverseComplement(File seqFile, File revFile) throws Exception {

		BufferedReader br = new BufferedReader(new FileReader(seqFile));
		// Buffersize (50% of available memory or 64 MB, whichever is the least)
		int chunkSize = Math.min((int) (0.5 * ((int) Runtime.getRuntime().totalMemory())),64*1024*1024);

		BufferedWriter revCompW = new BufferedWriter(new FileWriter(revFile));
		BufferedWriter bw = null;
		String fileName = revFile.getParent() + File.separator + "temp_";

		int fileIndex = 0;
		String name = new String();
		int i = chunkSize;
		int j = -1;
		char[] dBuff = new char[0];;
		char[] cBuff;
		String buff;
		String nl = System.getProperty("line.separator");

		while ((buff = br.readLine()) != null) {
			cBuff = new String(buff+nl).toCharArray();
			// A new sequence begins
			if (cBuff[0] == 62) {
				// If we have been processing a sequence, write that sequence to the output file
				if (i < (dBuff.length-1)) {
					// Write the part in the buffer to the temp file
					bw.write(dBuff,i+1,dBuff.length-(i+1));
					bw.close();
					// Write all the temp files to the output file
					writeReverseComplement(fileName,fileIndex,name,revCompW);
					fileIndex = 0;
				}
				// Start new temp files and buffer
				fileIndex++;
				bw = new BufferedWriter(new FileWriter(new File(fileName+String.valueOf(fileIndex))));
				dBuff = new char[chunkSize];
				i = dBuff.length-1;
				name = buff.substring(1);
				buff = br.readLine();
				cBuff = buff.toCharArray();
			}
			// Fill the buffer from the end with the reverse complement
			for (j=0; j<cBuff.length && i >= 0; j++, i--)
				dBuff[i] = reverseComplement(cBuff[j]);

			// End of buffer?
			if (i < 0) {
				// Write the buffer to the temp file
				bw.write(dBuff,0,dBuff.length);
				bw.close();
				// Start a new temp file
				fileIndex++;
				bw = new BufferedWriter(new FileWriter(new File(fileName+String.valueOf(fileIndex))));
				// Start a new buffer
				dBuff = new char[chunkSize];
				i = dBuff.length-1;
				// Write the part that didn't fit into the previous buffer
				for (; j<cBuff.length && i >= 0; j++, i--)
					dBuff[i] = reverseComplement(cBuff[j]);
			}
		}
		br.close();
		// Input file ended, write the last sequence to output file
		bw.write(dBuff,i+1,dBuff.length-(i+1));
		bw.close();
		// Write all the temp files to the output file
		writeReverseComplement(fileName,fileIndex,name,revCompW);
		revCompW.close();

		return revFile;
	}
	private static void writeReverseComplement(String fileName, int fileIndex, String seqId, BufferedWriter bw) throws Exception {
		bw.write(">" + seqId);
		bw.newLine();

		File tempFile;
		BufferedReader br;
		String buff;

		for (int i=fileIndex; i>0; i--) {
			tempFile = new File(fileName+String.valueOf(i));
			br = new BufferedReader(new FileReader(tempFile));
			while ((buff = br.readLine()) != null) {
				if (buff.length() > 0) {
					bw.write(buff);
					bw.newLine();
				}
			}
			br.close();
			tempFile.delete();
		}
	}

	public static char reverseComplement(char c) {
		int j;
		char r = c;
		// Make upper case
		if (c >= 97 && c <= 122)
			c -= 32;
		// Change U to T
		if (c == 85)
			c = 84;
		// Take complement if nucleotide is A C G or T
		if (c >= 65 && c <= 90 && (j = nucleotideCode.indexOf(c)) >= 1 && j <= 4)
			r = nucleotideCharCode[5-j];
		return r;
	}

	/**
	*	Set a column of a matrix
	*	@param src The matrix to set the column in
	*	@param index The index (zero based) of the column to set
	*	@param col The values of the column to set. Must have the same length as the number of rows in src
	*	@return A new matrix
	*/
	public static double[][] setColumn(double[][] src, int index, double[] col) {
		if (col.length != src.length)
			return new double[0][];
		double[][] nArr = new double[src.length][];
		for (int i=0; i<src.length; i++) {
			nArr[i] = new double[src[i].length];
			for (int j=0; j<src[i].length; j++) {
				if (j != index)
					nArr[i][j] = src[i][j];
				else
					nArr[i][j] = col[i];
			}
		}
		return nArr;
	}

	/**
	*	Extracts a subsegment of an array
	*	@param start The start index (inclusive)
	*	@param stop The stop index (exclusive)
	*	@param arr The input array
	*	@return A subsegment of the input array
	*/
	public static char[] subarray(int start, int stop, char[] src) {
		if (start >= src.length || stop > src.length)
			return new char[0];
		char[] newArr = new char[stop-start];
		for (int i=start; i<stop; i++)
			newArr[i-start] = src[i];
		return newArr;
	}
	/**
	*	Extracts a subsegment of an array
	*	@param start The start index (inclusive)
	*	@param stop The stop index (exclusive)
	*	@param arr The input array
	*	@return A subsegment of the input array
	*/
	public static double[] subarray(int start, int stop, double[] src) {
		if (start >= src.length || stop > src.length)
			return new double[0];
		double[] newArr = new double[stop-start];
		for (int i=start; i<stop; i++)
			newArr[i-start] = src[i];
		return newArr;
	}
	/**
	*	Extracts a subsegment of an array
	*	@param start The start index (inclusive)
	*	@param stop The stop index (exclusive)
	*	@param arr The input array
	*	@return A subsegment of the input array
	*/
	public static double[][] subarray(int start, int stop, double[][] arr) {
		if (start >= arr.length || stop > arr.length)
			return new double[0][0];
		double[][] newArr = new double[stop-start][];
		for (int i=start; i<stop; i++)
			newArr[i-start] = arr[i];
		return newArr;
	}
	/**
	*	Extracts the last subsegment of an array
	*	@param start The start index (inclusive)
	*	@param arr The input array
	*	@return A subsegment of the input array starting from the supplied start index and continuing up to the end
	*/
	public static int[] subarray(int start, int[] src) {return subarray(start,src.length,src);}
	/**
	*	Extracts a subsegment of an array
	*	@param start The start index (inclusive)
	*	@param stop The stop index (exclusive)
	*	@param arr The input array
	*	@return A subsegment of the input array
	*/
	public static int[] subarray(int start, int stop, int[] src) {
		if (start >= src.length || stop > src.length)
			return new int[0];
		int[] newArr = new int[stop-start];
		for (int i=start; i<stop; i++)
			newArr[i-start] = src[i];
		return newArr;
	}

	/**
	*	Extracts a subsegment of an array
	*	@param start The start index (inclusive)
	*	@param stop The stop index (exclusive)
	*	@param arr The input array
	*	@return A subsegment of the input array
	*/
	public static String[][] subarray(int start, String[][] src) {return subarray(start,src.length,src);}
	public static String[][] subarray(int start, int stop, String[][] src) {
		if (start >= src.length || stop > src.length)
			return new String[0][0];
		String[][] newArr = new String[stop-start][];
		for (int i=start; i<stop; i++)
			newArr[i-start] = src[i];
		return newArr;
	}
	/**
	*	Sums the element in an array
	*	@param arr The array whose elements to sum
	*	@return	The sum of the elements
	*/
	public static double sum(double[] arr)
	{
		int sz = arr.length;
		double sum = 0;
		for (int i=0; i<sz; i++)
			sum += arr[i];
		return sum;
	}
	/**
	*	Sums the element in an array
	*	@param arr The array whose elements to sum
	*	@return	The sum of the elements
	*/
	public static int sum(int[] arr)
	{
		int sz = arr.length;
		int sum = 0;
		for (int i=0; i<sz; i++)
			sum += arr[i];
		return sum;
	}

	/**
	*	Adds a scalar to an array
	*	@param arr Array to add to
	*	@param f The scalar to add to the input array
	*	@return A new array
	*/
	public static double[] vectorAdd(double[] arr, double f) {
		int len = arr.length;
		double[] nV = new double[len];
		for (int i=0; i<len; i++)
			nV[i] = arr[i] + f;
		return nV;
	}

	/**
	*	Adds to arrays together
	*	@param v1 Vector 1
	*	@param v2 Vector 2
	*	@return A new vector where the elements are the sum of the corresponding elements of v1 and v2
	*/
	public static int[] vectorAdd(int[] v1, int[] v2) {
		int sz = Math.min(v1.length,v2.length);
		int sSz = Math.max(v1.length,v2.length);
		int[] nV = new int[sSz];
		for (int i=0; i<sz; i++)
			nV[i] = v1[i] + v2[i];
		if (v1.length > sz)
			for (int i=sz; i<sSz; i++)
				nV[i] = v1[i];
		else if (v2.length > sz)
			for (int i=sz; i<sSz; i++)
				nV[i] = v2[i];
		return nV;
	}
	/**
	*	Adds to arrays together
	*	@param v1 Vector 1
	*	@param v2 Vector 2
	*	@return A new vector where the elements are the sum of the corresponding elements of v1 and v2
	*/
	public static int[][] vectorAdd(int[][] v1, int[][] v2)	{
		int sz = Math.min(v1.length,v2.length);
		int sSz = Math.max(v1.length,v2.length);
		int[][] nV = new int[sSz][];
		for (int i=0; i<sz; i++)
			nV[i] = vectorAdd(v1[i],v2[i]);
		if (v1.length > sz)
			for (int i=sz; i<v1.length; i++)
				nV[i] = v1[i];
		else if (v2.length > sz)
			for (int i=sz; i<v2.length; i++)
				nV[i] = v2[i];
		return nV;
	}
	/**
	*	Divides an array by a scalar
	*	@param arr Array to scale
	*	@param f The scalar to scale the input array by
	*	@return The input array scaled by f
	*/
	public static double[] vectorDivide(double[] arr, double f) {
		if (f == 0)
			return arr;
		f = 1.0/f;
		return vectorMultiply(arr,f);
	}
	/**
	*	Divides an array by a scalar
	*	@param arr Array to scale
	*	@param f The scalar to scale the input array by
	*	@return The input array scaled by f
	*/
	public static double[][] vectorDivide(double[][] arr, double f)	{
		int sz = arr.length;
		double[][] nV = new double[sz][];
		for (int i=0; i<sz; i++)
			nV[i] = vectorDivide(arr[i],f);
		return nV;
	}
	/**
	*	Calculates the inner product of two vectors
	*	@param v1 Vector 1
	*	@param v2 Vector 2
	*	@return The inner product of v1 and v2
	*/
	public static double vectorInnerProduct(double[] v1, double[] v2) {return sum(vectorMultiply(v1,v2));}
	/**
	*	Multiplies an array by a scalar
	*	@param arr Array to scale
	*	@param f The scalar to scale the input array by
	*	@return The input array scaled by f
	*/
	public static double[] vectorMultiply(double[] arr, double f) {
		int sz = arr.length;
		double[] nV = new double[sz];
		for (int i=0; i<sz; i++)
			nV[i] = arr[i] * f;
		return nV;
	}
	/**
	*	Multiplies the values of two arrays
	*	@param v1 The first array
	*	@param v2 The second array
	*	@return A new array where the elements are the products of the corresponding elements of the input arrays
	*/
	public static double[] vectorMultiply(double[] v1, double[] v2) {
		int sz = Math.min(v1.length,v2.length);
		int sSz = Math.max(v1.length,v2.length);
		double[] nV = new double[sSz];
		for (int i=0; i<sz; i++)
			nV[i] = v1[i] * v2[i];
		if (v1.length > sz)
			for (int i=sz; i<sSz; i++)
				nV[i] = 0;
		else if (v2.length > sz)
			for (int i=sz; i<sSz; i++)
				nV[i] = 0;
		return nV;
	}
	/**
	*	Multiplies a matrix by a scalar
	*	@param arr Matrix to scale
	*	@param f The scalar to scale the input matrix by
	*	@return The input matrix scaled by f
	*/
	public static double[][] vectorMultiply(double[][] arr, double f) {
		int sz = arr.length;
		double[][] nV = new double[sz][];
		for (int i=0; i<sz; i++)
			nV[i] = vectorMultiply(arr[i],f);
		return nV;
	}
	/**
	*	Multiplies an array by a scalar
	*	@param arr Array to scale
	*	@param f The scalar to scale the input array by
	*	@return The input array scaled by f
	*/
	public static int[] vectorMultiply(int[] arr, int f) {
		int sz = arr.length;
		int[] nV = new int[sz];
		for (int i=0; i<sz; i++)
			nV[i] = arr[i] * f;
		return nV;
	}
}

