/**
*	SIGRS - Identifying genomic regions of contrasting composition using a partial sum process
*	Copyright (C) 2008 Pontus Larsson
*	 
*	This file is part of SIGRS.
*	  
*	SIGRS is free software: you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation, either version 3 of the License, or
*	(at your option) any later version.
*
*	SIGRS is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with SIGRS. If not, see <http://www.gnu.org/licenses/>.
*/
package SIGRS;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.Writer;

       
/**
*	SIGRS is a collection of routines used in searching for regions of contrasting composition (CCRs) in sequence files using a partial sum process.
*	Significance of segments is evaluated using Karlin-Altschul statistics and specifically an extension by Karlin-Dembo allowing
*	for nucleotides to have a Markov-dependence (see e.g. <a href="http://www.pnas.org/cgi/reprint/90/12/5873" target="_blank">Karlin & Altschul (1993)</a> and
*	<a href="http://www.jstor.org/view/00018678/ap050087/05a00070/0" target="_blank">Karlin & Dembo (1992)</a>
*	<P>
*	The routines are provided as is and no guarantee regarding stability etc. is given so use at your own risk!
*	<P>
*	<P>
*	See publication <b>Larsson, P., Hinas, A., Ardell, D.H., Kirsebom, L.A., Virtanen, A. and Sderbom, F.</b> <i>De novo search for non-coding RNA genes in the AT-rich genome of Dictyostelium discoideum: performance of
*	Markov-dependent genome feature scoring</i>
*	<P>
*	Questions and comments can be directed to <a href="mailto:Pontus.Larsson@icm.uu.se">Pontus.Larsson@icm.uu.se</a>
*	@author Pontus Larsson
*
*/

public class LogWriter extends PrintWriter
{
	private boolean initialized;
	private boolean dontClose;
	private String parent;

	public LogWriter()
	{
		super(System.out);
		this.initialized = true;
		this.dontClose = true;
		this.parent = System.getProperty("user.dir");
	}
	public LogWriter(String logFile) throws Exception
	{
		super(new FileOutputStream(logFile));
		this.initialized = true;
		this.dontClose = false;
		this.parent = (new File(logFile)).getPath();
	}
	public LogWriter(File logFile) throws Exception
	{
		super(new FileOutputStream(logFile));
		this.initialized = true;
		this.dontClose = false;
		this.parent = logFile.getPath();
	}
	public LogWriter(File logFile, boolean autoFlush) throws Exception
	{
		super(new FileOutputStream(logFile),autoFlush);
		this.initialized = true;
		this.dontClose = false;
		this.parent = logFile.getPath();
	}
	public LogWriter(String logFile, boolean autoFlush) throws Exception
	{
		super(new FileOutputStream(logFile),autoFlush);
		this.initialized = true;
		this.dontClose = false;
		this.parent = (new File(logFile)).getPath();
	}
	public LogWriter(OutputStream logStream)
	{
		super(logStream);
		this.initialized = true;
		this.dontClose = false;
		this.parent = System.getProperty("user.dir");
	}
	public LogWriter(OutputStream logStream, boolean autoFlush)
	{
		super(logStream,autoFlush);
		this.initialized = true;
		this.dontClose = false;
		this.parent = System.getProperty("user.dir");
	}
	public LogWriter(Writer logWriter)
	{
		super(logWriter);
		this.initialized = true;
		this.dontClose = false;
		this.parent = System.getProperty("user.dir");
	}
	public LogWriter(Writer logWriter, boolean autoFlush)
	{
		super(logWriter,autoFlush);
		this.initialized = true;
		this.dontClose = false;
		this.parent = System.getProperty("user.dir");
	}

	public boolean checkError() {return (this.initialized ? super.checkError() : false);}
	public void close()
	{
		try{
			if (this.initialized && !this.dontClose)
			{
				super.out.close();
				this.initialized = false;
			}
		} catch (Exception e) {this.setError();}
	}
	public void flush()
	{
		try{
			if (this.initialized)
				super.out.flush();
		} catch (Exception e) {this.setError();}
	}
	public String getParent() {return this.parent;}
	public void newLine() {this.println();}
	public void print(boolean b)
	{
		if (this.initialized)
			super.print(b);
	}
	public void print(char c)
	{
		if (this.initialized)
			super.print(c);
	}
	public void print(char[] s)
	{
		if (this.initialized)
			super.print(s);
	}
	public void print(double d)
	{
		if (this.initialized)
			super.print(d);
	}
	public void print(float f)
	{
		if (this.initialized)
			super.print(f);
	}
	public void print(int i)
	{
		if (this.initialized)
			super.print(i);
	}
	public void print(long l)
	{
		if (this.initialized)
			super.print(l);
	}
	public void print(Object obj)
	{
		if (this.initialized)
			super.print(obj);
	}
	public void print(String s)
	{
		if (this.initialized)
			super.print(s);
	}
	public void println()
	{
		if (this.initialized)
			super.println();
	}
	public void println(boolean b)
	{
		if (this.initialized)
			super.println(b);
	}
	public void println(char c)
	{
		if (this.initialized)
			super.println(c);
	}
	public void println(char[] s)
	{
		if (this.initialized)
			super.println(s);
	}
	public void println(double d)
	{
		if (this.initialized)
			super.println(d);
	}
	public void println(float f)
	{
		if (this.initialized)
			super.println(f);
	}
	public void println(int i)
	{
		if (this.initialized)
			super.println(i);
	}
	public void println(long l)
	{
		if (this.initialized)
			super.println(l);
	}
	public void println(Object obj)
	{
		if (this.initialized)
			super.println(obj);
	}
	public void println(String s)
	{
		if (this.initialized)
			super.println(s);
	}
	protected void setError()
	{
		if (this.initialized)
			super.setError();
	}
	public void write(char[] buf)
	{
		try{
			if (this.initialized)
				super.out.write(buf);
			this.flush();
		} catch (Exception e) {this.setError();}
	}
	public void write(char[] buf, int off, int len)
	{
		try{
			if (this.initialized)
				super.out.write(buf,off,len);
			this.flush();
		} catch (Exception e) {this.setError();}
	}
	public void write(int i)
	{
		try{
			if (this.initialized)
				super.out.write(i);
			this.flush();
		} catch (Exception e) {this.setError();}
	}
	public void write(String s)
	{
		try{
			if (this.initialized)
				super.out.write(s);
			this.flush();
		} catch (Exception e) {this.setError();}
	}
	public void write(String s, int off, int len)
	{
		try{
			if (this.initialized)
				super.out.write(s,off,len);
			this.flush();
		} catch (Exception e) {this.setError();}
	}
}
