/**
*	SIGRS - Identifying genomic regions of contrasting composition using a partial sum process
*	Copyright (C) 2008 Pontus Larsson
*	 
*	This file is part of SIGRS.
*	  
*	SIGRS is free software: you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation, either version 3 of the License, or
*	(at your option) any later version.
*
*	SIGRS is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with SIGRS. If not, see <http://www.gnu.org/licenses/>.
*/ 

package SIGRS;

import java.util.Vector;
import Jama.Matrix;

      
/**
*	SIGRS is a collection of routines used in searching for regions of contrasting composition (CCRs) in sequence files using a partial sum process.
*	Significance of segments is evaluated using Karlin-Altschul statistics and specifically an extension by Karlin-Dembo allowing
*	for nucleotides to have a Markov-dependence (see e.g. <a href="http://www.pnas.org/cgi/reprint/90/12/5873" target="_blank">Karlin & Altschul (1993)</a> and
*	<a href="http://www.jstor.org/view/00018678/ap050087/05a00070/0" target="_blank">Karlin & Dembo (1992)</a>
*	<P>
*	The routines are provided as is and no guarantee regarding stability etc. is given so use at your own risk!
*	<P>
*	<P>
*	See publication <b>Larsson, P., Hinas, A., Ardell, D.H., Kirsebom, L.A., Virtanen, A. and Sderbom, F.</b> <i>De novo search for non-coding RNA genes in the AT-rich genome of Dictyostelium discoideum: performance of
*	Markov-dependent genome feature scoring</i>
*	<P>
*	Questions and comments can be directed to <a href="mailto:Pontus.Larsson@icm.uu.se">Pontus.Larsson@icm.uu.se</a>
*	@author Pontus Larsson
*
*/

public class KDTreeNode
{
	private int label;
	private Matrix value;
	private Object[] children;
	private boolean passed;
	private Matrix[] cache;

	public KDTreeNode(int label, Matrix value) {
		this.label = label;
		this.value = null;
		this.passed = false;
		this.cache = new Matrix[0];
		if (value != null)
			this.value = value.copy();
		this.children = new KDTreeNode[0];
	}

	public void addChild(KDTreeNode child) {this.children = Methods.addElement(child,this.children);}

	public void buildTree(int[] steps, int maxValue, KDTreeNode parent) {
		KDTreeNode tNode;
		KDTreeNode child;
		for (int i=0; i<steps.length; i++) {
			if (this.label <= maxValue+steps[i] && this.label >= steps[i]) {
				child = new KDTreeNode(this.label-steps[i],null);
				if ((tNode = parent.findNode(child.getLabel(),new Vector())) != null)
					child.setChildren(tNode.getChildren());
				else
					child.buildTree(steps,maxValue,this);
				this.addChild(child);
			}
		}
	}

	public KDTreeNode findNode(int lbl, Vector visited) {
		if (visited.contains(new Integer(this.label)))
			return null;
		visited.add(new Integer(this.label));

		if (this.label == lbl)
			return this;
		KDTreeNode tNode = null;
		for (int i=0; i<this.children.length; i++) {
			tNode = ((KDTreeNode) this.children[i]).findNode(lbl,visited);
			if (tNode != null)
				return tNode;
		}
		return null;
	}

	public int getLabel() {return this.label;}

	public Object[] getChildren() {return this.children;}

	public Matrix getPath(int destination, int maxValue) {
		if (this.cache.length == 0) {
			int levelsAbove = (maxValue-this.label);
			this.cache = new Matrix[levelsAbove+1];
			Matrix t;
			for (int i=1; i<=levelsAbove; i++) {
				t = new Matrix(this.value.getRowDimension(),this.value.getColumnDimension());
				for (int j=0; j<this.children.length; j++)
					if (((KDTreeNode) this.children[j]).getLabel() <= this.label+i)
						t = t.plus(((KDTreeNode) this.children[j]).getPath(this.label+i,maxValue));
				this.cache[i] = t.times(this.value);
			}
			this.cache[0] = this.value;
		}
		return this.cache[destination-this.label];
	}

	public void reset(Vector visited) {
		this.cache = new Matrix[0];
		this.passed = false;
		if (visited.contains(new Integer(this.label)))
			return;
		visited.add(new Integer(this.label));
		for (int i=0; i<this.children.length; i++)
			((KDTreeNode) this.children[i]).reset(visited);
	}

	public void setChildren(Object[] children) {this.children = children;}
	public void setValue(Matrix value) {this.value = value.copy();}

	public void updateTree(int[] steps, Matrix[] q, int parent) {
		if (this.passed)
			return;
		int index = Methods.indexOf(-1*(this.label-parent),steps);
		if (index >= 0)	{
			this.passed = true;
			this.setValue(q[index]);
		}
		for (int i=0; i<this.children.length; i++)
			((KDTreeNode) this.children[i]).updateTree(steps,q,this.label);
	}
}
