/*
Copyright 2007 Daniel Zerbino (zerbino@ebi.ac.uk)

    This file is part of Velvet.

    Velvet is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Velvet is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Velvet; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "run.h"

int main(int argc, char **argv)
{
	RoadMapArray *rdmaps;
	Graph *graph;
	ReadSet *sequences;
	double coverageCutoff;
	char *directory, *filename;
	boolean *dubious = NULL;
	Coordinate cutoffLength = 100;

	if (argc == 1) {
		puts("velvetg - de Bruijn graph construction, error removal and repeat resolution");
		puts("Usage:");
		puts("./velvetg directory hash_length [coverage_cutoff [max_insert_length [max_insert_length2]]]");
		puts("\t- directory: working directory name");
		puts("\t- hash_length: the same integer given to the hash program");
		puts("\t- coverage_cutoff: optional removal of low coverage nodes AFTER tour bus");
		puts("\t- max_insert_length : rough estimate of max insert length");
		puts("\t- max_insert_length2 : rough estimate of max insert length in second short paired read set"); 
		puts("Output (within the specified directory):");
		puts("\t- contigs.fa: fasta file of contigs longer than 100bp");
		puts("\t- stats.txt: stats file (tab-spaced) useful for determining appropriate coverage cutoff");
		return 1;
	}

	directory = argv[1];
	filename = malloc((strlen(directory) + 100) * sizeof(char));

	if (argc < 4) {
		strcpy(filename, directory);
		strcat(filename, "/Sequences");
		sequences = readFastAFile(filename);
		convertSequences(sequences);

		strcpy(filename, directory);
		strcat(filename, "/Graph");
		if (fopen(filename, "r") != NULL) {
			graph = importGraph(filename);
		} else {
			strcpy(filename, directory);
			strcat(filename, "/Roadmaps");
			rdmaps = importRoadMapArray(filename);

			categorizeRoadMapArray(rdmaps, 0);

			graph = newGraph(rdmaps, sequences);

			// Coopy for reuse if playing with factors
			strcpy(filename, directory);
			strcat(filename, "/Graph");
			exportGraph(filename, graph,
				    sequences->tSequences);
		}

		correctGraph(graph, 0.2);

		if (argc == 3) {
			sscanf(argv[2], "%lf", &coverageCutoff);
			removeLowCoverageNodes(graph, coverageCutoff);
		}
	} else {
		strcpy(filename, directory);
		strcat(filename, "/Sequences");
		sequences = importReadSet(filename);
		convertSequences(sequences);

		strcpy(filename, directory);
		strcat(filename, "/Graph2");
		if (fopen(filename, "r") != NULL) {
			graph = importGraph(filename);
			correctGraph(graph, 0.2);
		} else {
			strcpy(filename, directory);
			strcat(filename, "/Roadmaps");
			rdmaps = importRoadMapArray(filename);

			graph = newGraph(rdmaps, sequences);

			correctGraph(graph, 0.2);

			// Coopy for reuse if playing with factors
			strcpy(filename, directory);
			strcat(filename, "/Graph2");
			exportGraph(filename, graph,
				    sequences->tSequences);
		}

		printf
		    ("Starting graph has %li nodes and n50 of %li max %li\n",
		     nodeCount(graph), n50(graph), maxLength(graph));

		sscanf(argv[2], "%lf", &coverageCutoff);
		if (coverageCutoff > 0) {
			dubious =
			    removeLowCoverageNodesAndDenounceDubiousReads(graph,
									  coverageCutoff);
		}

		sscanf(argv[3], "%li", &cutoffLength);
		pairUpReads(sequences, 1);
		detachDubiousReads(sequences, dubious);
		exploitShortReadPairs(graph, sequences->mateReads,
				      cutoffLength, false);

		if (argc > 4) {
			sscanf(argv[3], "%li", &cutoffLength);
			pairUpReads(sequences, 3);
			detachDubiousReads(sequences, dubious);
			exploitShortReadPairs(graph, sequences->mateReads,
					      cutoffLength, true);
		}

		renumberNodes(graph);
	}

	strcpy(filename, directory);
	strcat(filename, "/stats.txt");
	displayGeneralStatistics(graph, 1, filename);

	strcpy(filename, directory);
	strcat(filename, "/contigs.fa");
	exportLongNodeSequences(filename, graph, cutoffLength);

	printf("Final graph has %li nodes and n50 of %li max %li\n",
	       nodeCount(graph), n50(graph), maxLength(graph));

	return 1;
}
