/*
Copyright 2007 Daniel Zerbino (zerbino@ebi.ac.uk)

    This file is part of Velvet.

    Velvet is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Velvet is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Velvet; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "globals.h"
#include "graph.h"
#include "recycleBin.h"
#include "tightString.h"
#include "roadMap.h"

#ifndef NULL
#define NULL 0
#endif

struct annotation_st {
	IDnum sequenceID;	// 32
	Coordinate position;	// 32
	Coordinate start;	// 32
	Coordinate finish;	// 32
	Annotation *next;	// 64
	InsertionMarker *startMarker, *finishMarker;	// 128
};

struct roadmap_st {
	Coordinate length;
	Annotation *annotation;
	Category category;
};

#define ROADMAPCHUNKSIZE 100000
#define ANNOTATIONCHUNKSIZE 100000

static RecycleBin *roadMapMemory;
static RecycleBin *annotationMemory;

// Modifies annot so that it spans over one more bp
void incrementAnnotation(Annotation * annot)
{
	if (annot->sequenceID > 0)
		annot->finish++;
	else
		annot->finish--;
}

// Creates new annotation from the given information
Annotation *newAnnotation(IDnum sequenceID, Coordinate position,
			  Coordinate insertedPosition)
{
	Annotation *newAnnot;

	if (annotationMemory == NULL)
		annotationMemory =
		    newRecycleBin(sizeof(Annotation), ANNOTATIONCHUNKSIZE);

	newAnnot = allocatePointer(annotationMemory);
	newAnnot->position = position;
	newAnnot->sequenceID = sequenceID;
	newAnnot->start = insertedPosition;
	newAnnot->finish = insertedPosition;
	newAnnot->next = NULL;
	incrementAnnotation(newAnnot);

	return newAnnot;
}

// Creates empty road map and memorizes tString's length
RoadMap *newRoadMap(TightString * tString)
{
	RoadMap *rdmap;

	if (roadMapMemory == NULL)
		roadMapMemory =
		    newRecycleBin(sizeof(RoadMap), ROADMAPCHUNKSIZE);

	rdmap = allocatePointer(roadMapMemory);
	rdmap->length = getLength(tString);
	rdmap->annotation = NULL;
	rdmap->category = CATEGORIES;
	return rdmap;
}

// Creates empty road map and memorizes tString's length
RoadMap *newRoadMapFromLength(Coordinate length)
{
	RoadMap *rdmap;

	if (roadMapMemory == NULL)
		roadMapMemory =
		    newRecycleBin(sizeof(RoadMap), ROADMAPCHUNKSIZE);

	rdmap = allocatePointer(roadMapMemory);
	rdmap->length = length;
	rdmap->annotation = NULL;
	rdmap->category = CATEGORIES;
	return rdmap;
}

char *readAnnotation(Annotation * annot)
{
	char *s = calloc(30, sizeof(char));
	sprintf(s, "<<%li[%li,%li](%li)>>", annot->sequenceID,
		annot->start, annot->finish, annot->position);
	return s;
}

char *readRoadMap(RoadMap * rdmap)
{
	char *s = malloc(sizeof(char));
	*s = '\0';
	Annotation *annot = rdmap->annotation;

	sprintf(s, "lgth %li :", rdmap->length);
	while (annot != NULL) {
		strcat(s, readAnnotation(annot));
		annot = annot->next;
	}

	return s;
}

Annotation *getNextAnnotation(Annotation * annot)
{
	return annot->next;
}

void setNextAnnotation(Annotation * previous, Annotation * next)
{
	previous->next = next;
}

Annotation *getAnnotation(RoadMap * rdmap)
{
	return rdmap->annotation;
}


void setAnnotation(RoadMap * rdmap, Annotation * annot)
{
	rdmap->annotation = annot;
}

Coordinate getFinish(Annotation * annot)
{
	return annot->finish;
}

void setFinish(Annotation * annot, Coordinate finish)
{
	annot->finish = finish;
}

IDnum getAnnotSequenceID(Annotation * annot)
{
	return annot->sequenceID;
}

Coordinate getMapLength(RoadMap * map)
{
	return map->length;
}

Coordinate getStart(Annotation * annot)
{
	return annot->start;
}

InsertionMarker *getStartMarker(Annotation * annot)
{
	return annot->startMarker;
}

void setStartMarker(Annotation * annot, InsertionMarker * marker)
{
	annot->startMarker = marker;
}

InsertionMarker *getFinishMarker(Annotation * annot)
{
	return annot->finishMarker;
}

void setFinishMarker(Annotation * annot, InsertionMarker * marker)
{
	annot->finishMarker = marker;
}

Coordinate getPosition(Annotation * annot)
{
	return annot->position;
}

void destroyRoadMap(RoadMap * rdmap)
{
	Annotation *annot;

	while (rdmap->annotation != NULL) {
		annot = rdmap->annotation;
		rdmap->annotation = annot->next;
		deallocatePointer(annotationMemory, annot);
	}

	deallocatePointer(roadMapMemory, rdmap);
}

Annotation *copyAnnotation(Annotation * annot)
{
	Annotation *copy;

	if (annot == NULL)
		return NULL;

	if (annotationMemory == NULL)
		annotationMemory =
		    newRecycleBin(sizeof(Annotation), ANNOTATIONCHUNKSIZE);

	copy = allocatePointer(annotationMemory);
	copy->sequenceID = annot->sequenceID;
	copy->position = annot->position;
	copy->start = annot->start;
	copy->finish = annot->finish;
	copy->next = annot->next;
	copy->startMarker = NULL;
	copy->finishMarker = NULL;

	return copy;
}

void trimRoadMap(RoadMap * rdmap, Coordinate length)
{
	rdmap->length = length;
}

Coordinate getAnnotationLength(Annotation * annot)
{
	if (annot == NULL)
		return 0;

	if (annot->sequenceID > 0)
		return annot->finish - annot->start;
	else
		return annot->start - annot->finish;
}

void destroyAllRoadMaps()
{
	destroyRecycleBin(roadMapMemory);
	destroyRecycleBin(annotationMemory);
	roadMapMemory = NULL;
	annotationMemory = NULL;
}

void setRoadMapCategory(RoadMap * rdmap, Category category)
{
	rdmap->category = category;
}

Category getRoadMapCategory(RoadMap * rdmap)
{
	return rdmap->category;
}

// Imports roadmap from the appropriate file format
// Memory allocated within the function
RoadMapArray *importRoadMapArray(char *filename)
{
	FILE *file;
	const int maxline = 100;
	char *line = malloc(sizeof(char) * maxline);
	RoadMap **array;
	RoadMap *rdmap;
	IDnum rdmapIndex;
	IDnum length;
	IDnum seqID;
	Coordinate position, start, finish;
	Annotation *latestAnnotation;
	Annotation *newAnnot;
	Category category;
	int holder;
	RoadMapArray *result = malloc(sizeof(RoadMapArray));
	IDnum sequenceCount;

	printf("Reading file %s\n", filename);

	file = fopen(filename, "r");

	fgets(line, maxline, file);
	sscanf(line, "%li\t%i\n", &sequenceCount, &(result->WORDLENGTH));
	result->length = sequenceCount;
	array = malloc(sequenceCount * sizeof(RoadMap *));
	result->array = array;

	rdmapIndex = -1;
	while (fgets(line, maxline, file) != NULL) {
		if (line[0] == 'R') {
			fgets(line, maxline, file);
			sscanf(line, "%li\t%i\n", &length, &holder);
			category = holder;
			rdmapIndex++;
			rdmap = newRoadMapFromLength(length);
			setRoadMapCategory(rdmap, category);
			array[rdmapIndex] = rdmap;
			latestAnnotation = NULL;
		}

		else {
			sscanf(line, "%li\t%li\t%li\t%li\n", &seqID,
			       &position, &start, &finish);
			newAnnot = newAnnotation(seqID, position, start);
			setFinish(newAnnot, finish);
			if (latestAnnotation == NULL)
				setAnnotation(rdmap, newAnnot);
			else
				setNextAnnotation(latestAnnotation,
						  newAnnot);
			latestAnnotation = newAnnot;
		}
	}
	fclose(file);
	free(line);
	return result;
}

// Exports the  contents of a specific roadMap into a given file
// Internal to exportRoadMapArray
void exportRoadMap(FILE * outfile, RoadMap * rdmap)
{
	Annotation *annot;

	fprintf(outfile, "ROADMAP\n");
	fprintf(outfile, "%li\t%i\n", getMapLength(rdmap),
		getRoadMapCategory(rdmap));

	annot = getAnnotation(rdmap);
	while (annot != NULL) {
		fprintf(outfile, "%li\t%li\t%li\t%li\n",
			getAnnotSequenceID(annot), getPosition(annot),
			getStart(annot), getFinish(annot));
		annot = getNextAnnotation(annot);
	}

	destroyRoadMap(rdmap);
	fflush(outfile);
}

// Exports the content of a RoadMAp array (to be later used for graph construction esp.) into a new file
// Especially useful when memory management overhead is unbearable after the contstruction deconstruction of the hashtable.
void exportRoadMapArray(char *filename, RoadMapArray * rdmapArray)
{
	IDnum sequenceCount = rdmapArray->length;
	RoadMap **array = rdmapArray->array;
	int WORDLENGTH = rdmapArray->WORDLENGTH;
	int index;
	FILE *outfile = fopen(filename, "w");

	if (outfile == NULL) {
		puts("Couldn't open file, sorry");
		return;
	} else
		puts("Writing into file...");

	fprintf(outfile, "%li\t%i\n", sequenceCount, WORDLENGTH);

	for (index = 0; index < sequenceCount; index++)
		exportRoadMap(outfile, array[index]);

	free(array);
	free(rdmapArray);
	fclose(outfile);
}

FILE *openRoadMapFile(char *filename, IDnum sequenceCount, int WORDLENGTH)
{
	FILE *outfile = fopen(filename, "w");

	if (outfile == NULL) {
		puts("Couldn't open file, sorry");
		exit(-1);
	} else
		puts("Writing into file...");

	fprintf(outfile, "%li\t%i\n", sequenceCount, WORDLENGTH);

	return outfile;
}

void categorizeRoadMapArray(RoadMapArray * rdmapArray, Category cat)
{
	IDnum index;
	RoadMap **array = rdmapArray->array;
	RoadMap *rdmap;

	for (index = 0; index < rdmapArray->length; index++) {
		rdmap = array[index];
		rdmap->category = cat;
	}
}
