/*
Copyright 2007 Daniel Zerbino (zerbino@ebi.ac.uk)

    This file is part of Velvet.

    Velvet is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Velvet is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Velvet; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include <stdlib.h>
#include <stdio.h>

#include "globals.h"
#include "graph.h"
#include "recycleBin.h"
#include "passageMarker.h"
#include "graphStats.h"
#include "concatenatedGraph.h"

#define LONG_NODE_CUTOFF 50
#define LN2 0.693147
#define PROBABILITY_CUTOFF 5

static Graph *graph = NULL;
static PassageMarker *path = NULL;
static RecycleBin *listMemory = NULL;
static boolean(*isUniqueFunction) (Node * node) = NULL;
static double expected_coverage = 1;

boolean isUniqueBasic(Node * node)
{
	printf("Testing node %li %li", readCoverage(node),
	       getNodeLength(node));

	if (getNodeLength(node) < LONG_NODE_CUTOFF) {
		puts("\tUNDETERMINED");
		return false;
	}
	if (readCoverage(node) / (double) getNodeLength(node) >
	    1.5 * expected_coverage) {
		puts("\tUNDETERMINED");
		return false;
	}

	puts("\tUNIQUE");
	return true;
}

boolean isUniqueSolexa(Node * node)
{

	Coordinate nodeLength = getNodeLength(node);
	Coordinate nodeCoverage =
	    (getVirtualCoverage(node, 0) + getVirtualCoverage(node, 1));
	double nodeDensity, probability;

	printf("Testing node %li %li", nodeCoverage, nodeLength);

	if (nodeLength > LONG_NODE_CUTOFF) {
		nodeDensity = nodeCoverage / (double) nodeLength;

		probability =
		    LN2 / 2 +
		    nodeLength / (2 * expected_coverage) *
		    (expected_coverage * expected_coverage -
		     nodeDensity * nodeDensity / 2);

		if (probability > PROBABILITY_CUTOFF)
			puts("\tUNIQUE");
		else
			puts("\tUNDETERMINED");

		return probability > PROBABILITY_CUTOFF;
	} else {
		probability =
		    expected_coverage * nodeLength - nodeCoverage / LN2;

		if (probability > 0)
			puts("\tUNIQUE");
		else
			puts("\tUNDETERMINED");

		return probability > 0;
	}
}

static void identifyUniqueNodes()
{
	IDnum index;
	Node *node;
	IDnum counter = 0;

	puts("Identifying unique nodes");

	for (index = 1; index <= nodeCount(graph); index++) {
		node = getNodeInGraph(graph, index);

		if (node == NULL)
			continue;

		setUniqueness(node, isUniqueFunction(node));

		if (getUniqueness(node))
			counter++;
	}

	printf("Done, %lu unique nodes counted\n", counter);
}

static void trimNodeLength(Node * node)
{
	PassageMarker *marker;
	Coordinate minOffset = getNodeLength(node);

	for (marker = getMarker(node); marker != NULL;
	     marker = getNextInNode(marker))
		if (getFinishOffset(marker) < minOffset)
			minOffset = getFinishOffset(marker);

	if (minOffset == 0)
		return;

	clipNodeLength(node, 0, minOffset);

	for (marker = getMarker(node); marker != NULL;
	     marker = getNextInNode(marker))
		setFinishOffset(marker,
				getFinishOffset(marker) - minOffset);
}

static void simplifyNode(Node * node)
{
	Node *twin = getTwinNode(node);
	Node *destination, *twinDestination;

	if (arcCount(node) == 0)
		trimNodeLength(node);

	if (simpleArcCount(node) != 1)
		return;

	destination = getDestination(getArc(node));
	twinDestination = getTwinNode(destination);

	while (simpleArcCount(node) == 1
	       && simpleArcCount(twinDestination) == 1
	       && destination != twin && destination != node) {
		if (getNode(path) == destination
		    || getNode(path) == twinDestination) {
			return;
		}

		if (getNode(getNextInSequence(path)) == destination
		    || getNode(getNextInSequence(path)) ==
		    twinDestination) {
			return;
		}

		if (getNodeStatus(destination))
			return;

		printf
		    ("Concatenating nodes %li (%li %li) and %li (%li %li)\n",
		     getNodeID(node), getNodeLength(node),
		     readCoverage(node), getNodeID(destination),
		     getNodeLength(destination),
		     readCoverage(destination));

		concatenateNodes(node, destination, graph);

		if (simpleArcCount(node) != 1)
			return;

		destination = getDestination(getArc(node));
		twinDestination = getTwinNode(destination);
	}

}

static void simplifyNodeConservative(Node * node, PassageMarker ** marker)
{
	return;
	Node *twin = getTwinNode(node);
	Node *destination, *twinDestination;

	if (arcCount(node) == 0)
		trimNodeLength(node);

	if (simpleArcCount(node) != 1)
		return;

	destination = getDestination(getArc(node));
	twinDestination = getTwinNode(destination);

	while (simpleArcCount(node) == 1
	       && simpleArcCount(twinDestination) == 1
	       && destination != twin && destination != node) {

		if (getNode(path) == destination
		    || getNode(path) == twinDestination) {
			return;
		}

		if (getNode(getNextInSequence(path)) == destination
		    || getNode(getNextInSequence(path)) ==
		    twinDestination) {
			return;
		}

		if (getNode(*marker) == destination && !isInitial(*marker)) {
			printf("Tmp marker moved from %li to %li\n",
			       getNodeID(getNode(*marker)),
			       getNodeID(getNode
					 (getPreviousInSequence
					  (*marker))));
			*marker = getPreviousInSequence(*marker);
		}

		printf("CConcatenating nodes %li and %li\n",
		       getNodeID(node), getNodeID(destination));

		concatenateNodes(node, destination, graph);

		if (simpleArcCount(node) != 1)
			return;

		destination = getDestination(getArc(node));
		twinDestination = getTwinNode(destination);
	}

}

static PassageMarkerList *copyMarkers(Node * node, IDnum * count)
{
	PassageMarkerList *list = NULL;
	PassageMarkerList *new;
	PassageMarker *currentMarker;

	*count = 0;
	for (currentMarker = getMarker(node); currentMarker != NULL;
	     currentMarker = getNextInNode(currentMarker)) {
		new = allocatePointer(listMemory);
		new->marker = currentMarker;
		new->next = list;
		list = new;
		(*count)++;
	}

	return list;
}

static void removeDead(PassageMarkerList ** list, IDnum * counter)
{
	PassageMarkerList *current, *next;

	if (*list == NULL)
		return;

	current = *list;

	while (current->next != NULL) {
		next = current->next;

		if (isTerminal(next->marker)) {
			current->next = next->next;
			deallocatePointer(listMemory, next);
			(*counter)--;
		} else
			current = current->next;
	}

	current = *list;
	if (isTerminal(current->marker)) {
		*list = current->next;
		deallocatePointer(listMemory, current);
		(*counter)--;
	}
}

static Node *chooseDestination(PassageMarkerList * list)
{
	PassageMarkerList *current;
	Node *destination;

	if (list == NULL)
		return NULL;

	destination = getNode(getNextInSequence(list->marker));
	for (current = list; current != NULL; current = current->next)
		if (getNode(getNextInSequence(current->marker)) !=
		    destination)
			return NULL;

	return destination;
}

static IDnum selectGroupiesForList(Node * node,
				   PassageMarkerList ** groupies)
{
	PassageMarker *current;
	PassageMarkerList *new;
	IDnum counter = 0;

	for (current = getMarker(node); current != NULL;
	     current = getNextInNode(current)) {
		if (isInitial(current)) {
			new = allocatePointer(listMemory);
			new->marker = current;
			new->next = *groupies;
			*groupies = new;
			counter++;
		}
	}

	return counter;
}

static IDnum updateGroupiesInList(PassageMarkerList ** groupies,
				  Node * nextNode)
{
	PassageMarkerList *current = *groupies;
	PassageMarkerList *next;
	IDnum counter = 0;

	if (*groupies == NULL)
		return counter;

	while (current->next != NULL) {
		next = current->next;
		if (!isDestinationToMarker(next->marker, nextNode)) {
			current->next = next->next;
			deallocatePointer(listMemory, next);
			counter++;
		} else {
			next->marker = getNextInSequence(next->marker);
			current = next;
		}
	}

	current = *groupies;
	if (current == NULL)
		return counter;

	if (!isDestinationToMarker(current->marker, nextNode)) {
		*groupies = current->next;
		deallocatePointer(listMemory, current);
		counter++;
	} else {
		current->marker = getNextInSequence(current->marker);
	}

	return counter;
}

static void updateMarkers(PassageMarkerList * list)
{
	PassageMarkerList *current;

	for (current = list; current != NULL; current = current->next)
		current->marker = getNextInSequence(current->marker);
}

static void destroyPassageMarkerList(PassageMarkerList ** list)
{
	PassageMarkerList *ptr;

	while (*list != NULL) {
		ptr = *list;
		*list = ptr->next;
		deallocatePointer(listMemory, ptr);
	}
}

static void destroyPath()
{
	PassageMarker *marker;
	Node *node;

	while (path != NULL) {
		marker = path;
		path = getNextInSequence(marker);
		node = getNode(marker);
		destroyPassageMarker(marker);

		if (node == NULL)
			continue;

		if (getMarker(node) == NULL) {
			printf("Destroying node %li\n", getNodeID(node));
			destroyNode(node, graph);
		} else {
			simplifyNode(node);
			simplifyNode(getTwinNode(node));
		}
	}
}

static void decrementArc(Node * origin, Node * destination)
{
	Arc *arc = getArcBetweenNodes(origin, destination, graph);

	if (arc == NULL)
		return;

	changeMultiplicity(arc, -1);

	if (getMultiplicity(arc) == 0) {
		//printf("Destroying arc between nodes %li and %li\n", getNodeID(origin), getNodeID(destination));
		destroyArc(arc, graph);
	}
}

static boolean uniqueNodesConnect(Node * startingNode,
				  boolean(*isUnique) (Node *))
{
	PassageMarkerList *list, *groupies;
	PassageMarker *pathHead;
	Node *nextNode, *currentNode;
	IDnum markerCounter, groupieCount = 0;
	IDnum count = 0;

	currentNode = startingNode;

	if (arcCount(currentNode) == 0) {
		puts("Dead end");
		return false;
	}

	if (getMarker(startingNode) == NULL) {
		puts("Empty node");
		return false;
	}

	printf("Testing connectivity of unique  node %li (%i)\n",
	       getNodeID(startingNode), arcCount(startingNode));

	path = addUncertainPassageMarker(0, startingNode);
	pathHead = path;
	list = copyMarkers(startingNode, &markerCounter);
	groupies = NULL;

	while (true) {
		removeDead(&list, &markerCounter);

		if (list == NULL) {
			destroyPassageMarkerList(&groupies);
			printf("MOut of readsi %li\n", count);
			destroyPath();
			return false;
			/*if (arcCount(currentNode) != 0) {
			   printf("MOut of readsi %li\n", count);
			   destroyPath();
			   return false;
			   } else {
			   puts("MCollective suicide");
			   connectPassageMarkers(pathHead,
			   addUncertainPassageMarker
			   (0, NULL), graph);
			   pathHead = getNextInSequence(pathHead);
			   if (getNode(path) != startingNode)
			   abort();
			   return true;
			   } */
		}

		nextNode = chooseDestination(list);
		if (nextNode == NULL) {
			printf
			    ("MDiscord at node %li length %li mult %li\n",
			     getNodeID(currentNode),
			     getNodeLength(currentNode),
			     markerCount(currentNode));
			destroyPassageMarkerList(&list);
			destroyPassageMarkerList(&groupies);
			destroyPath();
			return false;
		}

		printf
		    ("Going through %li length %li coverage %li unique %hi markers %li\n",
		     getNodeID(nextNode), getNodeLength(nextNode),
		     readCoverage(nextNode), getUniqueness(nextNode),
		     markerCounter);

		connectPassageMarkers(pathHead,
				      addUncertainPassageMarker(0,
								nextNode),
				      graph);
		pathHead = getNextInSequence(pathHead);
		groupieCount -= updateGroupiesInList(&groupies, nextNode);

		if (nextNode == startingNode
		    || nextNode == getTwinNode(startingNode)) {
			puts("MSelf loop");
			destroyPath();
			destroyPassageMarkerList(&list);
			destroyPassageMarkerList(&groupies);
			return false;
		}

		if (getUniqueness(nextNode)) {
			printf("Unique node tests %i %li %li %li\n",
			       simpleArcCount(getTwinNode(nextNode)),
			       getMultiplicity(getArc
					       (getTwinNode(nextNode))),
			       markerCounter, groupieCount);
			if (simpleArcCount(getTwinNode(nextNode)) == 1
			    &&
			    getMultiplicity(getArc(getTwinNode(nextNode)))
			    == markerCounter + groupieCount) {
				destroyPassageMarkerList(&list);
				destroyPassageMarkerList(&groupies);
				puts("MHit");
				while (!isTerminal(path))
					path = getNextInSequence(path);
				path = getTwinMarker(path);

				return true;

			}
		}

		if (simpleArcCount(getTwinNode(nextNode)) == 1 &&
		    getMultiplicity(getArc(getTwinNode(nextNode)))
		    == markerCounter) {
			destroyPassageMarkerList(&list);
			destroyPassageMarkerList(&groupies);
			puts("U2S connection");
			return true;
		}

		updateMarkers(list);
		groupieCount += selectGroupiesForList(nextNode, &groupies);

		currentNode = nextNode;
		count++;
	}
}

static boolean standardNodesConnect(Node * startingNode)
{
	PassageMarkerList *list;
	PassageMarker *pathHead;
	Node *nextNode, *currentNode;
	IDnum readCount = 0;

	if (arcCount(startingNode) == 0) {
		puts("Dead end");
		return false;
	}

	if (getMarker(startingNode) == NULL) {
		puts("Empty node");
		return false;
	}

	printf("Testing connectivity of node %li\n",
	       getNodeID(startingNode));

	list = copyMarkers(startingNode, &readCount);
	path = addUncertainPassageMarker(0, startingNode);
	pathHead = path;

	currentNode = startingNode;
	while (true) {
		/*
		   if (arcCount(currentNode) == 0) {
		   puts("Collective suicide");
		   connectPassageMarkers(pathHead,
		   addUncertainPassageMarker(0,
		   NULL),
		   graph);
		   pathHead = getNextInSequence(pathHead);
		   destroyPassageMarkerList(&list);
		   return true;
		   }
		 */

		nextNode = chooseDestination(list);
		if (nextNode == NULL) {
			printf
			    ("discord at node %li length %li mult %li : ",
			     getNodeID(currentNode),
			     getNodeLength(currentNode),
			     markerCount(currentNode));
			printf("Abandonning\n");
			destroyPassageMarkerList(&list);
			destroyPath();
			return false;
		}

		connectPassageMarkers(pathHead,
				      addUncertainPassageMarker(0,
								nextNode),
				      graph);
		pathHead = getNextInSequence(pathHead);

		if (nextNode == startingNode
		    || nextNode == getTwinNode(startingNode)) {
			puts("Self loop");
			destroyPath();
			return false;
		}

		if (simpleArcCount(getTwinNode(nextNode)) == 1
		    && getMultiplicity(getArc(getTwinNode(nextNode))) ==
		    readCount) {
			destroyPassageMarkerList(&list);
			printf("S2S connection to %li\n",
			       getNodeID(nextNode));
			return true;
		}

		updateMarkers(list);
		currentNode = nextNode;
	}
}

static void selectGroupies(Node * node, PassageMarker ** groupies,
			   Coordinate pastLength)
{
	PassageMarker *current, *new;

	if (node == NULL)
		return;

	for (current = getMarker(node); current != NULL;
	     current = getNextInNode(current)) {
		if (isInitial(current)) {
			new = copyPassageMarker(current);
			setPreviousInSequence(current, new);
			setNextInNode(new, *groupies);
			setStartOffset(new,
				       getStartOffset(new) + pastLength);
			*groupies = new;
		}
	}
}

static void detachRead(PassageMarker * marker)
{
	PassageMarker *current = getPreviousInSequence(marker);
	PassageMarker *tmp;
	Node *currentNode;
	Node *previousNode = getNode(getNextInSequence(marker));

	//printf("Detaching read %li %li %li %li %p\n", getPassageMarkerSequenceID(current), getNodeID(getNode(current)), getPassageMarkerStart(current), getPassageMarkerFinish(current), getTwinMarker(current));

	// Recreate proper arc
	setPreviousInSequence(marker, getNextInSequence(marker));
	createArc(getNode(marker), getNode(getNextInSequence(marker)),
		  graph);

	// Detach memo
	setPreviousInSequence(NULL, marker);

	// Proper detaching process
	while (current != NULL) {
		currentNode = getNode(current);
		tmp = getPreviousInSequence(current);

		destroyPassageMarker(current);

		if (getMarker(currentNode) == NULL) {
			destroyNode(currentNode, graph);
			if (previousNode != NULL)
				simplifyNodeConservative(getTwinNode
							 (previousNode),
							 &tmp);
			previousNode = NULL;
		} else if (previousNode == NULL) {
			simplifyNodeConservative(currentNode, &tmp);
			previousNode = currentNode;
		} else {
			decrementArc(currentNode, previousNode);
			simplifyNodeConservative(getTwinNode(previousNode),
						 &tmp);
			simplifyNodeConservative(currentNode, &tmp);
			previousNode = currentNode;
		}

		current = tmp;
	}

	if (previousNode != NULL)
		simplifyNode(getTwinNode(previousNode));
}

static void updateMembers(Node * bypass, Node * nextNode)
{
	PassageMarker *marker, *next, *tmp;
	Node *nextNextNode;
	Coordinate nextLength = getNodeLength(nextNode);

	// Remove unwanted arcs
	while (getArc(bypass) != NULL)
		destroyArc(getArc(bypass), graph);

	// Update  marker + arc info
	for (marker = getMarker(bypass); marker != NULL; marker = tmp) {
		tmp = getNextInNode(marker);

		if (!isTerminal(marker)) {
			next = getNextInSequence(marker);

			// FOR SHORT LONG READ MIX      
			if (getNode(next) != nextNode)
				destroyPassageMarker(marker);
			// END OF HACK

			nextNextNode = getNode(getNextInSequence(next));

			if (marker != path) {
				createArc(bypass, nextNextNode, graph);
				decrementArc(nextNode, nextNextNode);
			}

			disconnectNextPassageMarker(marker, graph);
			destroyPassageMarker(next);
		} else {
			incrementFinishOffset(marker, nextLength);
		}
	}
}

static void markParallelMarkers(PassageMarker * marker)
{
	PassageMarker *current;

	for (current = getPreviousInSequence(marker); current != NULL;
	     current = getPreviousInSequence(current))
		setPassageMarkerStatus(current, true);
}

static void removeDeadGroupies(PassageMarker ** groupies, Node * bypass)
{
	PassageMarker *current = *groupies;
	PassageMarker *next;

	while (getNextInNode(current) != NULL) {
		next = getNextInNode(current);
		if (isTerminal(next)) {
			//printf("Removing dead groupie %li %li\n", getPassageMarkerSequenceID(next), getPassageMarkerLength(next));
			setNextInNode(current, getNextInNode(next));
			markParallelMarkers(next);
			setPreviousInSequence(NULL, next);
			transposePassageMarker(next, bypass);
		} else
			current = next;
	}

	current = *groupies;
	if (isTerminal(current)) {
		//printf("Removing dead groupie %li %li\n", getPassageMarkerSequenceID(current), getPassageMarkerLength(current));
		*groupies = getNextInNode(current);
		markParallelMarkers(current);
		setPreviousInSequence(NULL, current);
		transposePassageMarker(current, bypass);
	}
}

static void removeDissidentGroupies(PassageMarker ** groupies,
				    Node * nextNode)
{
	PassageMarker *current = *groupies;
	PassageMarker *next;

	if (*groupies == NULL)
		return;

	while (getNextInNode(current) != NULL) {
		next = getNextInNode(current);
		if (!isDestinationToMarker(next, nextNode)) {
			setNextInNode(current, getNextInNode(next));
			destroyPassageMarker(next);
		} else
			current = next;
	}

	current = *groupies;
	if (current == NULL)
		return;

	if (!isDestinationToMarker(current, nextNode)) {
		*groupies = getNextInNode(current);
		destroyPassageMarker(current);
	}

}

static void updateGroupies(PassageMarker * groupies)
{
	PassageMarker *current;

	for (current = groupies; current != NULL;
	     current = getNextInNode(current)) {
		setPreviousInSequence(getNextInSequence(current), current);
		concatenatePassageMarkers(current,
					  getNextInSequence(current));
	}
}

static void admitGroupies(PassageMarker ** groupies, Node * bypass)
{
	PassageMarker *current = *groupies;
	PassageMarker *next;

	if (*groupies == NULL)
		return;

	while (getNextInNode(current) != NULL) {
		next = getNextInNode(current);
		if (getNextInSequence(next) == NULL
		    || getNode(getNextInSequence(next)) != NULL) {
			setNextInNode(current, getNextInNode(next));
			transposePassageMarker(next, bypass);
			detachRead(next);

		} else
			current = next;
	}

	current = *groupies;
	if (current == NULL)
		return;

	if (getNextInSequence(current) == NULL
	    || getNode(getNextInSequence(current)) != NULL) {
		*groupies = getNextInNode(current);
		transposePassageMarker(current, bypass);
		detachRead(current);
	}

	while (*groupies != NULL) {
		current = *groupies;
		*groupies = getNextInNode(current);
		destroyPassageMarker(current);
	}
}

static Node *bypass()
{
	Node *bypass = getNode(path);
	Node *next = NULL;
	PassageMarker *groupies = NULL;
	Coordinate oldLength;
	PassageMarker *marker;

	puts("Bypass start");

	PassageMarker *tmp;
	for (tmp = path; tmp != NULL; tmp = getNextInSequence(tmp))
		printf("Node %li\n", getNodeID(getNode(tmp)));
	printf
	    ("Bypass ID = %li, length = %li, coverage = %li, starts = %li, markers = %li\n",
	     getNodeID(bypass), getNodeLength(bypass),
	     readCoverage(bypass), readStarts(bypass),
	     strainMarkerCount(bypass, 1));

	for (marker = path; marker != NULL;
	     marker = getNextInSequence(marker))
		if (getNode(marker) != NULL)
			setNodeStatus(getNode(marker), true);

	// Update extensive variables (length + descriptors + passage markers)
	while (!isTerminal(path)) {
		next = getNode(getNextInSequence(path));

		if (next == NULL) {
			puts("Going over the edge");
			removeDeadGroupies(&groupies, bypass);
			if (groupies != NULL)
				abort();
			destroyPath();
			printf("Suicide bypass %li %li\n",
			       getNodeLength(bypass),
			       readCoverage(bypass));
			setNodeStatus(bypass, false);
			return bypass;
		}

		setNodeStatus(next, false);

		printf
		    ("Visiting node ID = %li, length = %li, coverage = %li, starts = %li, markers = %li\n",
		     getNodeID(next), getNodeLength(next),
		     readCoverage(next), readStarts(next),
		     strainMarkerCount(next, 1));

		// Overall node update 
		oldLength = getNodeLength(bypass);
		appendDescriptors(bypass, next);
		if (!getUniqueness(bypass) && getUniqueness(next))
			setUniqueness(bypass, true);

		// Updating passage markers:
		// Groupies
		removeDeadGroupies(&groupies, bypass);
		removeDissidentGroupies(&groupies, next);
		updateGroupies(groupies);
		selectGroupies(next, &groupies, oldLength);

		// Members
		updateMembers(bypass, next);

		// Clean data
		if (!isTerminal(path)) {
			if (getMarker(next) == NULL) {
				destroyNode(next, graph);
			} else {
				simplifyNode(next);
				simplifyNode(getTwinNode(next));
			}
		}
	}
	printf("Next %li Bypass %li\n", getNodeID(next),
	       getNodeID(bypass));

	// Pathological cases
	if (next == bypass || next == getTwinNode(bypass)) {
		puts("Next ==±bypass");
		abort();
	}
	// Sneaky marker to avoid the killing of the node
	addUncertainPassageMarker(0, next);

	// Remove unique groupies from original path 
	admitGroupies(&groupies, bypass);

	destroyNode(next, graph);

	// Clean up
	destroyPath();

	// Update
	//if (!getNodeStatus(bypass) && isUniqueFunction(bypass))
	//      setNodeStatus(bypass, true);
	//      setNodeStatus(bypass, isUniqueFunction(bypass));

	// DEBUG
	printf("Bypass length %li %li\n", getNodeLength(bypass),
	       readCoverage(bypass));

	simplifyNode(bypass);
	simplifyNode(getTwinNode(bypass));
	puts("Bypass finish main");
	setNodeStatus(bypass, false);

	if (!getUniqueness(bypass) && isUniqueFunction(bypass))
		setUniqueness(bypass, true);

	return bypass;
}

void readCoherentGraph(Graph * inGraph, boolean(*isUnique) (Node *),
		       double coverage)
{
	IDnum nodeIndex;
	Node *node;
	IDnum previousNodeCount = 0;

	graph = inGraph;
	isUniqueFunction = isUnique;
	listMemory = newRecycleBin(sizeof(PassageMarkerList), 100000);
	expected_coverage = coverage;

	puts("Read coherency...");
	reassessArcMultiplicities(graph);
	resetPassageMarkersStatus(graph);
	resetNodeStatus(graph);

	identifyUniqueNodes();

	previousNodeCount = 0;
	while (previousNodeCount != nodeCount(graph)) {

		previousNodeCount = nodeCount(graph);

		for (nodeIndex = 1; nodeIndex <= nodeCount(graph);
		     nodeIndex++) {

			node = getNodeInGraph(graph, nodeIndex);
			if (node == NULL)
				continue;

			if (!getUniqueness(node)) {
				while (standardNodesConnect(node))
					node = bypass();

				node = getTwinNode(node);

				while (standardNodesConnect(node))
					node = bypass();
			}
		}

		for (nodeIndex = 1; nodeIndex <= nodeCount(graph);
		     nodeIndex++) {

			node = getNodeInGraph(graph, nodeIndex);
			if (node == NULL)
				continue;

			if (getUniqueness(node)) {
				while (uniqueNodesConnect(node, isUnique))
					node = bypass();

				node = getTwinNode(node);

				while (uniqueNodesConnect(node, isUnique))
					node = bypass();
			}
		}

		concatenateGraph(graph);
	}

	destroyRecycleBin(listMemory);

	puts("Read coherency over!");
}
