/*
Copyright 2007 Daniel Zerbino (zerbino@ebi.ac.uk)

    This file is part of Velvet.

    Velvet is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Velvet is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Velvet; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "globals.h"
#include "graph.h"
#include "recycleBin.h"
#include "roadMap.h"
#include "passageMarker.h"
#include "concatenatedGraph.h"
#include "readSet.h"

// Internal structure used to mark the ends of an Annotation
struct insertionMarker_st {
	int direction;
	Coordinate position;
	Node *node;
	InsertionMarker *next;
};

static RecycleBin *markerMemory = NULL;
static const int BLOCKSIZE = 1000000;
static IDnum *referenceSequence;

static InsertionMarker *allocateInsertionMarker()
{
	if (markerMemory == NULL)
		markerMemory =
		    newRecycleBin(sizeof(InsertionMarker), BLOCKSIZE);

	return (InsertionMarker *) allocatePointer(markerMemory);
}

// Adds a new insert marker into the corresponding list (used in createInsertionMarker())
static void
insertIntoMarkerList(InsertionMarker * newMarker,
		     InsertionMarker ** insMarkers, IDnum sequenceID,
		     IDnum * markerCounters)
{
	newMarker->next = insMarkers[sequenceID];
	insMarkers[sequenceID] = newMarker;
	markerCounters[sequenceID]++;
}

// Merges two lists of annotations in order of increasing position (used in mergeSort mainly)
static InsertionMarker *mergeLists(InsertionMarker * left,
				   InsertionMarker * right)
{
	InsertionMarker *mergedList = NULL;
	InsertionMarker *tail = NULL;

	// Choose first element:
	if (left->position <= right->position) {
		mergedList = left;
		tail = left;
		left = left->next;
	} else {
		mergedList = right;
		tail = right;
		right = right->next;
	}

	// Iterate while both lists are still non empty
	while (left != NULL && right != NULL) {
		if (left->position <= right->position) {
			tail->next = left;
			left = left->next;
		} else {
			tail->next = right;
			right = right->next;
		}

		tail = tail->next;
	}

	// Concatenate the remaining list at the end of the merged list
	if (left != NULL)
		tail->next = left;

	if (right != NULL)
		tail->next = right;

	return mergedList;
}

// Good old merge sort for insertion markers
static void insertionMarkerMergeSort(InsertionMarker ** marker,
				     IDnum elems)
{
	IDnum half = elems / 2;
	InsertionMarker *left = *marker;
	InsertionMarker *ptr = left;
	InsertionMarker *right;
	IDnum index;

	if (elems == 0 || elems == 1)
		return;

	if (elems == 2) {
		if ((*marker)->position > (*marker)->next->position) {
			(*marker)->next->next = *marker;
			*marker = (*marker)->next;
			(*marker)->next->next = NULL;
		}
		return;
	}

	for (index = 0; index < half - 1; index++) {
		ptr = ptr->next;
		if (ptr == NULL)
			return;
	}

	right = ptr->next;
	ptr->next = NULL;

	insertionMarkerMergeSort(&left, half);
	insertionMarkerMergeSort(&right, elems - half);
	*marker = mergeLists(left, right);
}

// Applies mergeSort to each insertion marker list (in order of position)
static void
orderInsertionMarkers(InsertionMarker ** insMarkers,
		      IDnum * markerCounters, Graph * graph)
{
	IDnum sequenceIndex;
	IDnum sequenceCounter = sequenceCount(graph);

	puts("Ordering insertion markers");
	for (sequenceIndex = 1; sequenceIndex <= sequenceCounter;
	     sequenceIndex++) {
		//printf("Ordering sequence %li\n", sequenceIndex);
		insertionMarkerMergeSort(&(insMarkers[sequenceIndex]),
					 markerCounters[sequenceIndex]);
	}
}

// Creates insertion marker lists 
static void
setInsertionMarkers(RoadMap ** rdmaps, Graph * newGraph,
		    InsertionMarker ** insMarkers)
{
	IDnum sequenceCounter = sequenceCount(newGraph);
	IDnum sequenceIndex;
	IDnum *markerCounters = calloc(sequenceCounter + 1, sizeof(IDnum));
	Annotation *annot;
	InsertionMarker *newMarker;

	for (sequenceIndex = 1; sequenceIndex < sequenceCounter + 1;
	     sequenceIndex++) {
		//printf("Going through sequence %li\n", sequenceIndex);

		// Set insertion markers in previous sequences :
		if (rdmaps[sequenceIndex - 1] == NULL) {
			printf("Sequence index %li\n", sequenceIndex);
			exit(1);
		}
		annot = getAnnotation(rdmaps[sequenceIndex - 1]);
		while (annot != NULL) {
			// Starting marker:
			newMarker = allocateInsertionMarker();
			newMarker->position = getStart(annot);
			setStartMarker(annot, newMarker);

			if (getAnnotSequenceID(annot) > 0) {
				newMarker->direction = 1;
				insertIntoMarkerList(newMarker, insMarkers,
						     getAnnotSequenceID
						     (annot),
						     markerCounters);
			} else {
				newMarker->direction = -2;
				newMarker->position++;
				insertIntoMarkerList(newMarker, insMarkers,
						     -getAnnotSequenceID
						     (annot),
						     markerCounters);
			}

			// Finishing marker:
			newMarker = allocateInsertionMarker();
			newMarker->position = getFinish(annot);
			setFinishMarker(annot, newMarker);

			if (getAnnotSequenceID(annot) < 0) {
				newMarker->position++;
				newMarker->direction = 2;
				insertIntoMarkerList(newMarker, insMarkers,
						     -getAnnotSequenceID
						     (annot),
						     markerCounters);
			} else {
				newMarker->direction = -1;
				insertIntoMarkerList(newMarker, insMarkers,
						     getAnnotSequenceID
						     (annot),
						     markerCounters);
			}

			annot = getNextAnnotation(annot);
		}		// End of annotation list
	}			// End of sequence

	orderInsertionMarkers(insMarkers, markerCounters, newGraph);
	free(markerCounters);
}

// Reconnects each insertion marker to the appropriate node (not the default positive strand)
static void
correctInsertionMarkerPointers(Graph * graph,
			       InsertionMarker ** insMarkers,
			       Node ** chains)
{
	IDnum sequenceIndex;
	InsertionMarker *marker;

	for (sequenceIndex = 1; sequenceIndex < sequenceCount(graph) + 1;
	     sequenceIndex++) {
		marker = insMarkers[sequenceIndex];
		while (marker != NULL) {
			if (marker->node == NULL)
				marker->node = chains[sequenceIndex];
			else if (marker->direction > 0)
				marker->node =
				    getNodeInGraph(graph,
						   getNodeID(marker->
							     node) + 1);

			if (marker->direction == 2
			    || marker->direction == -2)
				marker->node = getTwinNode(marker->node);
			marker = marker->next;
		}
	}
}

// Counts how many nodes are to be created to allocate appropriate memory
static void
countNodes(RoadMap ** rdmaps, Graph * newGraph,
	   InsertionMarker ** insMarkers)
{
	Annotation *annot;
	InsertionMarker *currentMarker;
	IDnum sequenceIndex;
	Coordinate currentPosition, nextStop;
	IDnum nodeCounter = 0;

	// Now that we have read all of the annotations, we go on to create the nodes and tie them up
	for (sequenceIndex = 1; sequenceIndex <= sequenceCount(newGraph);
	     sequenceIndex++) {
		annot = getAnnotation(rdmaps[sequenceIndex - 1]);
		currentMarker = insMarkers[sequenceIndex];
		currentPosition = 0;

		while (annot != NULL) {
			if (currentMarker == NULL
			    || getPosition(annot) <=
			    currentMarker->position)
				nextStop = getPosition(annot);
			else
				nextStop = currentMarker->position;

			if (currentPosition != nextStop) {
				nodeCounter++;
				currentPosition = nextStop;
			}

			while (currentMarker != NULL
			       && currentMarker->position ==
			       currentPosition)
				currentMarker = currentMarker->next;

			while (annot != NULL
			       && getPosition(annot) == currentPosition)
				annot = getNextAnnotation(annot);

		}

		while (currentMarker != NULL) {
			if (currentPosition == currentMarker->position)
				currentMarker = currentMarker->next;
			else {
				nodeCounter++;
				currentPosition = currentMarker->position;
			}
		}

		if (currentPosition <
		    getMapLength(rdmaps[sequenceIndex - 1]))
			nodeCounter++;

	}

	allocateNodeSpace(newGraph, nodeCounter);
	referenceSequence = malloc((nodeCounter + 1) * sizeof(IDnum));
}


// Creates the node using insertion marker and annotation lists for each sequence
static void
createNodes(RoadMap ** rdmaps, Graph * newGraph,
	    InsertionMarker ** insMarkers, Node ** chains,
	    TightString ** seqs, int WORDLENGTH)
{
	Annotation *annot;
	Node *latestNode;
	InsertionMarker *currentMarker;
	IDnum sequenceIndex;
	Coordinate currentPosition, nextStop, offset;
	IDnum nodeCounter = 1;

	// Now that we have read all of the annotations, we go on to create the nodes and tie them up
	for (sequenceIndex = 1; sequenceIndex <= sequenceCount(newGraph);
	     sequenceIndex++) {
		if (sequenceIndex % 100000 == 0)
			printf("Sequence %li / %li\n", sequenceIndex,
			       sequenceCount(newGraph));

		annot = getAnnotation(rdmaps[sequenceIndex - 1]);
		currentMarker = insMarkers[sequenceIndex];
		currentPosition = 0;
		offset = 0;
		latestNode = NULL;

		while (annot != NULL) {
			if (currentMarker == NULL
			    || getPosition(annot) <=
			    currentMarker->position)
				nextStop = getPosition(annot);
			else {
				nextStop = currentMarker->position;
			}

			if (currentPosition != nextStop) {
				addNodeToGraph(newGraph,
					       newNode(sequenceIndex,
						       currentPosition,
						       nextStop, offset,
						       nodeCounter, seqs,
						       WORDLENGTH));
				referenceSequence[nodeCounter] =
				    sequenceIndex;
				if (latestNode == NULL) {
					chains[sequenceIndex] =
					    getNodeInGraph(newGraph,
							   nodeCounter);
				}
				latestNode =
				    getNodeInGraph(newGraph, nodeCounter);
				nodeCounter++;
				currentPosition = nextStop;
			}

			while (currentMarker != NULL
			       && currentMarker->position == nextStop) {
				currentMarker->node = latestNode;
				currentMarker = currentMarker->next;
			}

			while (annot != NULL
			       && getPosition(annot) == nextStop) {
				offset += getAnnotationLength(annot);
				annot = getNextAnnotation(annot);
			}

		}

		while (currentMarker != NULL) {
			if (currentPosition == currentMarker->position) {
				currentMarker->node = latestNode;
				currentMarker = currentMarker->next;
			} else {
				nextStop = currentMarker->position;
				addNodeToGraph(newGraph,
					       newNode(sequenceIndex,
						       currentPosition,
						       nextStop, offset,
						       nodeCounter, seqs,
						       WORDLENGTH));
				referenceSequence[nodeCounter] =
				    sequenceIndex;
				if (latestNode == NULL) {
					chains[sequenceIndex] =
					    getNodeInGraph(newGraph,
							   nodeCounter);
				}
				latestNode =
				    getNodeInGraph(newGraph, nodeCounter);
				nodeCounter++;
				currentPosition = currentMarker->position;
			}
		}

		if (currentPosition <
		    getMapLength(rdmaps[sequenceIndex - 1])) {
			addNodeToGraph(newGraph,
				       newNode(sequenceIndex,
					       currentPosition,
					       getMapLength(rdmaps
							    [sequenceIndex
							     - 1]), offset,
					       nodeCounter, seqs,
					       WORDLENGTH));
			referenceSequence[nodeCounter] = sequenceIndex;
			if (latestNode == NULL)
				chains[sequenceIndex] =
				    getNodeInGraph(newGraph, nodeCounter);
			nodeCounter++;
		}

	}
	correctInsertionMarkerPointers(newGraph, insMarkers, chains);
}

static void goThroughNode(Node * nextNode, Graph * graph)
{
	if (nextNode == NULL)
		return;

	if (getNodeStatus(nextNode))
		return;

	incrementReadStartCount(nextNode, graph);
	setSingleNodeStatus(nextNode, true);
}

static void connectNodeToTheNext(Node ** currentNode, Node * nextNode,
				 Coordinate * currentPosition,
				 PassageMarker ** latestPassageMarker,
				 IDnum sequenceIndex, Category category,
				 Graph * graph)
{
	if (nextNode == NULL)
		return;

	if (*currentNode != NULL)
		createArc(*currentNode, nextNode, graph);

	*currentNode = nextNode;

	if (category / 2 >= CATEGORIES) {
		addPassageMarker(sequenceIndex, *currentPosition,
				 *currentNode);
		connectPassageMarkers(*latestPassageMarker,
				      getMarker(*currentNode), graph);
		*latestPassageMarker = getMarker(*currentNode);
	} else {
		if (category % 2 != 0)
			addReadStart(*currentNode, sequenceIndex, graph);
		incrementVirtualCoverage(*currentNode, category / 2,
					 getNodeLength(*currentNode));
	}

	*currentPosition += getNodeLength(*currentNode);

}

static Node *chooseNextInternalNode(Node * currentNode,
				    IDnum sequenceIndex, Graph * graph)
{
	if (getNodeID(currentNode) < nodeCount(graph)
	    &&
	    referenceSequence[getNodeID(currentNode) + 1] == sequenceIndex)
		return getNodeInGraph(graph, getNodeID(currentNode) + 1);
	else
		return NULL;

}

static void unlockAnnotation(Annotation * annot, Graph * graph)
{
	Node *currentNode = getStartMarker(annot)->node;
	setSingleNodeStatus(currentNode, false);

	while (currentNode != getFinishMarker(annot)->node) {
		if (getAnnotSequenceID(annot) > 0) {
			currentNode =
			    getNodeInGraph(graph,
					   getNodeID(currentNode) + 1);
		} else {
			currentNode =
			    getTwinNode(getNodeInGraph
					(graph,
					 -getNodeID(currentNode) - 1));
		}

		setSingleNodeStatus(currentNode, false);
	}
}

static void goThroughAnnotation(Annotation * annot, Graph * graph)
{
	Node *currentNode = getStartMarker(annot)->node;

	goThroughNode(currentNode, graph);

	while (currentNode != getFinishMarker(annot)->node) {
		if (getAnnotSequenceID(annot) > 0) {
			currentNode =
			    getNodeInGraph(graph,
					   getNodeID(currentNode) + 1);
		} else {
			currentNode =
			    getTwinNode(getNodeInGraph
					(graph,
					 -getNodeID(currentNode) - 1));
		}

		goThroughNode(currentNode, graph);
	}
}

static void connectAnnotation(Node ** currentNode, Annotation * annot,
			      Coordinate * currentPosition,
			      PassageMarker ** latestPassageMarker,
			      IDnum sequenceIndex, Category category,
			      Graph * graph)
{
	Node *nextNode;

	nextNode = getStartMarker(annot)->node;

	connectNodeToTheNext(currentNode, nextNode, currentPosition,
			     latestPassageMarker, sequenceIndex, category,
			     graph);

	while (*currentNode != getFinishMarker(annot)->node) {
		if (getAnnotSequenceID(annot) > 0) {
			nextNode =
			    getNodeInGraph(graph,
					   getNodeID(*currentNode) + 1);
		} else {
			nextNode =
			    getTwinNode(getNodeInGraph
					(graph,
					 -getNodeID(*currentNode) - 1));
		}

		connectNodeToTheNext(currentNode, nextNode,
				     currentPosition, latestPassageMarker,
				     sequenceIndex, category, graph);
	}
}

// Threads each sequences and creates arcs according to road map indications
static void connectNodes(RoadMap ** rdmaps, Graph * newGraph,
			 Node ** chains)
{
	Coordinate currentPosition, currentInternalPosition;
	IDnum sequenceIndex;
	Annotation *annot;
	Node *currentNode, *nextInternalNode;
	PassageMarker *latestPassageMarker;
	Category category;

	// First quick pass to determine array sizes:
	for (sequenceIndex = 1; sequenceIndex <= sequenceCount(newGraph);
	     sequenceIndex++) {
		category = getRoadMapCategory(rdmaps[sequenceIndex - 1]);
		if (category / 2 >= CATEGORIES || category % 2 == 0)
			continue;

		if (!readStartsAreActivated(newGraph))
			activateReadStarts(newGraph);

		// Incrementing the counters
		annot = getAnnotation(rdmaps[sequenceIndex - 1]);
		while (annot != NULL) {
			goThroughAnnotation(annot, newGraph);
			annot = getNextAnnotation(annot);
		}

		nextInternalNode = chains[sequenceIndex];
		while (nextInternalNode != NULL) {
			goThroughNode(nextInternalNode, newGraph);
			nextInternalNode =
			    chooseNextInternalNode(nextInternalNode,
						   sequenceIndex,
						   newGraph);
		}

		// Cleaning boolean array
		annot = getAnnotation(rdmaps[sequenceIndex - 1]);
		while (annot != NULL) {
			unlockAnnotation(annot, newGraph);
			annot = getNextAnnotation(annot);
		}

		nextInternalNode = chains[sequenceIndex];
		while (nextInternalNode != NULL) {
			setSingleNodeStatus(nextInternalNode, false);
			nextInternalNode =
			    chooseNextInternalNode(nextInternalNode,
						   sequenceIndex,
						   newGraph);
		}
	}

	createNodeReadStartArrays(newGraph);

	// Second more complete pass to actually enter read info
	for (sequenceIndex = 1; sequenceIndex <= sequenceCount(newGraph);
	     sequenceIndex++) {

		if (sequenceIndex % 100000 == 0)
			printf("Connecting %li\n", sequenceIndex);

		category = getRoadMapCategory(rdmaps[sequenceIndex - 1]);
		annot = getAnnotation(rdmaps[sequenceIndex - 1]);

		if (annot == NULL
		    && getMapLength(rdmaps[sequenceIndex - 1]) == 0)
			continue;

		currentPosition = 0;
		currentInternalPosition = 0;
		currentNode = NULL;
		latestPassageMarker = NULL;
		nextInternalNode = chains[sequenceIndex];

		// Recursion up to last annotation
		while (annot != NULL || nextInternalNode != NULL) {
			if (annot == NULL
			    || (nextInternalNode != NULL
				&& currentInternalPosition <
				getPosition(annot))) {
				connectNodeToTheNext(&currentNode,
						     nextInternalNode,
						     &currentPosition,
						     &latestPassageMarker,
						     sequenceIndex,
						     category, newGraph);
				nextInternalNode =
				    chooseNextInternalNode(currentNode,
							   sequenceIndex,
							   newGraph);
				currentInternalPosition +=
				    getNodeLength(currentNode);

			} else {
				connectAnnotation(&currentNode, annot,
						  &currentPosition,
						  &latestPassageMarker,
						  sequenceIndex, category,
						  newGraph);
				annot = getNextAnnotation(annot);
			}
		}

		// Cleaning boolean array
		annot = getAnnotation(rdmaps[sequenceIndex - 1]);
		while (annot != NULL) {
			unlockAnnotation(annot, newGraph);
			annot = getNextAnnotation(annot);
		}

		nextInternalNode = chains[sequenceIndex];
		while (nextInternalNode != NULL) {
			setSingleNodeStatus(nextInternalNode, false);
			nextInternalNode =
			    chooseNextInternalNode(nextInternalNode,
						   sequenceIndex,
						   newGraph);
		}
	}
}

// Post construction memory deallocation routine (of sorts, could certainly be optimized)
static void
cleanUpMemory(Graph * graph, RoadMap ** rdmaps,
	      InsertionMarker ** insMarkers, Node ** chains)
{
	// Killing off roadmaps
	destroyAllRoadMaps();
	free(rdmaps);

	// Dealing with insertion markers
	if (markerMemory != NULL)
		destroyRecycleBin(markerMemory);
	markerMemory = NULL;

	// Finishing off the chain markers
	free(chains);
}

// The full monty, wrapped up in one function
Graph *newGraph(RoadMapArray * rdmapArray, ReadSet * reads)
{
	int WORDLENGTH = rdmapArray->WORDLENGTH;
	RoadMap **rdmaps = rdmapArray->array;
	IDnum sequenceCount = reads->readCount;
	TightString **seqs = reads->tSequences;
	InsertionMarker **insMarkers =
	    calloc((sequenceCount + 1), sizeof(InsertionMarker *));
	Node **chains = calloc((sequenceCount + 1), sizeof(Node *));

	Graph *newGraph =
	    emptyGraph(sequenceCount, rdmapArray->WORDLENGTH);
	puts("Creating insertion markers");
	setInsertionMarkers(rdmaps, newGraph, insMarkers);
	puts("Counting nodes");
	countNodes(rdmaps, newGraph, insMarkers);
	printf("%li nodes counted, creating them now\n",
	       nodeCount(newGraph));
	createNodes(rdmaps, newGraph, insMarkers, chains, seqs,
		    WORDLENGTH);
	puts("Connecting nodes");
	connectNodes(rdmaps, newGraph, chains);
	puts("Concatenating graph");
	concatenateGraph(newGraph);
	puts("Cleaning up memory");
	cleanUpMemory(newGraph, rdmaps, insMarkers, chains);
	puts("Done creating graph");
	return newGraph;
}
