/*
Copyright 2007 Daniel Zerbino (zerbino@ebi.ac.uk)

    This file is part of Velvet.

    Velvet is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Velvet is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Velvet; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#ifndef _GRAPH_H_
#define _GRAPH_H_

void testGraph();

////////////////////////////////////////////////////////////
// Node functions
////////////////////////////////////////////////////////////

//Creators/destructor
Node *newNode(IDnum sequenceID, Coordinate start, Coordinate finish,
	      Coordinate offset, IDnum ID, TightString ** sequences,
	      int WORDLENGTH);
Node *emptyNode();
void destroyNode(Node * node, Graph * graph);

// Locator
IDnum getNodeID(Node * node);
Node *getNodeInGraph(Graph * graph, IDnum nodeID);
Node *getTwinNode(Node * node);

// Arc info 
int arcCount(Node * node);
int simpleArcCount(Node * node);
Arc *getArc(Node * node);
boolean hasSingleArc(Node * node);

// Descriptor
Coordinate getNodeLength(Node * node);
void appendDescriptors(Node * target, Node * source);
void addBufferToDescriptor(Node * node, Coordinate length);
void splitNodeDescriptor(Node * source, Node * target, Coordinate offset);
void clipNodeLength(Node * node, Coordinate startClip,
		    Coordinate finishClip);
void reduceNode(Node * node);

// Passage markers
void insertPassageMarker(PassageMarker * marker, Node * destination);
PassageMarker *getMarker(Node * node);
void setMarker(Node * node, PassageMarker * marker);
IDnum markerCount(Node * node);

// Read starts
void incrementReadStartCount(Node * node, Graph * graph);
void addReadStart(Node * node, IDnum seqID, Graph * graph);
void spreadReadIDs(IDnum * reads, IDnum readCount, Node * node,
		   Graph * graph);
void transferReadStarts(Node * target, Node * source, Graph * graph);
void shareReadStarts(Node * target, Node * source, Graph * graph);
// Fridge :
IDnum *commonNodeReads(Node * nodeA, Node * nodeB, Graph * graph,
		       IDnum * length);
void mergeNodeReads(Node * nodeA, Node * nodeB, Graph * graph);

// Virtual coverage
void setVirtualCoverage(Node * node, Category category,
			Coordinate coverage);
void incrementVirtualCoverage(Node * node, Category category,
			      Coordinate coverage);
Coordinate getVirtualCoverage(Node * node, Category category);

// Status
void setNodeStatus(Node * node, boolean status);
void setSingleNodeStatus(Node * node, boolean status);
boolean getNodeStatus(Node * node);

// Uniqueness
void setUniqueness(Node * node, boolean value);
boolean getUniqueness(Node * node);

// IO
char *readNode(Node * node);
TightString *expandNode(Node * node, int WORDLENGTH);
void appendNodeSequence(Node * node, TightString * sequence,
			Coordinate writeIndex);

////////////////////////////////////////////////////////////
// Arc functions
////////////////////////////////////////////////////////////

// Creators/destructor
Arc *createArc(Node * origin, Node * destination, Graph * graph);
void createAnalogousArc(Node * origin, Node * destination, Arc * refArc,
			Graph * graph);
void destroyArc(Arc * arc, Graph * graph);

// Multiplicity
void setMultiplicity(Arc * arc, IDnum mult);
IDnum getMultiplicity(Arc * arc);
void changeMultiplicity(Arc * arc, IDnum variation);

// Extremities
Node *getOrigin(Arc * arc);
Node *getDestination(Arc * arc);

// Finding arcs
Arc *getArcBetweenNodes(Node * origin, Node * destination, Graph * graph);
Arc *getNextArc(Arc * arc);

// Lookup table option
void activateArcLookupTable(Graph * graph);
void deactivateArcLookupTable(Graph * graph);

////////////////////////////////////////////////////////////
// Graph functions
////////////////////////////////////////////////////////////

// Memory allocation
Graph *emptyGraph(IDnum sequenceCount, int wordLength);
void allocateNodeSpace(Graph * graph, IDnum nodeCount);
void addNodeToGraph(Graph * graph, Node * node);
Node *addEmptyNodeToGraph(Graph * graph, IDnum nodeID);

// Dimensions
IDnum nodeCount(Graph * graph);
IDnum sequenceCount(Graph * graph);
void renumberNodes(Graph * graph);

// Element status 
void resetNodeStatus(Graph * graph);
void resetPassageMarkersStatus(Graph * graph);

// Arc mults
void reassessArcMultiplicities(Graph * graph);

// File IO
void displayGraph(Graph * graph);
Graph *importGraph(char *filename);
void exportGraph(char *filename, Graph * graph, TightString ** sequences);
void exportDOTGraph(char *filename, Graph * graph);

// Read starts
void activateReadStarts(Graph * graph);
boolean readStartsAreActivated(Graph * graph);
void createNodeReadStartArrays(Graph * graph);
IDnum **getNodeToReadMappings(Graph * graph);
IDnum *getNodeReadCounts(Graph * graph);

int getWordLength(Graph * graph);
#endif
