/*
Copyright 2007 Daniel Zerbino (zerbino@ebi.ac.uk)

    This file is part of Velvet.

    Velvet is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Velvet is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Velvet; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "globals.h"
#include "graph.h"
#include "recycleBin.h"
#include "tightString.h"
#include "passageMarker.h"

#define ADENINE 0
#define CYTOSINE 1
#define GUANINE 2
#define THYMINE 3

typedef unsigned char Descriptor;
typedef struct read_list_st ReadList;

struct arc_st {
	IDnum multiplicity;
	Node *destination;
	Arc *twinArc;
	Arc *next;
	Arc *previous;
	Arc *nextInLookupTable;
};

struct node_st {
	IDnum ID;
	Node *twinNode;
	Arc *arc;
	IDnum arcCount;
	Descriptor *descriptor;
	Coordinate length;
	PassageMarker *marker;
	boolean status;
	boolean uniqueness;
	Coordinate virtualCoverage[CATEGORIES];
};

struct graph_st {
	IDnum sequenceCount;
	IDnum nodeCount;
	Node **nodes;
	Arc **arcLookupTable;
	IDnum **nodeReads;
	IDnum *nodeReadCounts;
	int wordLength;
};

static RecycleBin *arcMemory = NULL;
static RecycleBin *nodeMemory = NULL;

#define BLOCKSIZE 50

Arc *allocateArc()
{
	if (arcMemory == NULL)
		arcMemory = newRecycleBin(sizeof(Arc), BLOCKSIZE);

	return allocatePointer(arcMemory);
}

void deallocateArc(Arc * arc)
{
	deallocatePointer(arcMemory, arc);
}

Node *allocateNode()
{
	if (nodeMemory == NULL)
		nodeMemory = newRecycleBin(sizeof(Node), BLOCKSIZE);

	return (Node *) allocatePointer(nodeMemory);
}

void deallocateNode(Node * node)
{
	deallocatePointer(nodeMemory, node);
}

// Returns the twin node of a given node
Node *getTwinNode(Node * node)
{
	return node->twinNode;
}

// Inserts new passage marker in the marker list of destination node
void insertPassageMarker(PassageMarker * marker, Node * destination)
{
	setTopOfTheNode(marker);
	setNextInNode(marker, destination->marker);
	destination->marker = marker;
}

// Returns the length of the node's descriptor list
Coordinate getNodeLength(Node * node)
{
	return node->length;
}

// Returns the number of nodes in the graph
IDnum nodeCount(Graph * graph)
{
	return graph->nodeCount;
}

// returns the number of sequences used to buid the graph
IDnum sequenceCount(Graph * graph)
{
	return graph->sequenceCount;
}

// Creates an arc from node origin to node destination.
// If this arc already exists, increments its multiplicity by 1.
Arc *createArc(Node * originNode, Node * destinationNode, Graph * graph)
{
	Arc *arc, *twinArc;
	Node *destinationTwin;
	IDnum lookupIndex;

	if (originNode == NULL || destinationNode == NULL)
		return NULL;

//      printf("Connecting nodes %i -> %i\n", originNode->ID, destinationNode->ID);

	arc = getArcBetweenNodes(originNode, destinationNode, graph);

	if (arc != NULL) {
		arc->multiplicity++;
		arc->twinArc->multiplicity++;
		return arc;
	}
	// If not found
	arc = allocateArc();
	arc->destination = destinationNode;
	arc->multiplicity = 1;
	arc->previous = NULL;
	arc->next = originNode->arc;
	if (originNode->arc != NULL)
		originNode->arc->previous = arc;
	originNode->arc = arc;
	originNode->arcCount++;

	destinationTwin = destinationNode->twinNode;

	// Hairpin case
	if (destinationTwin == originNode) {
		arc->multiplicity++;
		arc->twinArc = arc;
		if (graph->arcLookupTable != NULL) {
			lookupIndex =
			    2 * originNode->ID + destinationNode->ID +
			    3 * graph->nodeCount;
			arc->nextInLookupTable =
			    graph->arcLookupTable[lookupIndex];
			graph->arcLookupTable[lookupIndex] = arc;
		}
		return arc;
	}

	twinArc = allocateArc();
	twinArc->destination = originNode->twinNode;
	twinArc->multiplicity = 1;
	twinArc->previous = NULL;
	twinArc->next = destinationTwin->arc;
	if (destinationTwin->arc != NULL)
		destinationTwin->arc->previous = twinArc;
	destinationTwin->arc = twinArc;
	destinationTwin->arcCount++;

	arc->twinArc = twinArc;
	twinArc->twinArc = arc;

	if (graph->arcLookupTable != NULL) {
		lookupIndex =
		    2 * originNode->ID + destinationNode->ID +
		    3 * graph->nodeCount;
		arc->nextInLookupTable =
		    graph->arcLookupTable[lookupIndex];
		graph->arcLookupTable[lookupIndex] = arc;

		lookupIndex =
		    -2 * destinationNode->ID - originNode->ID +
		    3 * graph->nodeCount;
		twinArc->nextInLookupTable =
		    graph->arcLookupTable[lookupIndex];
		graph->arcLookupTable[lookupIndex] = twinArc;
	}
	return arc;
}

void createAnalogousArc(Node * originNode, Node * destinationNode,
			Arc * refArc, Graph * graph)
{
	Arc *arc, *twinArc;
	Node *destinationTwin;
	IDnum lookupIndex;

	if (originNode == NULL || destinationNode == NULL)
		return;

//      printf("Connecting nodes %i -> %i\n", originNode->ID, destinationNode->ID);

	arc = getArcBetweenNodes(originNode, destinationNode, graph);

	if (arc != NULL) {
		if (refArc->twinArc != refArc) {
			arc->multiplicity += getMultiplicity(refArc);
			arc->twinArc->multiplicity +=
			    getMultiplicity(refArc);
		} else {
			arc->multiplicity += getMultiplicity(refArc) / 2;
			arc->twinArc->multiplicity +=
			    getMultiplicity(refArc) / 2;
		}
		return;
	}
	// If not found
	arc = allocateArc();
	arc->destination = destinationNode;
	arc->multiplicity = getMultiplicity(refArc);
	arc->previous = NULL;
	arc->next = originNode->arc;
	if (originNode->arc != NULL)
		originNode->arc->previous = arc;
	originNode->arc = arc;
	originNode->arcCount++;

	destinationTwin = destinationNode->twinNode;

	// Hairpin case
	if (destinationTwin == originNode) {
		arc->twinArc = arc;
		if (refArc->twinArc != refArc)
			arc->multiplicity *= 2;

		if (graph->arcLookupTable != NULL) {
			lookupIndex =
			    2 * originNode->ID + destinationNode->ID
			    + 3 * graph->nodeCount;
			arc->nextInLookupTable =
			    graph->arcLookupTable[lookupIndex];
			graph->arcLookupTable[lookupIndex] = arc;
		}
		return;
	}

	twinArc = allocateArc();
	twinArc->destination = originNode->twinNode;
	twinArc->multiplicity = getMultiplicity(refArc);
	twinArc->previous = NULL;
	twinArc->next = destinationTwin->arc;
	if (destinationTwin->arc != NULL)
		destinationTwin->arc->previous = twinArc;
	destinationTwin->arc = twinArc;
	destinationTwin->arcCount++;

	arc->twinArc = twinArc;
	twinArc->twinArc = arc;

	if (graph->arcLookupTable != NULL) {
		lookupIndex =
		    2 * originNode->ID + destinationNode->ID +
		    3 * graph->nodeCount;
		arc->nextInLookupTable =
		    graph->arcLookupTable[lookupIndex];
		graph->arcLookupTable[lookupIndex] = arc;

		lookupIndex =
		    -2 * destinationNode->ID - originNode->ID +
		    3 * graph->nodeCount;
		twinArc->nextInLookupTable =
		    graph->arcLookupTable[lookupIndex];
		graph->arcLookupTable[lookupIndex] = twinArc;
	}
}

void changeMultiplicity(Arc * arc, IDnum variation)
{
	if (arc == NULL)
		return;
	arc->multiplicity += variation;
	arc->twinArc->multiplicity += variation;
}

Arc *getArcBetweenNodes(Node * originNode, Node * destinationNode,
			Graph * graph)
{
	Arc *arc;
	Node *twinDestination, *twinOrigin;

	if (originNode == NULL || destinationNode == NULL)
		return NULL;

	if (graph->arcLookupTable != NULL) {
		for (arc =
		     graph->arcLookupTable[2 * originNode->ID +
					   destinationNode->ID +
					   3 * graph->nodeCount];
		     arc != NULL; arc = arc->nextInLookupTable) {
			if (arc->destination == destinationNode) {
				return arc;
			}
		}
		return NULL;
	}

	twinDestination = destinationNode->twinNode;
	if (originNode->arcCount <= twinDestination->arcCount) {
		for (arc = originNode->arc; arc != NULL; arc = arc->next)
			if (arc->destination == destinationNode)
				return arc;
		return NULL;
	}

	twinOrigin = originNode->twinNode;
	for (arc = twinDestination->arc; arc != NULL; arc = arc->next)
		if (arc->destination == twinOrigin)
			return arc->twinArc;
	return NULL;
}

void destroyArc(Arc * arc, Graph * graph)
{
	Node *origin, *destination;
	Arc *twinArc;
	Arc *currentArc;
	IDnum lookupIndex;

	if (arc == NULL)
		return;

	twinArc = arc->twinArc;
	origin = twinArc->destination->twinNode;
	destination = arc->destination->twinNode;

	//printf("Destroying arc %p\n", arc);

	// Removing arc from list
	if (origin->arc == arc) {
		origin->arc = arc->next;
		if (origin->arc != NULL)
			origin->arc->previous = NULL;
	} else {
		arc->previous->next = arc->next;
		if (arc->next != NULL)
			arc->next->previous = arc->previous;
	}

	origin->arcCount--;

	if (destination == origin) {
		if (graph->arcLookupTable != NULL) {
			lookupIndex =
			    2 * origin->ID - destination->ID +
			    3 * graph->nodeCount;
			currentArc = graph->arcLookupTable[lookupIndex];
			if (currentArc == arc)
				graph->arcLookupTable[lookupIndex] =
				    arc->nextInLookupTable;
			else {
				while (currentArc->nextInLookupTable !=
				       arc)
					currentArc =
					    currentArc->nextInLookupTable;

				currentArc->nextInLookupTable =
				    twinArc->nextInLookupTable;
			}
		}

		deallocateArc(arc);
		return;
	}
	// Removing arc's twin from list
	if (destination->arc == twinArc) {
		destination->arc = twinArc->next;
		if (destination->arc != NULL)
			destination->arc->previous = NULL;
	} else {
		twinArc->previous->next = twinArc->next;
		if (twinArc->next != NULL)
			twinArc->next->previous = twinArc->previous;
	}

	destination->arcCount--;

	if (graph->arcLookupTable != NULL) {
		lookupIndex =
		    2 * origin->ID - destination->ID +
		    3 * graph->nodeCount;
		currentArc = graph->arcLookupTable[lookupIndex];
		if (currentArc == arc)
			graph->arcLookupTable[lookupIndex] =
			    arc->nextInLookupTable;
		else {
			while (currentArc->nextInLookupTable != arc)
				currentArc = currentArc->nextInLookupTable;

			currentArc->nextInLookupTable =
			    arc->nextInLookupTable;
		}

		lookupIndex =
		    2 * destination->ID - origin->ID +
		    3 * graph->nodeCount;
		currentArc = graph->arcLookupTable[lookupIndex];
		if (currentArc == twinArc)
			graph->arcLookupTable[lookupIndex] =
			    twinArc->nextInLookupTable;
		else {
			while (currentArc->nextInLookupTable != twinArc)
				currentArc = currentArc->nextInLookupTable;

			currentArc->nextInLookupTable =
			    twinArc->nextInLookupTable;
		}
	}
	// Freeing memory
	deallocateArc(arc);
	deallocateArc(twinArc);
}

void destroyNode(Node * node, Graph * graph)
{
	Node *twin = node->twinNode;
	IDnum ID = node->ID;
	IDnum index;

	//printf("Destroying %li\n and twin %li\n", getNodeID(node), getNodeID(twin));

	if (ID < 0)
		ID = -ID;

	// Node arcs:
	while (node->arc != NULL)
		destroyArc(node->arc, graph);
	while (twin->arc != NULL)
		destroyArc(twin->arc, graph);

	// Descriptors
	free(node->descriptor);
	free(twin->descriptor);

	// Passage markers
	while (node->marker != NULL)
		destroyPassageMarker(node->marker);

	// Reads starts
	if (graph->nodeReads != NULL) {
		index = ID + graph->nodeCount;
		free(graph->nodeReads[index]);
		graph->nodeReads[index] = NULL;
		graph->nodeReadCounts[index] = 0;

		index = -ID + graph->nodeCount;
		free(graph->nodeReads[index]);
		graph->nodeReads[index] = NULL;
		graph->nodeReadCounts[index] = 0;
	}

	graph->nodes[ID] = NULL;
	deallocateNode(node);
	deallocateNode(twin);
}

int outDegree(Node * node)
{
	int result = 0;
	Arc *arc = node->arc;
	while (arc != NULL) {
		result += arc->multiplicity;
		arc = arc->next;
	}

	return result;
}

int simpleArcCount(Node * node)
{
	return node->arcCount;
}

int arcCount(Node * node)
{
	int result = 0;
	Arc *arc;

	if (node == NULL)
		return result;

	arc = node->arc;
	while (arc != NULL) {
		result++;
		if (arc->destination == node->twinNode)
			result++;
		arc = arc->next;
	}

	return result;

}

static Nucleotide getNucleotideInDescriptor(Descriptor * descriptor,
					    Coordinate i)
{
	Descriptor *fourMer = descriptor + i / 4;

	switch (i % 4) {
	case 0:
		return (*fourMer & 3);
	case 1:
		return (*fourMer & 12) >> 2;
	case 2:
		return (*fourMer & 48) >> 4;
	case 3:
		return (*fourMer & 192) >> 6;
	}
	return 0;
}

char *readNode(Node * node)
{
	char *s = malloc(10000 * sizeof(char));
	char tmpString[1000];
	Arc *arc = node->arc;
	Descriptor *descriptor;
	Nucleotide nucleotide;
	Coordinate i;

	sprintf(s, "NODE %li :", node->ID);

	for (i = 0; i < node->length; i++) {
		nucleotide = getNucleotideInDescriptor(descriptor, i);
		switch (nucleotide) {
		case ADENINE:
			tmpString[i] = 'A';
			break;
		case CYTOSINE:
			tmpString[i] = 'C';
			break;
		case GUANINE:
			tmpString[i] = 'G';
			break;
		case THYMINE:
			tmpString[i] = 'T';
			break;
		}
	}
	tmpString[i] = '\0';
	strcat(s, tmpString);

	while (arc != NULL) {
		sprintf(tmpString, " %li(%lix);", arc->destination->ID,
			arc->multiplicity);
		strcat(s, tmpString);
		arc = arc->next;
	}

	sprintf(tmpString, " lgth: %li", node->length);
	strcat(s, tmpString);

	return s;
}

void displayGraph(Graph * graph)
{
	Node *currentNode;
	IDnum nodeIndex;

	printf("%li sequences\n", graph->sequenceCount);
	printf("%li*2 nodes\n", graph->nodeCount);

	for (nodeIndex = 1; nodeIndex <= graph->nodeCount; nodeIndex++) {
		currentNode = graph->nodes[nodeIndex];
		printf("%s\n", readNode(currentNode));
		printf("%s\n", readNode(currentNode->twinNode));
	}
}

PassageMarker *getMarker(Node * node)
{
	return node->marker;
}

void setMarker(Node * node, PassageMarker * marker)
{
	if (node == NULL)
		return;

	if (marker == NULL) {
		node->marker = NULL;
		node->twinNode->marker = NULL;
		return;
	}

	node->marker = marker;
	setTopOfTheNode(marker);
	node->twinNode->marker = getTwinMarker(marker);
	setTopOfTheNode(getTwinMarker(marker));
}

void setNodeStatus(Node * node, boolean status)
{
	node->status = status;
	node->twinNode->status = status;
}

void setSingleNodeStatus(Node * node, boolean status)
{
	node->status = status;
}

boolean getNodeStatus(Node * node)
{
	if (node == NULL)
		return false;
	return node->status;
}

IDnum getNodeID(Node * node)
{
	if (node == NULL)
		return 0;

	return node->ID;
}

void resetNodeStatus(Graph * graph)
{
	IDnum nodeIndex;
	Node *node;

	for (nodeIndex = 1; nodeIndex <= graph->nodeCount; nodeIndex++) {
		node = graph->nodes[nodeIndex];
		if (node == NULL)
			continue;

		node->status = false;
		node->twinNode->status = false;
	}
}

void resetPassageMarkersStatus(Graph * graph)
{
	IDnum nodeIndex;
	Node *node;
	PassageMarker *marker;

	for (nodeIndex = 1; nodeIndex <= graph->nodeCount; nodeIndex++) {
		node = graph->nodes[nodeIndex];
		if (node == NULL)
			continue;

		for (marker = node->marker; marker != NULL;
		     marker = getNextInNode(marker))
			setPassageMarkerStatus(marker, false);
	}
}

Node *getNodeInGraph(Graph * graph, IDnum nodeID)
{
	// DEBUG
	if (nodeID > graph->nodeCount || nodeID < -graph->nodeCount) {
		printf
		    ("Required node does not exist !!! Node %li Max node ID %li\n",
		     nodeID, graph->nodeCount);
		abort();
	}
	// END OF DEBUG

	if (nodeID == 0)
		return NULL;
	else if (nodeID > 0)
		return graph->nodes[nodeID];
	else if (graph->nodes[-nodeID] == NULL)
		return NULL;
	else
		return graph->nodes[-nodeID]->twinNode;
}

Arc *getArc(Node * node)
{
	return node->arc;
}

Arc *getNextArc(Arc * arc)
{
	return arc->next;
}

IDnum getMultiplicity(Arc * arc)
{
	if (arc == NULL)
		return 0;

	return arc->multiplicity;
}

Node *getOrigin(Arc * arc)
{
	if (arc == NULL)
		return NULL;

	return arc->twinArc->destination->twinNode;
}

Node *getDestination(Arc * arc)
{
	if (arc == NULL)
		return NULL;

	return arc->destination;
}

IDnum markerCount(Node * node)
{
	IDnum count = 0;
	PassageMarker *marker;

	for (marker = getMarker(node); marker != NULL;
	     marker = getNextInNode(marker))
		count++;

	return count;
}

void appendNodeSequence(Node * node, TightString * sequence,
			Coordinate writeIndex)
{
	Coordinate i;
	Nucleotide nucleotide;

	//printf("Getting sequence from node %li of length %li (%li)\n", getNodeID(node), getNodeLength(node), getLength(nodeLabel));

	for (i = 0; i < getNodeLength(node); i++) {
		nucleotide =
		    getNucleotideInDescriptor(node->descriptor, i);
		writeNucleotideAtPosition(nucleotide, i + writeIndex,
					  sequence);
	}
}

static void writeNucleotideInDescriptor(Nucleotide nucleotide,
					Descriptor * descriptor,
					Coordinate i)
{
	Descriptor *fourMer = descriptor + i / 4;
	switch (i % 4) {
	case 3:
		*fourMer &= 63;
		*fourMer += nucleotide << 6;
		return;
	case 2:
		*fourMer &= 207;
		*fourMer += nucleotide << 4;
		return;
	case 1:
		*fourMer &= 243;
		*fourMer += nucleotide << 2;
		return;
	case 0:
		*fourMer &= 252;
		*fourMer += nucleotide;
	}
}

static inline Descriptor *mergeDescriptors(Descriptor * descr,
					   Coordinate destinationLength,
					   Descriptor * copy,
					   Coordinate sourceLength,
					   size_t arrayLength)
{
	Descriptor *readPtr, *writePtr;
	Descriptor readCopy;
	int readOffset, writeOffset;
	Descriptor *new = malloc(arrayLength * sizeof(Descriptor));
	Coordinate index;

	readPtr = descr;
	readCopy = *readPtr;
	writePtr = new;
	writeOffset = 0;
	for (index = 0; index < destinationLength; index++) {
		(*writePtr) >>= 2;
		(*writePtr) += (readCopy & 3) << 6;
		readCopy >>= 2;

		writeOffset++;
		if (writeOffset == 4) {
			writePtr++;
			readPtr++;
			readCopy = *readPtr;
			writeOffset = 0;
		}
	}

	readPtr = copy;
	readCopy = *readPtr;
	readOffset = 0;
	for (index = 0; index < sourceLength; index++) {
		(*writePtr) >>= 2;
		(*writePtr) += (readCopy & 3) << 6;
		readCopy >>= 2;

		writeOffset++;
		if (writeOffset == 4) {
			writePtr++;
			writeOffset = 0;
		}

		readOffset++;
		if (readOffset == 4) {
			readPtr++;
			readCopy = *readPtr;
			readOffset = 0;
		}
	}

	if (writeOffset != 0) {
		while (writeOffset != 4) {
			(*writePtr) >>= 2;
			writeOffset++;
		}
	}

	return new;
}

void addBufferToDescriptor(Node * node, Coordinate length)
{
	Descriptor *descr;
	Descriptor *twinDescr;
	Coordinate newLength;
	size_t arrayLength;
	Node *twinNode;
	Coordinate index;

	if (node == NULL)
		return;

	twinNode = node->twinNode;
	descr = node->descriptor;
	twinDescr = twinNode->descriptor;

	// Amendments for empty descriptors
	if (descr == NULL) {
		arrayLength = length / 4;
		if (length % 4 != 0)
			arrayLength++;

		node->descriptor = calloc(arrayLength, sizeof(Descriptor));
		node->length = length;
		twinNode->descriptor =
		    calloc(arrayLength, sizeof(Descriptor));;
		twinNode->length = length;
		return;
	}

	newLength = node->length + length;
	arrayLength = newLength / 4;
	if (newLength % 4 != 0)
		arrayLength++;

	// Merging forward descriptors
	node->descriptor =
	    realloc(node->descriptor, arrayLength * sizeof(Descriptor));
	for (index = node->length; index < newLength; index++)
		writeNucleotideInDescriptor(ADENINE, node->descriptor,
					    index);
	node->length = newLength;

	// Merging reverse descriptors
	twinNode->descriptor =
	    realloc(twinNode->descriptor,
		    arrayLength * sizeof(Descriptor));
	for (index = 0; index < twinNode->length; index++)
		writeNucleotideInDescriptor(getNucleotideInDescriptor
					    (twinNode->descriptor, index),
					    twinNode->descriptor,
					    index + length);
	for (index = 0; index < length; index++)
		writeNucleotideInDescriptor(THYMINE, twinNode->descriptor,
					    index);
	twinNode->length = newLength;

}

void appendDescriptors(Node * destination, Node * source)
{
	Descriptor *copy;
	Descriptor *twinCopy;
	Descriptor *descr;
	Descriptor *twinDescr;
	Coordinate newLength, destinationLength, sourceLength;
	size_t arrayLength;
	Descriptor *new;
	Node *twinDestination;

	if (source == NULL || destination == NULL)
		return;

	twinDestination = destination->twinNode;
	descr = destination->descriptor;
	twinDescr = twinDestination->descriptor;
	copy = source->descriptor;
	twinCopy = source->twinNode->descriptor;

	// Amendments for empty descriptors
	if (copy == NULL)
		return;
	if (descr == NULL) {
		destination->descriptor = copy;
		twinDestination->descriptor = twinCopy;
		source->descriptor = NULL;
		source->twinNode->descriptor = NULL;
		return;
	}

	destinationLength = destination->length;
	sourceLength = source->length;
	newLength = destinationLength + sourceLength;
	arrayLength = newLength / 4;
	if (newLength % 4 != 0)
		arrayLength++;

	// Merging forward descriptors
	new =
	    mergeDescriptors(descr, destinationLength, copy, sourceLength,
			     arrayLength);
	free(descr);
	destination->descriptor = new;
	destination->length = newLength;

	// Merging reverse descriptors
	new =
	    mergeDescriptors(twinCopy, sourceLength, twinDescr,
			     destinationLength, arrayLength);
	free(twinDescr);
	twinDestination->descriptor = new;
	twinDestination->length = newLength;
}

void setMultiplicity(Arc * arc, IDnum mult)
{
	arc->multiplicity = mult;
	arc->twinArc->multiplicity = mult;
}

// Reshuffles the graph->nodes array to remove NULL pointers
// Beware that node IDs are accordingly reshuffled (all pointers remain valid though)
void renumberNodes(Graph * graph)
{
	IDnum nodeIndex;
	Node *currentNode;
	IDnum counter = 0;
	IDnum nodes = graph->nodeCount;
	IDnum newIndex;

	puts("Renumbering nodes");
	printf("Initial node count %li\n", graph->nodeCount);

	for (nodeIndex = 1; nodeIndex <= nodes; nodeIndex++) {
		currentNode = getNodeInGraph(graph, nodeIndex);

		if (currentNode == NULL)
			counter++;
		else if (counter != 0) {
			newIndex = nodeIndex - counter;
			currentNode->ID = newIndex;
			currentNode->twinNode->ID = -newIndex;
			graph->nodes[newIndex] = currentNode;

			if (graph->nodeReads != NULL) {
				graph->nodeReads[newIndex + nodes] =
				    graph->nodeReads[nodeIndex + nodes];
				graph->nodeReadCounts[newIndex + nodes] =
				    graph->nodeReadCounts[nodeIndex +
							  nodes];
				graph->nodeReads[nodeIndex + nodes] = NULL;
				graph->nodeReadCounts[nodeIndex + nodes] =
				    0;

				graph->nodeReads[-newIndex + nodes] =
				    graph->nodeReads[-nodeIndex + nodes];
				graph->nodeReadCounts[-newIndex + nodes] =
				    graph->nodeReadCounts[-nodeIndex +
							  nodes];
				graph->nodeReads[-nodeIndex + nodes] =
				    NULL;
				graph->nodeReadCounts[-nodeIndex + nodes] =
				    0;
			}
		}
	}

	if (graph->nodeReads != NULL && counter != 0) {
		for (nodeIndex = counter; nodeIndex <= 2 * nodes - counter;
		     nodeIndex++) {
			graph->nodeReads[nodeIndex - counter] =
			    graph->nodeReads[nodeIndex];
			graph->nodeReadCounts[nodeIndex - counter] =
			    graph->nodeReadCounts[nodeIndex];
		}
	}

	graph->nodeCount -= counter;
	realloc(graph->nodes, (graph->nodeCount + 1) * sizeof(Node *));

	if (graph->nodeReads != NULL) {
		graph->nodeReads =
		    realloc(graph->nodeReads,
			    (2 * graph->nodeCount + 1) * sizeof(IDnum *));
		graph->nodeReadCounts =
		    realloc(graph->nodeReadCounts,
			    (2 * graph->nodeCount + 1) * sizeof(IDnum));
	}
	printf("Destroyed %li nodes\n", counter);
}

void splitNodeDescriptor(Node * source, Node * target, Coordinate offset)
{
	Coordinate originalLength = source->length;
	Coordinate backLength = originalLength - offset;
	Coordinate index;
	Descriptor *descriptor, *new;
	size_t arrayLength;
	Nucleotide nucleotide;

	source->length = offset;
	source->twinNode->length = offset;
	target->length = backLength;
	target->twinNode->length = backLength;

	free(target->descriptor);
	free(target->twinNode->descriptor);
	target->descriptor = NULL;
	target->twinNode->descriptor = NULL;

	if (backLength == 0)
		return;

	arrayLength = backLength / 4;
	if (backLength % 4 > 0)
		arrayLength++;

	// One way
	descriptor = source->descriptor;
	new = malloc(arrayLength * sizeof(Descriptor));
	target->descriptor = new;

	for (index = 0; index < backLength; index++) {
		nucleotide = getNucleotideInDescriptor(descriptor, index);
		writeNucleotideInDescriptor(nucleotide, new, index);
	}

	for (; index < originalLength; index++) {
		nucleotide = getNucleotideInDescriptor(descriptor, index);
		writeNucleotideInDescriptor(nucleotide, descriptor,
					    index - backLength);
	}

	// Same thing in the direction
	descriptor = source->twinNode->descriptor;
	new = malloc(arrayLength * sizeof(Descriptor));
	target->twinNode->descriptor = new;

	for (index = offset; index < originalLength; index++) {
		nucleotide = getNucleotideInDescriptor(descriptor, index);
		writeNucleotideInDescriptor(nucleotide, new,
					    index - offset);
	}
}

void reduceNode(Node * node)
{
	free(node->descriptor);
	node->descriptor = NULL;
	node->length = 0;

	free(node->twinNode->descriptor);
	node->twinNode->descriptor = NULL;
	node->twinNode->length = 0;
}

void checkPassageMarkersStatus(Graph * graph)
{
	IDnum nodeIndex;
	Node *node;
	PassageMarker *marker;

	for (nodeIndex = 1; nodeIndex <= graph->nodeCount; nodeIndex++) {
		node = graph->nodes[nodeIndex];
		if (node == NULL)
			continue;

		for (marker = node->marker; marker != NULL;
		     marker = getNextInNode(marker))
			if (getPassageMarkerStatus(marker)) {
				printf("TRUE marker %s\n",
				       readPassageMarker(marker));
				exit(-1);
			}
	}
}

void reassessArcMultiplicities(Graph * graph)
{
	IDnum index;
	Node *node, *twin;
	Arc *arc;
	PassageMarker *marker;

	for (index = 1; index <= graph->nodeCount; index++) {
		node = getNodeInGraph(graph, index);

		if (node == NULL)
			continue;

		for (arc = getArc(node); arc != NULL;
		     arc = getNextArc(arc))
			setMultiplicity(arc, 0);
		for (arc = getArc(getTwinNode(node)); arc != NULL;
		     arc = getNextArc(arc))
			setMultiplicity(arc, 0);
	}

	for (index = 1; index <= graph->nodeCount; index++) {
		node = getNodeInGraph(graph, index);

		if (node == NULL)
			continue;

		twin = getTwinNode(node);

		for (marker = getMarker(node); marker != NULL;
		     marker = getNextInNode(marker)) {
			if (getPassageMarkerSequenceID(marker) > 0
			    && !isTerminal(marker))
				createArc(node,
					  getNode(getNextInSequence
						  (marker)), graph);
			else if (getPassageMarkerSequenceID(marker) < 0
				 && !isInitial(marker))
				createArc(twin,
					  getTwinNode(getNode
						      (getPreviousInSequence
						       (marker))), graph);
		}

	}
}

// Allocate memory for an empty graph created with sequenceCount different sequences
Graph *emptyGraph(IDnum sequenceCount, int wordLength)
{
	Graph *newGraph = malloc(sizeof(Graph));
	newGraph->sequenceCount = sequenceCount;
	newGraph->arcLookupTable = NULL;
	newGraph->nodeReads = NULL;
	newGraph->nodeReadCounts = NULL;
	newGraph->wordLength = wordLength;
	return newGraph;
}

static Descriptor *newPositiveDescriptor(IDnum sequenceID,
					 Coordinate start,
					 Coordinate finish,
					 TightString ** sequences,
					 int WORDLENGTH)
{
	Coordinate index;
	Nucleotide nucleotide;
	TightString *tString = sequences[sequenceID - 1];
	Coordinate length = finish - start;
	Descriptor *res;
	size_t arrayLength = length / 4;

	if (length % 4 > 0)
		arrayLength++;

	res = malloc(arrayLength * sizeof(Descriptor));

	for (index = 0; index < length; index++) {
		nucleotide =
		    getNucleotide(start + index + WORDLENGTH - 1, tString);
		writeNucleotideInDescriptor(nucleotide, res, index);
	}

	return res;

}

static Descriptor *newNegativeDescriptor(IDnum sequenceID,
					 Coordinate start,
					 Coordinate finish,
					 TightString ** sequences,
					 int WORDLENGTH)
{
	Coordinate index;
	Nucleotide nucleotide;
	TightString *tString = sequences[-sequenceID - 1];
	Coordinate length = start - finish;
	Descriptor *res;
	size_t arrayLength = length / 4;

	if (length % 4 > 0)
		arrayLength++;

	res = malloc(arrayLength * sizeof(Descriptor));

	for (index = 0; index < length; index++) {
		nucleotide = getNucleotide(start - index, tString);
		writeNucleotideInDescriptor(3 - nucleotide, res, index);
	}

	return res;

}

static Descriptor *newDescriptor(IDnum sequenceID, Coordinate start,
				 Coordinate finish,
				 TightString ** sequences, int WORDLENGTH)
{
	if (sequenceID > 0)
		return newPositiveDescriptor(sequenceID, start, finish,
					     sequences, WORDLENGTH);
	else
		return newNegativeDescriptor(sequenceID, start, finish,
					     sequences, WORDLENGTH);
}

// Constructor
// Memory allocated
Node *newNode(IDnum sequenceID, Coordinate start, Coordinate finish,
	      Coordinate offset, IDnum ID, TightString ** sequences,
	      int WORDLENGTH)
{
	Node *newnd = allocateNode();
	Node *antiNode = allocateNode();
	Category cat;

	newnd->ID = ID;
	newnd->descriptor =
	    newDescriptor(sequenceID, start + offset, finish + offset,
			  sequences, WORDLENGTH);
	newnd->arc = NULL;
	newnd->arcCount = 0;
	newnd->marker = NULL;
	newnd->status = false;
	for (cat = 0; cat < CATEGORIES; cat++)
		newnd->virtualCoverage[cat] = 0;

	antiNode->ID = -ID;
	antiNode->descriptor =
	    newDescriptor(-sequenceID, finish + offset - 1,
			  start + offset - 1, sequences, WORDLENGTH);
	antiNode->arc = NULL;
	antiNode->arcCount = 0;
	antiNode->marker = NULL;
	antiNode->status = false;
	for (cat = 0; cat < CATEGORIES; cat++)
		antiNode->virtualCoverage[cat] = 0;

	newnd->twinNode = antiNode;
	antiNode->twinNode = newnd;

	if (sequenceID > 0) {
		newnd->length = finish - start;
		antiNode->length = finish - start;
	} else {
		newnd->length = start - finish;
		antiNode->length = start - finish;
	}

	return newnd;
}

void allocateNodeSpace(Graph * graph, IDnum nodeCount)
{
	graph->nodes = malloc(sizeof(Node *) * nodeCount + 1);
	graph->nodeCount = nodeCount;
}

void addNodeToGraph(Graph * graph, Node * node)
{
	graph->nodes[node->ID] = node;
}

boolean getUniqueness(Node * node)
{
	return node->uniqueness;
}

void setUniqueness(Node * node, boolean value)
{
	node->uniqueness = value;
	node->twinNode->uniqueness = value;
}

Node *emptyNode()
{
	Node *newnd = allocateNode();
	Node *antiNode = allocateNode();
	Category cat;

	newnd->ID = 0;
	newnd->descriptor = NULL;
	newnd->arc = NULL;
	newnd->arcCount = 0;
	//newnd->arcTree = NULL;
	newnd->marker = NULL;
	newnd->length = 0;
	for (cat = 0; cat < CATEGORIES; cat++)
		newnd->virtualCoverage[cat] = 0;

	antiNode->ID = 0;
	antiNode->descriptor = NULL;
	antiNode->arc = NULL;
	antiNode->arcCount = 0;
	//antiNode->arcTree = NULL;
	antiNode->marker = NULL;
	antiNode->length = 0;
	for (cat = 0; cat < CATEGORIES; cat++)
		antiNode->virtualCoverage[cat] = 0;

	newnd->twinNode = antiNode;
	antiNode->twinNode = newnd;

	return newnd;

}

Node *addEmptyNodeToGraph(Graph * graph, IDnum ID)
{
	Node *newnd = emptyNode();

	newnd->ID = ID;
	newnd->twinNode->ID = -ID;

	graph->nodes[ID] = newnd;

	return newnd;

}

void setVirtualCoverage(Node * node, Category category,
			Coordinate coverage)
{
	node->virtualCoverage[category] = coverage;
	node->twinNode->virtualCoverage[category] =
	    node->virtualCoverage[category];
}

void incrementVirtualCoverage(Node * node, Category category,
			      Coordinate coverage)
{
	node->virtualCoverage[category] += coverage;
	node->twinNode->virtualCoverage[category] =
	    node->virtualCoverage[category];
}

Coordinate getVirtualCoverage(Node * node, Category category)
{
	return node->virtualCoverage[category];
}

void clipNodeLength(Node * node, Coordinate startClip,
		    Coordinate finishClip)
{
	Descriptor *descriptor;
	Coordinate finalLength =
	    getNodeLength(node) - startClip - finishClip;
	Coordinate index;
	Node *twin = getTwinNode(node);
	Nucleotide nucleotide;

	if (finalLength < 0) {
		puts("Can't clip node that much!!");
		exit(-1);
	}

	if (getNodeLength(node) == 0) {
		puts("Short enough as is");
		exit(-1);
	}
	// One way
	descriptor = node->descriptor;
	for (index = 0; index < finalLength; index++) {
		nucleotide =
		    getNucleotideInDescriptor(descriptor,
					      index + startClip);
		writeNucleotideInDescriptor(nucleotide, descriptor, index);
	}

	// Same thing in the other direction
	descriptor = twin->descriptor;
	for (index = 0; index < finalLength; index++) {
		nucleotide =
		    getNucleotideInDescriptor(descriptor,
					      index + finishClip);
		writeNucleotideInDescriptor(nucleotide, descriptor, index);
	}

	// Length
	node->length = finalLength;
	node->twinNode->length = node->length;
}

boolean hasSingleArc(Node * node)
{
	return node->arcCount == 1;
}

void activateArcLookupTable(Graph * graph)
{
	IDnum index;
	Node *node;
	Arc *arc;
	IDnum nodes = graph->nodeCount;
	IDnum twinOriginID, destinationID, hash;
	Arc **table;

	puts("Activating arc lookup table");

	graph->arcLookupTable = calloc(6 * nodes + 1, sizeof(Arc *));

	table = graph->arcLookupTable;

	for (index = -nodes; index <= nodes; index++) {
		if (index == 0)
			continue;

		node = getNodeInGraph(graph, index);
		if (node == 0)
			continue;

		for (arc = getArc(node); arc != NULL;
		     arc = getNextArc(arc)) {
			twinOriginID = arc->twinArc->destination->ID;
			destinationID = arc->destination->ID;
			hash =
			    3 * nodes - 2 * twinOriginID + destinationID;
			arc->nextInLookupTable = table[hash];
			table[hash] = arc;
		}
	}

	puts("Done activating arc lookup table");
}

void deactivateArcLookupTable(Graph * graph)
{
	free(graph->arcLookupTable);
	graph->arcLookupTable = NULL;
}

static void exportNode(FILE * outfile, Node * node)
{
	Coordinate index;
	Nucleotide nucleotide;

	if (node == NULL)
		return;

	fprintf(outfile, "NODE\t%li\t%li\t%li\n", node->ID,
		node->virtualCoverage[0], node->virtualCoverage[1]);

	for (index = 0; index < node->length; index++) {
		nucleotide =
		    getNucleotideInDescriptor(node->descriptor, index);
		switch (nucleotide) {
		case ADENINE:
			fprintf(outfile, "A");
			break;
		case CYTOSINE:
			fprintf(outfile, "C");
			break;
		case GUANINE:
			fprintf(outfile, "G");
			break;
		case THYMINE:
			fprintf(outfile, "T");
			break;
		}
	}
	fprintf(outfile, "\n");

	for (index = 0; index < node->length; index++) {
		nucleotide =
		    getNucleotideInDescriptor(node->twinNode->descriptor,
					      index);
		switch (nucleotide) {
		case ADENINE:
			fprintf(outfile, "A");
			break;
		case CYTOSINE:
			fprintf(outfile, "C");
			break;
		case GUANINE:
			fprintf(outfile, "G");
			break;
		case THYMINE:
			fprintf(outfile, "T");
			break;
		}
	}
	fprintf(outfile, "\n");
}

static void exportArc(FILE * outfile, Arc * arc)
{
	IDnum originID, destinationID;
	IDnum absOriginID, absDestinationID;

	if (arc == NULL)
		return;

	absOriginID = originID = -arc->twinArc->destination->ID;
	absDestinationID = destinationID = arc->destination->ID;

	if (absOriginID < 0)
		absOriginID = -absOriginID;
	if (absDestinationID < 0)
		absDestinationID = -absDestinationID;

	if (absDestinationID < absOriginID)
		return;

	if (originID == destinationID && originID < 0)
		return;

	fprintf(outfile, "ARC\t%li\t%li\t%li\n", originID, destinationID,
		arc->multiplicity);
}

// Merges two lists of annotations in order of increasing position (used in mergeSort mainly)
static Arc *mergeArcLists(Arc * left,
				   Arc * right)
{
	Arc *mergedList = NULL;
	Arc *tail = NULL;

	// Choose first element:
	if (left->destination->ID <= right->destination->ID) {
		mergedList = left;
		tail = left;
		left = left->next;
	} else {
		mergedList = right;
		tail = right;
		right = right->next;
	}

	// Iterate while both lists are still non empty
	while (left != NULL && right != NULL) {
		if (left->destination->ID <= right->destination->ID) {
			tail->next = left;
			left->previous = tail;
			left = left->next;
		} else {
			tail->next = right;
			right->previous = tail;
			right = right->next;
		}

		tail = tail->next;
	}

	// Concatenate the remaining list at the end of the merged list
	if (left != NULL) {
		tail->next = left;
		left->previous = tail;	
	}

	if (right != NULL) {
		tail->next = right;
		right->previous = tail;
	}

	return mergedList;
}

static void arcMergeSort(Arc** arcPtr, IDnum count) {

	IDnum half = count / 2;
	Arc *left = *arcPtr;
	Arc *ptr = left;
	Arc *right;
	IDnum index;

	if (count == 0 || count == 1)
		return;

	if (count == 2) {
		if ((*arcPtr)->destination->ID> (*arcPtr)->next->destination->ID) {
			(*arcPtr)->next->next = *arcPtr;
			(*arcPtr)->previous = (*arcPtr)->next;
			*arcPtr = (*arcPtr)->next;
			(*arcPtr)->next->next = NULL;
			(*arcPtr)->previous = NULL;
		}
		return;
	}

	for (index = 0; index < half - 1; index++) {
		ptr = ptr->next;
		if (ptr == NULL)
			return;
	}

	right = ptr->next;
	ptr->next = NULL;
	right->previous = NULL;

	arcMergeSort(&left, half);
	arcMergeSort(&right, count - half);
	*arcPtr = mergeArcLists(left, right);
}

static void sortNodeArcs(Node* node) {
	Arc* arc;
	IDnum count = 0;

	for (arc = getArc(node); arc != NULL; arc = getNextArc(arc)) 
		count++;

	if (count == 0)
		return;

	arc = getArc(node);
	arcMergeSort(&arc, count);

	node->arc = arc;
}

void exportGraph(char *filename, Graph * graph, TightString ** sequences)
{
	IDnum index;
	FILE *outfile;
	Node *node;
	Arc *arc;
	PassageMarker *marker;
	IDnum *reads;
	IDnum readCount, readIndex;

	if (graph == NULL) {
		fclose(outfile);
		return;
	}

	outfile = fopen(filename, "w");
	if (outfile == NULL) {
		puts("Couldn't open file, sorry");
		return;
	} else
		puts("Writing into file...");

	// General data
	fprintf(outfile, "%li\t%li\t%i\n", graph->nodeCount,
		graph->sequenceCount, graph->wordLength);

	// Node info
	for (index = 1; index <= graph->nodeCount; index++) {
		node = getNodeInGraph(graph, index);
		exportNode(outfile, node);
	}

	// Arc info
	for (index = 1; index <= graph->nodeCount; index++) {
		node = getNodeInGraph(graph, index);
		if (node == NULL)
			continue;

		sortNodeArcs(node);
		sortNodeArcs(getTwinNode(node));

		for (arc = node->arc; arc != NULL; arc = arc->next)
			exportArc(outfile, arc);
		for (arc = node->twinNode->arc; arc != NULL;
		     arc = arc->next)
			exportArc(outfile, arc);
	}

	// Sequence info
	for (index = 1; index <= graph->nodeCount; index++) {
		node = getNodeInGraph(graph, index);
		if (node == NULL)
			continue;
		for (marker = node->marker; marker != NULL;
		     marker = getNextInNode(marker))
			exportMarker(outfile, marker, sequences,
				     graph->wordLength);
	}

	// Node reads
	if (readStartsAreActivated(graph)) {
		for (index = 0; index <= graph->nodeCount * 2; index++) {
			readCount = graph->nodeReadCounts[index];
			if (readCount == 0)
				continue;

			fprintf(outfile, "NR\t%li\t%li\n",
				index - graph->nodeCount, readCount);
			reads = graph->nodeReads[index];
			for (readIndex = 0; readIndex < readCount;
			     readIndex++)
				fprintf(outfile, "%li\n",
					reads[readIndex]);
		}
	}

	fclose(outfile);
}

Graph *importGraph(char *filename)
{
	FILE *file = fopen(filename, "r");
	const int maxline = 1000000;
	char line[maxline];
	Graph *graph;
	Coordinate coverage1, coverage2;
	IDnum nodeCounter, sequenceCount;
	Node *node, *twin;
	Arc *arc;
	IDnum originID, destinationID, multiplicity;
	PassageMarker *newMarker, *marker;
	IDnum nodeID, seqID;
	Coordinate index;
	Coordinate start, finish;
	Coordinate startOffset, finishOffset;
	boolean finished = false;
	size_t arrayLength;
	IDnum readCount, *array;
	int wordLength;

	//printf("Reading file %s\n", filename);

	// First  line
	fgets(line, maxline, file);
	sscanf(line, "%li\t%li\t%i\n", &nodeCounter, &sequenceCount,
	       &wordLength);
	graph = emptyGraph(sequenceCount, wordLength);
	allocateNodeSpace(graph, nodeCounter);
	printf("Graph has %li nodes and %li sequences\n", nodeCounter,
	       sequenceCount);

	// Read nodes
	fgets(line, maxline, file);
	while (line[0] == 'N') {
		sscanf(line, "NODE\t%li\t%li\t%li\n", &nodeID, &coverage1,
		       &coverage2);
		node = addEmptyNodeToGraph(graph, nodeID);
		setVirtualCoverage(node, 0, coverage1);
		setVirtualCoverage(node, 1, coverage2);

		fgets(line, maxline, file);
		node->length = strlen(line) - 1;
		arrayLength = node->length / 4;
		if (node->length % 4 > 0)
			arrayLength++;
		node->descriptor =
		    malloc(arrayLength * sizeof(Descriptor));
		for (index = 0; index < node->length; index++) {
			if (line[index] == 'A')
				writeNucleotideInDescriptor(ADENINE,
							    node->
							    descriptor,
							    index);
			else if (line[index] == 'C')
				writeNucleotideInDescriptor(CYTOSINE,
							    node->
							    descriptor,
							    index);
			else if (line[index] == 'G')
				writeNucleotideInDescriptor(GUANINE,
							    node->
							    descriptor,
							    index);
			else if (line[index] == 'T')
				writeNucleotideInDescriptor(THYMINE,
							    node->
							    descriptor,
							    index);
		}

		twin = node->twinNode;
		fgets(line, maxline, file);
		twin->length = strlen(line) - 1;
		twin->descriptor =
		    malloc(arrayLength * sizeof(Descriptor));
		for (index = 0; index < twin->length; index++) {
			if (line[index] == 'A')
				writeNucleotideInDescriptor(ADENINE,
							    twin->
							    descriptor,
							    index);
			else if (line[index] == 'C')
				writeNucleotideInDescriptor(CYTOSINE,
							    twin->
							    descriptor,
							    index);
			else if (line[index] == 'G')
				writeNucleotideInDescriptor(GUANINE,
							    twin->
							    descriptor,
							    index);
			else if (line[index] == 'T')
				writeNucleotideInDescriptor(THYMINE,
							    twin->
							    descriptor,
							    index);
		}

		if (fgets(line, maxline, file) == NULL)
			finished = true;
	}

	// Read arcs
	while (line[0] == 'A' && !finished) {
		sscanf(line, "ARC\t%li\t%li\t%li\n", &originID,
		       &destinationID, &multiplicity);
		arc =
		    createArc(getNodeInGraph(graph, originID),
			      getNodeInGraph(graph, destinationID), graph);
		setMultiplicity(arc, multiplicity);
		if (fgets(line, maxline, file) == NULL)
			finished = true;
	}

	// Read sequences
	while (!finished && line[0] != 'N') {
		sscanf(line, "SEQ\t%li\n", &seqID);
		marker = NULL;
		fgets(line, maxline, file);

		while (!finished && line[0] != 'N' && line[0] != 'S') {
			sscanf(line, "%li\t%li\t%li\t%li\t%li\n", &nodeID,
			       &startOffset, &start, &finish,
			       &finishOffset);
			newMarker =
			    newPassageMarker(seqID, start, finish,
					     startOffset, finishOffset);
			transposePassageMarker(newMarker,
					       getNodeInGraph(graph,
							      nodeID));
			connectPassageMarkers(marker, newMarker, graph);
			marker = newMarker;
			if (fgets(line, maxline, file) == NULL)
				finished = true;
		}
	}

	// Node reads
	while (!finished) {
		sscanf(line, "NR\t%li\t%li\n", &nodeID, &readCount);
		if (!readStartsAreActivated(graph))
			activateReadStarts(graph);

		graph->nodeReadCounts[nodeID + graph->nodeCount] =
		    readCount;
		array = malloc(readCount * sizeof(IDnum));
		graph->nodeReads[nodeID + graph->nodeCount] = array;

		readCount = 0;
		fgets(line, maxline, file);
		while (!finished && line[0] != 'N') {
			sscanf(line, "%li\n", &seqID);
			array[readCount] = seqID;
			readCount++;
			if (fgets(line, maxline, file) == NULL)
				finished = true;
		}
	}

	//printf("New graph has %li nodes\n", graph->nodeCount);

	fclose(file);
	//puts("Done, exiting");
	return graph;
}

// Prints out the information relative to the topology of a node into a new file
// Internal to exportDOTGraph()
void DOTNode(Node * node, FILE * outfile)
{
	IDnum ID;
	Arc *arc;
	Node *otherNode;

	ID = node->ID;
	if (ID < 0)
		return;

	fprintf(outfile, "\t%li [label=\"<left>|%li|<right>\"]\n", ID, ID);

	for (arc = node->arc; arc != NULL; arc = arc->next) {
		otherNode = arc->destination;
		if (!(otherNode->ID >= ID || otherNode->ID <= -ID)) {
			continue;
		}

		if (otherNode->ID > 0)
			fprintf(outfile, "\t%li:right -> %li:left\n", ID,
				otherNode->ID);
		else
			fprintf(outfile, "\t%li:right -> %li:right\n", ID,
				-otherNode->ID);
	}

	for (arc = node->twinNode->arc; arc != NULL; arc = arc->next) {
		otherNode = arc->destination;
		if (!(otherNode->ID >= ID || otherNode->ID <= -ID)) {
			continue;
		}

		if (otherNode->ID > 0)
			fprintf(outfile, "\t%li:left -> %li:left\n", ID,
				otherNode->ID);
		else
			fprintf(outfile, "\t%li:left -> %li:right\n", ID,
				-otherNode->ID);
	}
}

// Exports the topology of a graph into a new file (designated by its filename)
void exportDOTGraph(char *filename, Graph * graph)
{
	IDnum nodeIndex;
	Node *currentNode;

	FILE *outfile = fopen(filename, "w");
	if (outfile == NULL) {
		puts("Couldn't open file, sorry");
		return;
	} else
		puts("Writing into file...");

	fprintf(outfile, "digraph G {\n");
	fprintf(outfile, "\tRANKDIR=LR\n");
	fprintf(outfile, "\tnode [shape=record]\n");

	for (nodeIndex = 1; nodeIndex <= graph->nodeCount; nodeIndex++) {
		currentNode = getNodeInGraph(graph, nodeIndex);
		DOTNode(currentNode, outfile);
	}

	fprintf(outfile, "}\n");
	fclose(outfile);
}

TightString *expandNode(Node * node, int WORDLENGTH)
{
	Nucleotide nucleotide;
	Coordinate index;
	TightString *tString = newTightString(node->length + WORDLENGTH - 1);
	Node *twin = node->twinNode;
	Coordinate length = node->length;

	for (index = 0; index < WORDLENGTH; index++) {
		nucleotide =
		    getNucleotideInDescriptor(twin->descriptor,
					      length - index - 1);
		writeNucleotideAtPosition(3 - nucleotide, index, tString);
	}

	for (index = 1; index < node->length; index++) {
		nucleotide =
		    getNucleotideInDescriptor(node->descriptor, index);
		writeNucleotideAtPosition(nucleotide, index + WORDLENGTH - 1,
					  tString);
	}

	return tString;
}

boolean readStartsAreActivated(Graph * graph)
{
	return graph->nodeReads != NULL;
}

void activateReadStarts(Graph * graph)
{
	graph->nodeReads =
	    calloc((2 * graph->nodeCount + 1), sizeof(IDnum *));
	graph->nodeReadCounts =
	    calloc((2 * graph->nodeCount + 1), sizeof(IDnum));
}

void deactivateReadStarts(Graph * graph)
{
	free(graph->nodeReads);
	free(graph->nodeReadCounts);

	graph->nodeReads = NULL;
	graph->nodeReadCounts = NULL;
}

boolean findIDnumInArray(IDnum query, IDnum * array, IDnum arrayLength)
{
	IDnum leftIndex = 0;
	IDnum rightIndex = arrayLength;
	IDnum middleIndex;

	if (arrayLength == 0)
		return false;

	while (true) {
		middleIndex = leftIndex + (rightIndex - leftIndex) / 2;

		if (array[middleIndex] == query)
			return true;
		else if (leftIndex >= rightIndex)
			return false;
		else if (array[middleIndex] > query)
			rightIndex = middleIndex;
		else if (leftIndex == middleIndex)
			leftIndex++;
		else
			leftIndex = middleIndex;
	}
}

static inline int compareIDnums(const void *A, const void *B)
{
	IDnum a = *((IDnum *) A);
	IDnum b = *((IDnum *) B);

	if (a > b)
		return 1;
	if (a == b)
		return 0;
	return -1;
}

void incrementReadStartCount(Node * node, Graph * graph)
{
	graph->nodeReadCounts[node->ID + graph->nodeCount]++;
}

void createNodeReadStartArrays(Graph * graph)
{
	IDnum index;

	if (graph->nodeReads == NULL)
		return;

	for (index = 0; index <= 2 * (graph->nodeCount); index++) {
		if (graph->nodeReadCounts[index] != 0) {
			graph->nodeReads[index] =
			    malloc(graph->nodeReadCounts[index] *
				   sizeof(IDnum));
			graph->nodeReadCounts[index] = 0;
		} else {
			graph->nodeReads[index] = NULL;
		}
	}
}

void addReadStart(Node * node, IDnum seqID, Graph * graph)
{
	IDnum nodeIndex = getNodeID(node) + graph->nodeCount;

	IDnum *array = graph->nodeReads[nodeIndex];
	IDnum arrayLength = graph->nodeReadCounts[nodeIndex];

	if (node->status)
		return;
	node->status = true;

	array[arrayLength] = seqID;
	graph->nodeReadCounts[nodeIndex]++;
	arrayLength++;

	qsort(array, arrayLength, sizeof(IDnum), compareIDnums);
}

IDnum *commonNodeReads(Node * nodeA, Node * nodeB, Graph * graph,
		       IDnum * length)
{
	IDnum targetID, *targetArray, targetLength, targetIndex, targetVal;
	IDnum sourceID, *sourceArray, sourceLength, sourceIndex, sourceVal;
	IDnum *mergeArray, mergeLength;

	if (graph->nodeReads == NULL)
		return NULL;

	if (nodeA == NULL || nodeB == NULL)
		return NULL;

	targetID = getNodeID(nodeA) + graph->nodeCount;
	targetArray = graph->nodeReads[targetID];
	targetLength = graph->nodeReadCounts[targetID];

	sourceID = getNodeID(nodeB) + graph->nodeCount;
	sourceArray = graph->nodeReads[sourceID];
	sourceLength = graph->nodeReadCounts[sourceID];

	if (sourceArray == NULL || targetArray == NULL)
		return NULL;

	mergeArray = malloc((sourceLength + targetLength) * sizeof(IDnum));

	mergeLength = 0;
	sourceIndex = 0;
	targetIndex = 0;
	sourceVal = sourceArray[0];
	targetVal = targetArray[0];

	while (sourceIndex < sourceLength && targetIndex < targetLength) {
		switch (compareIDnums(&sourceVal, &targetVal)) {
		case -1:
			sourceIndex++;
			sourceVal = sourceArray[sourceIndex];
			break;
		case 0:
			mergeArray[mergeLength] = sourceVal;
			mergeLength++;
			sourceIndex++;
			sourceVal = sourceArray[sourceIndex];
			targetIndex++;
			targetVal = targetArray[targetIndex];
			break;
		case 1:
			targetIndex++;
			targetVal = targetArray[targetIndex];
		}

	}

	while (sourceIndex < sourceLength) {
		mergeArray[mergeLength] = sourceArray[sourceIndex];
		mergeLength++;
		sourceIndex++;
	}

	while (targetIndex < targetLength) {
		mergeArray[mergeLength] = targetArray[targetIndex];
		mergeLength++;
		targetIndex++;
	}

	*length = mergeLength;
	return mergeArray;
}

void spreadReadIDs(IDnum * reads, IDnum readCount, Node * node,
		   Graph * graph)
{
	IDnum targetID, *targetArray, targetLength, targetIndex, targetVal;
	IDnum *sourceArray, sourceLength, sourceIndex, sourceVal;
	IDnum *mergeArray, mergeLength;
	IDnum *sourcePtr, *targetPtr, *mergePtr;

	if (graph->nodeReads == NULL || reads == NULL || node == NULL)
		return;

	targetID = getNodeID(node) + graph->nodeCount;
	targetArray = graph->nodeReads[targetID];
	targetLength = graph->nodeReadCounts[targetID];
	targetPtr = targetArray;

	sourceArray = reads;
	sourceLength = readCount;
	sourcePtr = sourceArray;

	if (targetArray == NULL) {
		mergeArray = malloc(sourceLength * sizeof(IDnum));
		mergePtr = mergeArray;

		mergeLength = 0;
		sourceIndex = 0;
		while (sourceIndex < sourceLength) {
			*mergePtr = *sourcePtr;
			mergePtr++;
			sourcePtr++;
			sourceIndex++;
		}

		graph->nodeReads[targetID] = mergeArray;
		graph->nodeReadCounts[targetID] = mergeLength;
		return;
	}

	mergeArray = malloc((sourceLength + targetLength) * sizeof(IDnum));
	mergePtr = mergeArray;

	mergeLength = 0;
	sourceIndex = 0;
	targetIndex = 0;
	sourceVal = sourceArray[0];
	targetVal = targetArray[0];

	while (sourceIndex < sourceLength && targetIndex < targetLength) {
		if (sourceVal < targetVal) {
			*mergePtr = sourceVal;
			sourceIndex++;
			sourcePtr++;
			sourceVal = *sourcePtr;
		} else if (sourceVal == targetVal) {
			*mergePtr = sourceVal;
			sourceIndex++;
			sourcePtr++;
			sourceVal = *sourcePtr;
			targetIndex++;
			targetPtr++;
			targetVal = *targetPtr;
		} else {
			*mergePtr = targetVal;
			targetIndex++;
			targetPtr++;
			targetVal = *targetPtr;
		}

		mergePtr++;
		mergeLength++;
	}

	while (sourceIndex < sourceLength) {
		*mergePtr = *sourcePtr;
		mergeLength++;
		mergePtr++;
		sourceIndex++;
		sourcePtr++;
	}

	while (targetIndex < targetLength) {
		*mergePtr = *targetPtr;
		mergeLength++;
		mergePtr++;
		targetIndex++;
		targetPtr++;
	}

	free(targetArray);
	graph->nodeReads[targetID] = mergeArray;
	graph->nodeReadCounts[targetID] = mergeLength;

}

void transferReadStarts(Node * target, Node * source, Graph * graph)
{
	IDnum sourceID, sourceLength, *sourceArray;

	if (graph->nodeReads == NULL)
		return;

	if (target == NULL || source == NULL)
		return;

	sourceID = source->ID + graph->nodeCount;
	sourceArray = graph->nodeReads[sourceID];
	sourceLength = graph->nodeReadCounts[sourceID];

	if (sourceArray == NULL)
		return;

	spreadReadIDs(sourceArray, sourceLength, target, graph);

	free(sourceArray);
	graph->nodeReads[sourceID] = NULL;
	graph->nodeReadCounts[sourceID] = 0;
}

void shareReadStarts(Node * target, Node * source, Graph * graph)
{
	IDnum *sourceArray, sourceLength, sourceID;

	if (graph->nodeReads == NULL)
		return;

	if (target == NULL || source == NULL)
		return;

	sourceID = source->ID + graph->nodeCount;
	sourceArray = graph->nodeReads[sourceID];
	sourceLength = graph->nodeReadCounts[sourceID];

	if (sourceArray == NULL)
		return;

	spreadReadIDs(sourceArray, sourceLength, target, graph);
}

void mergeNodeReads(Node * nodeA, Node * nodeB, Graph * graph)
{
	IDnum targetID, *targetArray, targetLength, targetIndex, targetVal;
	IDnum sourceID, *sourceArray, sourceLength, sourceIndex, sourceVal;
	IDnum *mergeArrayA, *mergeArrayB, mergeLength;

	if (graph->nodeReads == NULL)
		return;

	if (nodeA == NULL || nodeB == NULL)
		return;

	targetID = getNodeID(nodeA) + graph->nodeCount;
	targetArray = graph->nodeReads[targetID];
	targetLength = graph->nodeReadCounts[targetID];

	sourceID = getNodeID(nodeB) + graph->nodeCount;
	sourceArray = graph->nodeReads[sourceID];
	sourceLength = graph->nodeReadCounts[sourceID];

	mergeArrayA =
	    malloc((sourceLength + targetLength) * sizeof(IDnum));
	mergeArrayB =
	    malloc((sourceLength + targetLength) * sizeof(IDnum));

	mergeLength = 0;
	sourceIndex = 0;
	targetIndex = 0;

	if (sourceArray != NULL)
		sourceVal = sourceArray[0];
	if (targetArray != NULL)
		targetVal = targetArray[0];

	while (sourceIndex < sourceLength && targetIndex < targetLength) {
		switch (compareIDnums(&sourceVal, &targetVal)) {
		case -1:
			mergeArrayA[mergeLength] = sourceVal;
			mergeArrayB[mergeLength] = sourceVal;
			sourceIndex++;
			sourceVal = sourceArray[sourceIndex];
			break;
		case 0:
			mergeArrayA[mergeLength] = sourceVal;
			mergeArrayB[mergeLength] = sourceVal;
			sourceIndex++;
			sourceVal = sourceArray[sourceIndex];
			targetIndex++;
			targetVal = targetArray[targetIndex];
			break;
		case 1:
			mergeArrayA[mergeLength] = targetVal;
			mergeArrayB[mergeLength] = targetVal;
			targetIndex++;
			targetVal = targetArray[targetIndex];
		}

		mergeLength++;
	}

	while (sourceIndex < sourceLength) {
		mergeArrayA[mergeLength] = sourceArray[sourceIndex];
		mergeArrayB[mergeLength] = sourceArray[sourceIndex];
		mergeLength++;
		sourceIndex++;
	}

	while (targetIndex < targetLength) {
		mergeArrayA[mergeLength] = targetArray[targetIndex];
		mergeArrayB[mergeLength] = targetArray[targetIndex];
		mergeLength++;
		targetIndex++;
	}

	if (targetArray != NULL)
		free(targetArray);
	graph->nodeReads[targetID] = mergeArrayA;
	graph->nodeReadCounts[targetID] = mergeLength;

	if (sourceArray != NULL)
		free(sourceArray);
	graph->nodeReads[sourceID] = mergeArrayB;
	graph->nodeReadCounts[sourceID] = mergeLength;
}

IDnum **getNodeToReadMappings(Graph * graph)
{
	return graph->nodeReads;
}

IDnum *getNodeReadCounts(Graph * graph)
{
	return graph->nodeReadCounts;
}

int getWordLength(Graph * graph)
{
	return graph->wordLength;
}
