/*
Copyright 2007 Daniel Zerbino (zerbino@ebi.ac.uk)

    This file is part of Velvet.

    Velvet is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Velvet is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Velvet; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include <stdlib.h>
#include <stdio.h>
#include <time.h>

#include "globals.h"
#include "graph.h"
#include "tightString.h"
#include "dfibHeap.h"
#include "fibHeap.h"
#include "recycleBin.h"
#include "passageMarker.h"
#include "concatenatedGraph.h"
#include "graphStats.h"

#define MAXREADLENGTH 100
#define MAXNODELENGTH 200
#define TICKET_BLOCK_SIZE 10000
#define ARCSPLAY_BLOCK_SIZE 1000

static const Time INDEL = 0;
static const Time SIM[4][4] = {
	{1, 0, 0, 0},
	{0, 1, 0, 0},
	{0, 0, 1, 0},
	{0, 0, 0, 1}
};

typedef struct tkt_st Ticket;

struct tkt_st {
	IDnum id_a;
	IDnum id_b;
	Ticket *next;
};

//Global variables used throughout this procedure(internal use only !)
static Time *times;
static Node **previous;

static DFibHeapNode **dheapNodes;
static DFibHeap *dheap;

static TightString *fastSequence;
static TightString *slowSequence;

static Node *activeNode;
static Node *startingNode;
static int WORDLENGTH;
static double cutoff;
static Graph *graph;

static PassageMarker *fastPath;
static PassageMarker *slowPath;

static IDnum *eligibleStartingPoints;

static double Fmatrix[MAXREADLENGTH + 1][MAXREADLENGTH + 1];
static Coordinate slowToFastMapping[MAXREADLENGTH + 1];

static RecycleBin *ticketMemory;
static Ticket *ticketQueue;

static Ticket **todoLists;
static Ticket **todo;
static Ticket *done;
static boolean *progressStatus;

// DEBUG
static IDnum dbgCounter;
static IDnum modifCounter;
static IDnum arcCounter;
static IDnum btCounter;
static IDnum btLengthCounter;
static IDnum swCounter;
static IDnum swLengthCounter;
static IDnum replaceCounter;
static IDnum vaccinateCounter;
// END OF DEBUG
//End of global variables;

static void setNodeTime(Node * node, Time time)
{
	times[getNodeID(node) + nodeCount(graph)] = time;
}

static Time getNodeTime(Node * node)
{
	return times[getNodeID(node) + nodeCount(graph)];
}

static void setNodePrevious(Node * previousNode, Node * node)
{
	previous[getNodeID(node) + nodeCount(graph)] = previousNode;
}

static Node *getNodePrevious(Node * node)
{
	return previous[getNodeID(node) + nodeCount(graph)];
}

static void setNodeDHeapNode(Node * node, DFibHeapNode * dheapNode)
{
	dheapNodes[getNodeID(node) + nodeCount(graph)] = dheapNode;
}

static DFibHeapNode *getNodeDHeapNode(Node * node)
{
	return dheapNodes[getNodeID(node) + nodeCount(graph)];
}

static Ticket *newTicket()
{
	if (ticketMemory == NULL)
		ticketMemory =
		    newRecycleBin(sizeof(Ticket), TICKET_BLOCK_SIZE);

	return allocatePointer(ticketMemory);
}

static void newQueueTicket(IDnum id_a, IDnum id_b)
{
	Ticket *tkt = newTicket();
	tkt->id_a = id_a;
	tkt->id_b = id_b;
	tkt->next = ticketQueue;
	ticketQueue = tkt;
}

static boolean isPreviousToNode(Node * previous, Node * target)
{
	Node *currentNode = target;
	Node *previousNode = NULL;
	Time targetTime = getNodeTime(target);

	//printf("Testing if %li is previous to %li\n", getNodeID(previous), getNodeID(target));

	while (true) {
		//printf("CCC %li %f\n", getNodeID(currentNode), getNodeTime(currentNode));

		if (currentNode == previous)
			return true;

		if (currentNode == previousNode)
			return false;

		if (getNodeID(currentNode) > nodeCount(graph)
		    || getNodeID(currentNode) < -nodeCount(graph)) {
			printf("Node ID??? %li %li\n",
			       getNodeID(currentNode),
			       getNodeID(previousNode));
		}

		if (getNodeTime(currentNode) != targetTime)
			return false;

		previousNode = currentNode;
		currentNode = getNodePrevious(currentNode);
	}
}

static void concatenateCommonTodoLists(Node * nodeA, Node * nodeB)
{
	Ticket **listA = &todoLists[getNodeID(nodeA) + nodeCount(graph)];
	Ticket **listB = &todoLists[getNodeID(nodeB) + nodeCount(graph)];
	Ticket *head = NULL;
	Ticket *tail = NULL;
	Ticket *tmp;
	IDnum idA, idB;
	IDnum targetID = getNodeID(nodeA);
	IDnum indexA, indexB;
	IDnum nodes = nodeCount(graph);

	//printf("Merging todo list %li into %li\n", getNodeID(nodeB),
	//       getNodeID(nodeA));

	if (*listB == NULL)
		return;

	if (*listA == NULL) {
		*listA = *listB;
		*listB = NULL;
		return;
	}

	while (*listA != NULL && *listB != NULL) {
		idA = (*listA)->id_a;
		idB = (*listB)->id_a;
		indexA = idA + nodes;
		indexB = idB + nodes;

		if (previous[indexA] == nodeA) {
			tmp = *listA;
			*listA = (*listA)->next;
			deallocatePointer(ticketMemory, tmp);
			continue;
		}

		if (idB == targetID || previous[indexB] == nodeA) {
			tmp = *listB;
			*listB = (*listB)->next;
			deallocatePointer(ticketMemory, tmp);
			continue;
		}

		if (idA > idB) {
			tmp = *listB;
			*listB = (*listB)->next;
		} else if (idA < idB) {
			tmp = *listA;
			*listA = (*listA)->next;
		} else {
			tmp = *listB;
			*listB = (*listB)->next;
			deallocatePointer(ticketMemory, tmp);

			tmp = *listA;
			*listA = (*listA)->next;
		}

		if (tail == NULL) {
			tail = tmp;
			head = tail;
		} else {
			tail->next = tmp;
			tail = tail->next;
		}
	}

	while (*listA != NULL) {
		idA = (*listA)->id_a;
		indexA = idA + nodes;

		if (previous[indexA] == nodeA) {
			tmp = *listA;
			*listA = (*listA)->next;
			deallocatePointer(ticketMemory, tmp);
		} else if (tail != NULL) {
			tail->next = *listA;
			*listA = (*listA)->next;
			tail = tail->next;
		} else {
			head = *listA;
			*listA = (*listA)->next;
			tail = head;
		}
	}

	while (*listB != NULL) {
		idB = (*listB)->id_a;
		indexB = idB + nodes;

		if (idB == targetID || previous[indexB] == nodeA) {
			tmp = *listB;
			*listB = (*listB)->next;
			deallocatePointer(ticketMemory, tmp);
		} else if (tail != NULL) {
			tail->next = *listB;
			*listB = (*listB)->next;
			tail = tail->next;
		} else {
			head = *listB;
			*listB = (*listB)->next;
			tail = head;

		}
	}

	if (tail != NULL)
		tail->next = NULL;

	*listA = head;
	*listB = NULL;
}

static void concatenateTodoListIntoActive(Node * nodeB)
{
	Ticket **listB = &todoLists[getNodeID(nodeB) + nodeCount(graph)];
	Ticket *head = NULL;
	Ticket *tail = NULL;
	Ticket *tmp;
	IDnum nodes = nodeCount(graph);
	IDnum idA, idB;
	IDnum activeID = getNodeID(activeNode);
	IDnum indexB, indexA;

	//printf("Merging todo list %li into active node %li\n",
	//       getNodeID(nodeB), getNodeID(activeNode));

	if (*listB == NULL)
		return;

	if (*todo == NULL) {
		*todo = *listB;
		*listB = NULL;
		return;
	}

	while (*todo != NULL && *listB != NULL) {
		idA = (*todo)->id_a;
		idB = (*listB)->id_a;
		indexA = idA + nodes;
		indexB = idB + nodes;

		if (previous[indexA] == activeNode
		    || progressStatus[indexA]) {
			tmp = *todo;
			*todo = (*todo)->next;
			deallocatePointer(ticketMemory, tmp);
			continue;
		}

		if (idB == activeID || previous[indexB] == activeNode
		    || progressStatus[indexB]) {
			tmp = *listB;
			*listB = (*listB)->next;
			deallocatePointer(ticketMemory, tmp);
			continue;
		}

		if (idA > idB) {
			tmp = *listB;
			*listB = (*listB)->next;
		} else if (idA < idB) {
			tmp = *todo;
			*todo = (*todo)->next;
		} else {
			tmp = *listB;
			*listB = (*listB)->next;
			deallocatePointer(ticketMemory, tmp);

			tmp = *todo;
			*todo = (*todo)->next;
		}

		if (tail == NULL) {
			tail = tmp;
			head = tail;
		} else {
			tail->next = tmp;
			tail = tmp;
		}
	}

	while (*todo != NULL) {
		idA = (*todo)->id_a;
		indexA = idA + nodes;

		if (previous[indexA] == activeNode
		    || progressStatus[indexA]) {
			tmp = *todo;
			*todo = (*todo)->next;
			deallocatePointer(ticketMemory, tmp);
		} else if (tail != NULL) {
			tail->next = *todo;
			*todo = (*todo)->next;
			tail = tail->next;
		} else {
			head = *todo;
			*todo = (*todo)->next;
			tail = head;
		}
	}

	while (*listB != NULL) {
		idB = (*listB)->id_a;
		indexB = idB + nodes;

		if (idB == activeID || previous[indexB] == activeNode
		    || progressStatus[indexA]) {
			tmp = *listB;
			*listB = (*listB)->next;
			deallocatePointer(ticketMemory, tmp);
		} else if (tail != NULL) {
			tail->next = *listB;
			*listB = (*listB)->next;
			tail = tail->next;
		} else {
			head = *listB;
			*listB = (*listB)->next;
			tail = head;

		}
	}

	if (tail != NULL)
		tail->next = NULL;
	*todo = head;
	*listB = NULL;
}

static void concatenateTodoLists(Node * nodeA, Node * nodeB)
{
	if (nodeA == activeNode)
		concatenateTodoListIntoActive(nodeB);
	else
		concatenateCommonTodoLists(nodeA, nodeB);
}

static IDnum nextTodoTicket()
{
	Ticket *tkt;
	IDnum index;

	while (*todo != NULL) {
		tkt = *todo;
		*todo = tkt->next;

		// DEBUG
		if (*todo == tkt)
			abort();
		// END OF DEBUG

		index = tkt->id_a + nodeCount(graph);

		if (previous[index] == activeNode) {
			deallocatePointer(ticketMemory, tkt);
			continue;
		}

		progressStatus[index] = true;

		tkt->next = done;
		done = tkt;

		// DEBUG
		if (tkt->id_a == 0)
			abort();
		// END OF DEBUG

		return tkt->id_a;
	}

	return 0;
}

static void freeDoneTickets()
{
	Ticket *tkt;
	IDnum nodes = nodeCount(graph);

	while (done != NULL) {
		tkt = done;
		done = tkt->next;
		progressStatus[tkt->id_a + nodes] = false;
		deallocatePointer(ticketMemory, tkt);
	}
}

static void updateNodeStatus(Node * node)
{
	Arc *arc;

	if (getNodeStatus(node) <= 1)
		return;

	setNodeStatus(node, true);

	for (arc = getArc(node); arc != NULL; arc = getNextArc(arc))
		updateNodeStatus(getDestination(arc));
}

static void determineEligibleStartingPoints()
{
	IDnum nodeIndex;
	IDnum maxmult;
	Node *node;
	Arc *arc;
	IDnum counter;
	FibHeap *heap = newFibHeap();

	puts("Determining eligible starting points");

	for (nodeIndex = 1; nodeIndex <= nodeCount(graph); nodeIndex++) {
		node = getNodeInGraph(graph, nodeIndex);
		if (node == NULL || getNodeStatus(node))
			continue;

		maxmult = 0;
		for (arc = getArc(node); arc != NULL;
		     arc = getNextArc(arc))
			if (getMultiplicity(arc) > maxmult)
				maxmult = getMultiplicity(arc);

		insertNodeIntoHeap(heap, -maxmult, node);

		// Same for twin node
		node = getNodeInGraph(graph, -nodeIndex);
		if (node == NULL || getNodeStatus(node))
			continue;

		maxmult = 0;
		for (arc = getArc(node); arc != NULL;
		     arc = getNextArc(arc))
			if (getMultiplicity(arc) > maxmult)
				maxmult = getMultiplicity(arc);

		insertNodeIntoHeap(heap, -maxmult, node);
	}

	counter = 0;
	node = removeNextNodeFromHeap(heap);
	while (node != NULL) {
		eligibleStartingPoints[counter] = getNodeID(node);
		counter++;
		node = removeNextNodeFromHeap(heap);
	}

	destroyHeap(heap);
	puts("Done listing starting nodes");
}

static Node *nextStartingPoint()
{
	static IDnum counter = 0;
	Node *result = NULL;

	while (result == NULL || getNodeStatus(result) > 0) {
		if (counter >= nodeCount(graph) * 2)
			return NULL;

		result =
		    getNodeInGraph(graph, eligibleStartingPoints[counter]);
		counter++;
	}

	return result;
}

static boolean
extractSequence(PassageMarker * path, TightString * sequence)
{
	PassageMarker *marker;
	Coordinate seqLength = 0;
	Coordinate writeIndex = 0;

	//printf("Extracting sequence %li ... ", pathLength);

	//Measure length
	for (marker = getNextInSequence(path); !isTerminal(marker);
	     marker = getNextInSequence(marker))
		seqLength += getNodeLength(getNode(marker));

	if (seqLength > MAXREADLENGTH)
		return false;
	else
		setTightStringLength(sequence, seqLength);

	//Copy sequences
	for (marker = getNextInSequence(path); !isTerminal(marker);
	     marker = getNextInSequence(marker)) {
		appendNodeSequence(getNode(marker), sequence, writeIndex);
		writeIndex += getNodeLength(getNode(marker));
	}

	return true;
}

static Time max(Time A, Time B, Time C)
{
	if (A >= B && A >= C)
		return A;
	else if (B >= C)
		return B;
	else
		return C;
}

static boolean
compareSequences(TightString * sequence1, TightString * sequence2)
{
	Coordinate i, j;
	Coordinate length1 = getLength(sequence1);
	Coordinate length2 = getLength(sequence2);
	Coordinate avgLength;
	Time Choice1, Choice2, Choice3;
	Time maxScore;

	swCounter++;
	swLengthCounter += length1 * length2;

	if (length1 < WORDLENGTH || length2 < WORDLENGTH)
		return (length1 - length2 < 10 && length1 - length2 > -10);

	avgLength = (length1 + length2) / 2;

	for (i = 0; i <= length1; i++)
		Fmatrix[i][0] = 0;
	for (j = 0; j <= length2; j++)
		Fmatrix[0][j] = 0;

	for (i = 1; i <= length1; i++) {
		for (j = 1; j <= length2; j++) {
			Choice1 =
			    Fmatrix[i - 1][j - 1] +
			    SIM[(int) getNucleotide(i - 1, sequence1)]
			    [(int) getNucleotide(j - 1, sequence2)];
			Choice2 = Fmatrix[i - 1][j] + INDEL;
			Choice3 = Fmatrix[i][j - 1] + INDEL;
			Fmatrix[i][j] = max(Choice1, Choice2, Choice3);
		}
	}

	maxScore = Fmatrix[length1][length2];

	if ((1 - maxScore / avgLength) > cutoff)
		return false;

	return true;
}

static void mapSlowOntoFast()
{
	Coordinate slowIndex = getLength(slowSequence);
	Coordinate fastIndex = getLength(fastSequence);

	if (slowIndex == 0) {
		slowToFastMapping[0] = 0;
		return;
	}

	if (fastIndex == 0) {
		while (slowIndex >= 0)
			slowToFastMapping[slowIndex--] = 0;
		return;
	}

	while (slowIndex != 0 && fastIndex != 0) {
		slowToFastMapping[slowIndex] = fastIndex;

		if (Fmatrix[fastIndex][slowIndex] ==
		    Fmatrix[fastIndex - 1][slowIndex] + INDEL)
			fastIndex--;
		else if (Fmatrix[fastIndex][slowIndex] ==
			 Fmatrix[fastIndex][slowIndex - 1] + INDEL)
			slowIndex--;
		else {
			fastIndex--;
			slowIndex--;
		}
	}

	while (slowIndex >= 0)
		slowToFastMapping[slowIndex--] = 0;

	slowToFastMapping[getLength(slowSequence)] =
	    getLength(fastSequence);
}

static void createAnalogousArcAndVaccinate(Node * nodeA, Node * nodeB,
					   Arc * arc)
{
	boolean aNull = (getNodeLength(nodeA) == 0);
	boolean bNull = (getNodeLength(nodeB) == 0);

	createAnalogousArc(nodeA, nodeB, arc, graph);

	if (aNull && bNull)
		newQueueTicket(getNodeID(nodeA), getNodeID(nodeB));
}

static void remapNodeArcsOntoTarget(Node * source, Node * target)
{
	Arc *arc;

	if (source == activeNode) {
		activeNode = target;
		todo =
		    &todoLists[getNodeID(activeNode) + nodeCount(graph)];
	}
	concatenateTodoLists(target, source);

	arc = getArc(source);
	while (arc != NULL) {
		createAnalogousArcAndVaccinate(target, getDestination(arc),
					       arc);
		destroyArc(arc, graph);
		arc = getArc(source);
	}
}

static void remapNodeArcsOntoNeighbour(Node * source, Node * target)
{
	remapNodeArcsOntoTarget(source, target);
	remapNodeArcsOntoTarget(getTwinNode(source), getTwinNode(target));
}

static void remapNodeMarkersOntoNeighbour(Node * source, Node * target)
{
	PassageMarker *marker;
	Coordinate offset;
	Category cat;

	while (getMarker(source) != NULL) {
		marker = getMarker(source);
		extractPassageMarker(marker);
		transposePassageMarker(marker, target);

		if (getNodeLength(source) != 0) {
			offset = getStartOffset(marker);
			offset *= getNodeLength(target);
			offset /= getNodeLength(source);
			setStartOffset(marker, offset);

			offset = getFinishOffset(marker);
			offset *= getNodeLength(target);
			offset /= getNodeLength(source);
			setFinishOffset(marker, offset);
		} else {
			setStartOffset(marker, 0);
			setFinishOffset(marker, 0);
		}
	}

	// Read starts
	transferReadStarts(target, source, graph);
	transferReadStarts(getTwinNode(target), getTwinNode(source),
			   graph);

	// Virtual reads
	for (cat = 0; cat < CATEGORIES; cat++)
		incrementVirtualCoverage(target, cat,
					 getVirtualCoverage(source, cat));
}

static void remapBackOfNodeArcsOntoNeighbour(Node * source, Node * target)
{
	Arc *arc;

	remapNodeArcsOntoTarget(getTwinNode(source), getTwinNode(target));
	for (arc = getArc(source); arc != NULL; arc = getNextArc(arc))
		createAnalogousArcAndVaccinate(target, source, arc);

}

static void
remapBackOfNodeMarkersOntoNeighbour(Node * source, Node * target,
				    Coordinate refLength)
{
	PassageMarker *marker, *newMarker, *previousMarker;
	Coordinate halfwayPoint, newStartOffset;
	Category cat;
	Coordinate targetLength = getNodeLength(target);
	Coordinate sourceLength = getNodeLength(source);
	Coordinate coverage;

	// Complete markers
	for (marker = getMarker(source); marker != NULL;
	     marker = getNextInNode(marker)) {
		if (getNode(getPreviousInSequence(marker)) == target)
			continue;

		if (sourceLength != 0)
			newStartOffset =
			    getStartOffset(marker) * refLength /
			    sourceLength;
		else
			newStartOffset = 0;

		if (newStartOffset > targetLength) {
			setStartOffset(marker,
				       newStartOffset - targetLength);
			continue;
		}

		newMarker =
		    addPassageMarker(getPassageMarkerSequenceID(marker),
				     getPassageMarkerStart(marker),
				     target);

		setPassageMarkerStart(newMarker,
				      getPassageMarkerStart(marker));
		setPassageMarkerStatus(newMarker,
				       getPassageMarkerStatus(marker));

		if (refLength != 0) {
			halfwayPoint =
			    getPassageMarkerFinish(marker) -
			    getPassageMarkerStart(marker);
			halfwayPoint *= targetLength;
			halfwayPoint /= refLength;
			halfwayPoint += getPassageMarkerStart(marker);
		} else
			halfwayPoint = getPassageMarkerStart(marker);

		setPassageMarkerFinish(newMarker, halfwayPoint);
		setPassageMarkerStart(marker, halfwayPoint);

		previousMarker = getPreviousInSequence(marker);
		setNextInSequence(previousMarker, newMarker);
		setPreviousInSequence(previousMarker, newMarker);

		setStartOffset(newMarker, newStartOffset);
		setFinishOffset(newMarker, 0);
		setStartOffset(marker, 0);

		setPreviousInSequence(newMarker, marker);
		setNextInSequence(newMarker, marker);
	}

	// Read starts
	shareReadStarts(target, source, graph);
	shareReadStarts(getTwinNode(target), getTwinNode(source), graph);

	// Virtual coverage
	if (refLength != 0) {
		for (cat = 0; cat < CATEGORIES; cat++) {
			coverage =
			    getVirtualCoverage(source,
					       cat) * targetLength /
			    refLength;
			incrementVirtualCoverage(target, cat, coverage);
			incrementVirtualCoverage(source, cat, -coverage);
		}
	}

}

static void remapNodeInwardReferencesOntoNode(Node * source, Node * target)
{
	Arc *arc;
	Node *destination;

	for (arc = getArc(source); arc != NULL; arc = getNextArc(arc)) {
		destination = getDestination(arc);
		if (destination == target || destination == source)
			continue;
		if (getNodePrevious(destination) == source)
			setNodePrevious(target, destination);
	}
}

static void remapNodeTimesOntoTargetNode(Node * source, Node * target)
{
	Time nodeTime = getNodeTime(source);
	Node *previous = getNodePrevious(source);
	Time targetTime = getNodeTime(target);

	if (nodeTime == -1)
		return;

	if (previous == source) {
		setNodeTime(target, nodeTime);
		setNodePrevious(target, target);
	} else if (targetTime == -1
		   || targetTime > nodeTime
		   || (targetTime == nodeTime
		       && !isPreviousToNode(target, previous))) {
		setNodeTime(target, nodeTime);
		if (previous != getTwinNode(source))
			setNodePrevious(previous, target);
		else
			setNodePrevious(getTwinNode(target), target);
	}

	remapNodeInwardReferencesOntoNode(source, target);

	setNodePrevious(NULL, source);
}

static void foldSymmetricalNode(Node * node)
{
	Node *twinNode = getTwinNode(node);
	Node *tmp, *destination;
	Arc *arc;
	PassageMarker *oldMarker = getMarker(node);
	PassageMarker *currentMarker, *newMarker, *previousMarker;
	Coordinate halfwayPoint;
	IDnum totalMult;

	// Reduce time complexity of damn thing
	if (simpleArcCount(node) < simpleArcCount(twinNode)) {
		tmp = twinNode;
		twinNode = node;
		node = tmp;
	}
	// Destroy link to old markers 
	setMarker(node, NULL);

	// Reinsert markers properly
	while (oldMarker != NULL) {
		currentMarker = oldMarker;
		oldMarker = getNextInNode(currentMarker);
		previousMarker = getPreviousInSequence(currentMarker);

		if (getNode(previousMarker) != twinNode) {
			newMarker =
			    addUncertainPassageMarker
			    (getPassageMarkerSequenceID(currentMarker),
			     twinNode);
			setPassageMarkerStatus(newMarker,
					       getPassageMarkerStatus
					       (currentMarker));

			setPassageMarkerStart(newMarker,
					      getPassageMarkerStart
					      (currentMarker));

			// For security issues:
			if (currentMarker == slowPath)
				slowPath = newMarker;
			else if (currentMarker == fastPath)
				fastPath = newMarker;

			halfwayPoint =
			    (getPassageMarkerStart(currentMarker) +
			     getPassageMarkerFinish(currentMarker))
			    / 2;
			setPassageMarkerFinish(newMarker, halfwayPoint);
			setPassageMarkerStart(currentMarker, halfwayPoint);

			setStartOffset(newMarker,
				       getStartOffset(currentMarker));
			setFinishOffset(newMarker, 0);
			setStartOffset(currentMarker, 0);

			setPreviousInSequence(previousMarker, newMarker);
			setNextInSequence(previousMarker, newMarker);

			setPreviousInSequence(newMarker, currentMarker);
			setNextInSequence(newMarker, currentMarker);
		}

		transposePassageMarker(currentMarker, node);
	}

	// Read start info
	mergeNodeReads(node, twinNode, graph);

	// Coverage => already balanced!

	// References
	if (getNodeTime(node) == -1 && getNodeTime(twinNode) == -1);
	else if (getNodeTime(node) == -1) {
		setNodeTime(node, getNodeTime(twinNode));
	} else if (getNodeTime(twinNode) == -1) {
		setNodeTime(twinNode, getNodeTime(node));
		setNodePrevious(getNodePrevious(node), twinNode);
	} else if (getNodePrevious(node) == node) {
		setNodeTime(twinNode, getNodeTime(node));
		setNodePrevious(twinNode, twinNode);
	} else if (getNodeTime(node) < getNodeTime(twinNode)) {
		setNodeTime(twinNode, getNodeTime(node));
		setNodePrevious(getNodePrevious(node), twinNode);
	} else if (getNodeTime(node) == getNodeTime(twinNode)
		   && isPreviousToNode(node, twinNode)) {
		setNodePrevious(getNodePrevious(node), twinNode);
	} else {
		setNodeTime(node, getNodeTime(twinNode));
	}

	setNodePrevious(twinNode, node);
	remapNodeInwardReferencesOntoNode(twinNode, node);

	// Active node
	if (twinNode == activeNode) {
		activeNode = node;
		todo =
		    &todoLists[getNodeID(activeNode) + nodeCount(graph)];
	}
	concatenateTodoLists(node, twinNode);

	// Remap arcs properly
	arc = getArc(twinNode);
	totalMult = 0;
	while (arc != NULL) {
		destination = getDestination(arc);
		if (destination != node)
			createAnalogousArc(node, destination, arc, graph);
		totalMult += getMultiplicity(arc);
		destroyArc(arc, graph);
		arc = getArc(twinNode);
	}

	arc = createArc(twinNode, node, graph);
	setMultiplicity(arc, totalMult);

	// Uniqueness
	setUniqueness(node, false);

	// Starting node
	if (startingNode == node)
		startingNode = twinNode;
}

static void remapNodeTimesOntoNeighbour(Node * source, Node * target)
{
	remapNodeTimesOntoTargetNode(source, target);
	remapNodeTimesOntoTargetNode(getTwinNode(source),
				     getTwinNode(target));
}

static void remapNodeTimesOntoForwardMiddlePath(Node * source,
						PassageMarker * path)
{
	PassageMarker *marker;
	Node *target;
	Time nodeTime = getNodeTime(source);
	Node *previousNode = getNodePrevious(source);
	Time targetTime;

	if (nodeTime == -1) {
		puts("unvisited tip of bridge");
		abort();
	}

	for (marker = path; getNode(marker) != source;
	     marker = getNextInSequence(marker)) {
		target = getNode(marker);
		targetTime = getNodeTime(target);

		if (targetTime == -1
		    || targetTime > nodeTime
		    || (targetTime == nodeTime
			&& !isPreviousToNode(target, previousNode))) {
			setNodeTime(target, nodeTime);
			setNodePrevious(previousNode, target);
		}

		previousNode = target;
	}

	setNodePrevious(previousNode, source);

}

static void remapNodeTimesOntoTwinMiddlePath(Node * source,
					     PassageMarker * path)
{
	PassageMarker *marker;
	Node *target;
	Node *previousNode = getTwinNode(source);
	Time targetTime;
	PassageMarker *limit = getTwinMarker(getPreviousInSequence(path));
	Time nodeTime = getNodeTime(getNode(limit));

	if (nodeTime == -1) {
		puts("Unvisited bridgehead");
		abort();
	}
	// Revving up
	marker = path;
	while (getNode(marker) != source)
		marker = getNextInSequence(marker);
	marker = getTwinMarker(marker);

	// Going down the path
	while (marker != limit) {
		marker = getNextInSequence(marker);
		target = getNode(marker);
		targetTime = getNodeTime(target);

		if (targetTime == -1
		    || targetTime > nodeTime
		    || (targetTime == nodeTime
			&& !isPreviousToNode(target, previousNode))) {
			setNodeTime(target, nodeTime);
			getNodeTime(target);
			setNodePrevious(previousNode, target);
		}

		previousNode = target;
	}
}

static void
remapNodeFibHeapReferencesOntoNode(Node * source, Node * target)
{
	DFibHeapNode *sourceDHeapNode = getNodeDHeapNode(source);
	DFibHeapNode *targetDHeapNode = getNodeDHeapNode(target);

	if (sourceDHeapNode == NULL)
		return;

	if (targetDHeapNode == NULL) {
		setNodeDHeapNode(target, sourceDHeapNode);
		replaceValueInDHeap(sourceDHeapNode, target);
	} else if (getKey(targetDHeapNode) > getKey(sourceDHeapNode)) {
		setNodeDHeapNode(target, sourceDHeapNode);
		replaceValueInDHeap(sourceDHeapNode, target);
		destroyNodeInDHeap(targetDHeapNode, dheap);
	} else
		destroyNodeInDHeap(sourceDHeapNode, dheap);

	setNodeDHeapNode(source, NULL);
}

static void remapNodeOntoNeighbour(Node * source, Node * target)
{
	//printf("Remapping node %li onto middle path %li\n", getNodeID(source), getNodeID(target));
	if (getNodeID(source) == 51536 && getNodeID(target) == 38567) {
		printf("Source\t51536\ttarget\t38567\n");
		printf("prev\t%li\tprev\t%li\n",
		       getNodeID(getNodePrevious(source)),
		       getNodeID(getNodePrevious(target)));
		printf("-Source\t-51536\t-target\t-38567\n");
		printf("-prev\t%li\t-prev\t%li\n",
		       getNodeID(getNodePrevious(getTwinNode(source))),
		       getNodeID(getNodePrevious(getTwinNode(target))));
	}

	remapNodeMarkersOntoNeighbour(source, target);

	remapNodeTimesOntoNeighbour(source, target);
	remapNodeArcsOntoNeighbour(source, target);

	remapNodeFibHeapReferencesOntoNode(getTwinNode(source),
					   getTwinNode(target));
	remapNodeFibHeapReferencesOntoNode(source, target);

	if (startingNode == source)
		startingNode = target;
	if (startingNode == getTwinNode(source))
		startingNode = getTwinNode(target);

	destroyNode(source, graph);
}

static void remapBackOfNodeTimesOntoNeighbour(Node * source, Node * target)
{
	Time targetTime = getNodeTime(target);
	Time nodeTime = getNodeTime(source);
	Node *twinTarget = getTwinNode(target);
	Node *twinSource = getTwinNode(source);
	Node *previous;

	if (nodeTime != -1) {
		previous = getNodePrevious(source);

		if (previous == source) {
			setNodeTime(target, nodeTime);
			setNodePrevious(target, target);
		} else if (targetTime == -1
			   || targetTime > nodeTime
			   || (targetTime == nodeTime
			       && !isPreviousToNode(target, previous))) {
			setNodeTime(target, nodeTime);
			setNodePrevious(previous, target);
		}

		setNodePrevious(target, source);
	}

	targetTime = getNodeTime(twinTarget);
	nodeTime = getNodeTime(twinSource);

	if (nodeTime != -1) {
		if (targetTime == -1
		    || targetTime > nodeTime
		    || (targetTime == nodeTime
			&& !isPreviousToNode(twinTarget, twinSource))) {
			setNodeTime(twinTarget, nodeTime);
			setNodePrevious(twinSource, twinTarget);
		}
	}

	remapNodeInwardReferencesOntoNode(twinSource, twinTarget);
}

static void
remapBackOfNodeOntoNeighbour(Node * source, Node * target,
			     Coordinate refLength)
{
	//printf("Remapping node %li onto middle path\n", getNodeID(node));

	remapBackOfNodeMarkersOntoNeighbour(source, target, refLength);

	remapBackOfNodeTimesOntoNeighbour(source, target);

	remapBackOfNodeArcsOntoNeighbour(source, target);

	remapNodeFibHeapReferencesOntoNode(getTwinNode(source),
					   getTwinNode(target));

	if (getTwinNode(source) == startingNode)
		startingNode = getTwinNode(target);
}

static void remapEmptyPathArcsOntoMiddlePathSimple(PassageMarker *
						   emptyPath,
						   PassageMarker *
						   targetPath)
{
	PassageMarker *pathMarker;
	Node *start = getNode(getPreviousInSequence(emptyPath));
	Node *finish = getNode(emptyPath);
	Node *previousNode = start;
	Node *currentNode;
	Arc *originalArc = getArcBetweenNodes(start, finish, graph);

	for (pathMarker = targetPath; getNode(pathMarker) != finish;
	     pathMarker = getNextInSequence(pathMarker)) {
		currentNode = getNode(pathMarker);
		createAnalogousArcAndVaccinate(previousNode, currentNode,
					       originalArc);
		previousNode = currentNode;
	}

	createAnalogousArcAndVaccinate(previousNode, finish, originalArc);

	destroyArc(originalArc, graph);
}

static void remapEmptyPathMarkersOntoMiddlePathSimple(PassageMarker *
						      emptyPath,
						      PassageMarker *
						      targetPath)
{
	PassageMarker *marker, *newMarker, *previousMarker, *pathMarker;
	Node *start = getNode(getPreviousInSequence(emptyPath));
	Node *finish = getNode(emptyPath);
	PassageMarker *oldMarker = getMarker(finish);
	Coordinate markerStart;
	IDnum intersectionLength, twinIntersectionLength;
	IDnum *intersectionReads =
	    commonNodeReads(start, finish, graph, &intersectionLength);
	IDnum *twinIntersectionReads =
	    commonNodeReads(getTwinNode(start), getTwinNode(finish), graph,
			    &twinIntersectionLength);

	//printf("SIMPLE %li\t%li\t%i\t%i\n", markerCount(finish),
	//       getNodeID(finish), arcCount(finish),
	//       arcCount(getTwinNode(finish)));

	// Destroy link to old nodes
	setMarker(finish, NULL);

	while (oldMarker != NULL) {
		marker = oldMarker;
		oldMarker = getNextInNode(marker);
		newMarker = getPreviousInSequence(marker);

		if (getNode(newMarker) != start) {
			transposePassageMarker(marker, finish);
			continue;
		}

		markerStart = getPassageMarkerStart(marker);
		for (pathMarker = targetPath;
		     getNode(pathMarker) != finish;
		     pathMarker = getNextInSequence(pathMarker)) {
			previousMarker = newMarker;

			newMarker =
			    addUncertainPassageMarker
			    (getPassageMarkerSequenceID(marker),
			     getNode(pathMarker));
			setPassageMarkerStatus(newMarker,
					       getPassageMarkerStatus
					       (marker));
			setPassageMarkerStart(newMarker, markerStart);
			setPassageMarkerFinish(newMarker, markerStart);

			setNextInSequence(previousMarker, newMarker);
			setPreviousInSequence(previousMarker, newMarker);

			setStartOffset(newMarker, 0);
			setFinishOffset(newMarker, 0);

			// Read starts
			spreadReadIDs(intersectionReads,
				      intersectionLength,
				      getNode(pathMarker), graph);
			spreadReadIDs(twinIntersectionReads,
				      twinIntersectionLength,
				      getTwinNode(getNode(pathMarker)),
				      graph);
		}

		setNextInSequence(newMarker, marker);
		setPreviousInSequence(newMarker, marker);
		transposePassageMarker(marker, finish);
	}


}

static boolean markerFollowsPath(PassageMarker * marker,
				 PassageMarker * start,
				 PassageMarker * finish, Node * stopNode)
{
	PassageMarker *current, *path;

	path = start;
	current = marker;

	while (true) {
		if (current == NULL || path == finish || path == NULL)
			return true;

		if (getNode(current) != getNode(path))
			return false;

		current = getNextInSequence(current);
		path = getNextInSequence(path);
	}
}

static PassageMarkerList *getAnchors(PassageMarker * marker, Node * nodeA,
				     Node * nodeB)
{
	PassageMarker *current, *next;
	Node *twinA = getTwinNode(nodeA);
	Node *twinB = getTwinNode(nodeB);
	PassageMarkerList *result = NULL;

	current = marker;
	while (current != NULL) {
		next = getNextInSequence(current);
		if (getNode(current) == nodeA && getNode(next) == nodeB) {
			result = newPassageMarkerList(next, result);
		}
		if (getNode(current) == twinB && getNode(next) == twinA) {
			result =
			    newPassageMarkerList(getTwinMarker(current),
						 result);
		}
		current = next;
	}

	return result;
}

static void destroyPassageMarkerList(PassageMarkerList ** list)
{
	PassageMarkerList *ptr;

	while (*list != NULL) {
		ptr = *list;
		*list = ptr->next;
		deallocatePassageMarkerList(ptr);
	}
}

static void remapEmptyPathMarkersOntoMiddlePathDevious(PassageMarker *
						       emptyPath,
						       PassageMarker *
						       targetPath)
{
	PassageMarker *marker, *newMarker, *previousMarker, *pathMarker;
	Node *start = getNode(getPreviousInSequence(emptyPath));
	Node *finish = getNode(emptyPath);
	PassageMarkerList *anchors = getAnchors(targetPath, start, finish);
	PassageMarkerList *currentAnchor;
	boolean untouchable;
	Coordinate markerStart;

	printf("DEVIOUS %li\t%li\t%i\t%i\n", markerCount(finish),
	       getNodeID(finish), arcCount(finish),
	       arcCount(getTwinNode(finish)));

	for (marker = getMarker(finish); marker != NULL;
	     marker = getNextInNode(marker)) {
		newMarker = getPreviousInSequence(marker);

		if (getNode(newMarker) != start)
			continue;


		for (currentAnchor = anchors; currentAnchor != NULL;
		     currentAnchor = currentAnchor->next)
			if (markerFollowsPath
			    (marker, currentAnchor->marker, targetPath,
			     finish)) {
				untouchable = true;
				break;
			}

		if (untouchable)
			continue;

		markerStart = getPassageMarkerStart(marker);
		for (pathMarker = targetPath;
		     getNode(pathMarker) != finish;
		     pathMarker = getNextInSequence(pathMarker)) {
			previousMarker = newMarker;
			newMarker =
			    addUncertainPassageMarker
			    (getPassageMarkerSequenceID(marker),
			     getNode(pathMarker));
			setPassageMarkerStatus(newMarker,
					       getPassageMarkerStatus
					       (marker));
			setPassageMarkerStart(newMarker, markerStart);
			setPassageMarkerFinish(newMarker, markerStart);

			setNextInSequence(previousMarker, newMarker);
			setPreviousInSequence(previousMarker, newMarker);

			setStartOffset(newMarker, 0);
			setFinishOffset(newMarker, 0);
		}

		setNextInSequence(newMarker, marker);
		setPreviousInSequence(newMarker, marker);
	}

	destroyPassageMarkerList(&anchors);
}

static boolean markerLeadsToArc(PassageMarker * marker, Node * nodeA,
				Node * nodeB)
{
	PassageMarker *current, *next;
	Node *twinA = getTwinNode(nodeA);
	Node *twinB = getTwinNode(nodeB);

	current = marker;
	while (current != NULL) {
		next = getNextInSequence(current);
		if (getNode(current) == nodeA && getNode(next) == nodeB)
			return true;
		if (getNode(current) == twinB && getNode(next) == twinA)
			return true;
		current = next;
	}

	return false;
}

static void
remapEmptyPathOntoMiddlePath(PassageMarker * emptyPath,
			     PassageMarker * targetPath)
{
	Node *start = getNode(getPreviousInSequence(emptyPath));
	Node *finish = getNode(emptyPath);

	// Remapping markers
	if (!markerLeadsToArc(targetPath, start, finish)) {
		remapEmptyPathArcsOntoMiddlePathSimple(emptyPath,
						       targetPath);
		remapEmptyPathMarkersOntoMiddlePathSimple(emptyPath,
							  targetPath);
	} else {
		remapEmptyPathMarkersOntoMiddlePathDevious(emptyPath,
							   targetPath);
	}

	//Remap times and previous(if necessary)
	if (getNodePrevious(finish) == start)
		remapNodeTimesOntoForwardMiddlePath(finish, targetPath);

	if (getNodePrevious(getTwinNode(start)) == getTwinNode(finish))
		remapNodeTimesOntoTwinMiddlePath(finish, targetPath);
}

static void reduceSlowNodes(PassageMarker * slowMarker, Node * finish)
{
	PassageMarker *marker;

	for (marker = slowMarker; getNode(marker) != finish;
	     marker = getNextInSequence(marker))
		reduceNode(getNode(marker));
}

static void destroyPaths()
{
	PassageMarker *marker;

	while (slowPath != NULL) {
		marker = slowPath;
		getNodeTime(getNode(marker));
		getNodeTime(getTwinNode(getNode(marker)));

		slowPath = getNextInSequence(marker);
		destroyPassageMarker(marker);
	}

	while (fastPath != NULL) {
		marker = fastPath;
		getNodeTime(getNode(marker));
		getNodeTime(getTwinNode(getNode(marker)));
		fastPath = getNextInSequence(marker);
		destroyPassageMarker(marker);
	}
}

static void mapDistancesOntoPaths()
{
	PassageMarker *marker;
	Coordinate totalDistance = 0;

	for (marker = getNextInSequence(slowPath); !isTerminal(marker);
	     marker = getNextInSequence(marker)) {
		setPassageMarkerStart(marker,
				      slowToFastMapping[totalDistance]);
		totalDistance += getNodeLength(getNode(marker));
		setPassageMarkerFinish(marker,
				       slowToFastMapping[totalDistance]);
	}

	setPassageMarkerStart(marker, slowToFastMapping[totalDistance]);
	setPassageMarkerFinish(marker,
			       slowToFastMapping[totalDistance] +
			       getNodeLength(getNode(marker)));

	totalDistance = 0;

	for (marker = getNextInSequence(fastPath); !isTerminal(marker);
	     marker = getNextInSequence(marker)) {
		setPassageMarkerStart(marker, totalDistance);
		totalDistance += getNodeLength(getNode(marker));
		setPassageMarkerFinish(marker, totalDistance);
	}

	setPassageMarkerStart(marker, totalDistance);
	setPassageMarkerFinish(marker,
			       totalDistance +
			       getNodeLength(getNode(marker)));
}

static boolean markerLeadsToNode(PassageMarker * marker, Node * node)
{
	PassageMarker *currentMarker;

	for (currentMarker = marker; currentMarker != NULL;
	     currentMarker = getNextInSequence(currentMarker))
		if (getNode(currentMarker) == node)
			return true;

	return false;
}

static void transferNullNodeData(Node * source, Node * target)
{
	Arc *arc;
	Node *twinSource = getTwinNode(source);
	Node *twinTarget = getTwinNode(target);
	Node *destination;

	// Time & Outward references
	if (getNodeTime(source) == -1);
	else if (getNodePrevious(source) == source) {
		setNodeTime(target, getNodeTime(source));
		setNodePrevious(target, target);
	} else if (getNodeTime(target) == -1
		   || getNodeTime(source) < getNodeTime(target)
		   || (getNodeTime(source) == getNodeTime(target)
		       && !isPreviousToNode(target, source))) {
		setNodeTime(target, getNodeTime(source));
		setNodePrevious(getNodePrevious(source), target);
	}

	if (getNodePrevious(target) == twinSource)
		setNodePrevious(twinTarget, target);

	if (getNodeTime(twinSource) == -1);
	else if (getNodePrevious(twinSource) == twinSource) {
		setNodeTime(twinTarget, getNodeTime(twinSource));
		setNodePrevious(twinTarget, twinTarget);
	} else if (getNodeTime(twinTarget) == -1
		   || getNodeTime(twinSource) < getNodeTime(twinTarget)
		   || (getNodeTime(twinSource) == getNodeTime(twinTarget)
		       && !isPreviousToNode(twinTarget, twinSource))) {
		setNodeTime(twinTarget, getNodeTime(twinSource));
		setNodePrevious(getNodePrevious(twinSource), twinTarget);
	}

	if (getNodePrevious(twinTarget) == source)
		setNodePrevious(target, twinTarget);

	// Inward references:
	for (arc = getArc(source); arc != NULL; arc = getNextArc(arc)) {
		destination = getDestination(arc);
		if (getNodePrevious(destination) == source)
			setNodePrevious(target, destination);
	}

	for (arc = getArc(twinSource); arc != NULL; arc = getNextArc(arc)) {
		destination = getDestination(arc);
		if (destination == twinTarget)
			continue;
		if (getNodePrevious(destination) == twinSource)
			setNodePrevious(twinTarget, destination);
	}

	// Fib Heap refs
	remapNodeFibHeapReferencesOntoNode(source, target);
	remapNodeFibHeapReferencesOntoNode(twinSource, twinTarget);

	// Starting point
	if (startingNode == source)
		startingNode = target;
	else if (startingNode == twinSource)
		startingNode = twinTarget;

	// Next node 
	if (source == activeNode) {
		activeNode = target;
		todo =
		    &todoLists[getNodeID(activeNode) + nodeCount(graph)];
	}
	concatenateTodoLists(target, source);

	if (twinSource == activeNode) {
		activeNode = twinTarget;
		todo =
		    &todoLists[getNodeID(activeNode) + nodeCount(graph)];
	}
	concatenateTodoLists(twinTarget, twinSource);
}

static void transferNodeData(Node * source, Node * target)
{
	Arc *arc;
	Node *twinSource = getTwinNode(source);
	Node *twinTarget = getTwinNode(target);
	Node *destination;

	// Time & Outward references
	if (getNodePrevious(source) == source) {
		setNodeTime(target, getNodeTime(source));
		setNodePrevious(target, target);
	}

	if (getNodeTime(twinSource) == -1);
	else if (getNodePrevious(twinSource) == twinSource) {
		setNodeTime(twinTarget, getNodeTime(twinSource));
		setNodePrevious(twinTarget, twinTarget);
	} else if (getNodeTime(twinTarget) == -1
		   || getNodeTime(twinSource) < getNodeTime(twinTarget)
		   || (getNodeTime(twinSource) == getNodeTime(twinTarget)
		       && !isPreviousToNode(twinTarget, twinSource))) {
		setNodeTime(twinTarget, getNodeTime(twinSource));
		setNodePrevious(getNodePrevious(twinSource), twinTarget);
	}

	if (getNodePrevious(twinTarget) == source)
		setNodePrevious(target, twinTarget);

	// Inward references:
	for (arc = getArc(source); arc != NULL; arc = getNextArc(arc)) {
		destination = getDestination(arc);
		if (getNodePrevious(destination) == source)
			setNodePrevious(target, destination);
	}

	// Fib Heap refs
	remapNodeFibHeapReferencesOntoNode(source, target);
	remapNodeFibHeapReferencesOntoNode(twinSource, twinTarget);

	// Starting point
	if (startingNode == source)
		startingNode = target;
	else if (startingNode == twinSource)
		startingNode = twinTarget;

	if (getNode(slowPath) == twinSource)
		slowPath = getNextInSequence(slowPath);
	if (getNode(fastPath) == twinSource)
		fastPath = getNextInSequence(fastPath);

	// Next node 
	if (source == activeNode) {
		activeNode = target;
		todo =
		    &todoLists[getNodeID(activeNode) + nodeCount(graph)];
	}
	concatenateTodoLists(target, source);

	if (twinSource == activeNode) {
		activeNode = twinTarget;
		todo =
		    &todoLists[getNodeID(activeNode) + nodeCount(graph)];
	}
}

static void removeNullLoop(Node * node)
{
	PassageMarker *marker, *next, *nextNext;
	Arc *arc = getArcBetweenNodes(node, node, graph);

	if (arc == NULL)
		return;

	//puts("Self looping null node");

	destroyArc(arc, graph);

	for (marker = getMarker(node); marker != NULL;
	     marker = getNextInNode(marker)) {
		next = getNextInSequence(marker);
		nextNext = getNextInSequence(next);
		while (getNode(next) == node) {
			//puts("\tCandidate next");
			if (getTwinMarker(next) == fastPath)
				fastPath = getTwinMarker(marker);
			if (getTwinMarker(next) == slowPath)
				slowPath = getTwinMarker(marker);

			disconnectNextPassageMarker(marker, graph);
			destroyPassageMarker(next);

			next = nextNext;
			nextNext = getNextInSequence(next);
		}
	}
}

static boolean concatenateNullNodes(Node * nodeA, Node * nodeB)
{
	Arc *arc;
	Node *origin;
	PassageMarker *marker, *tmpMarker;
	Node *twinA = getTwinNode(nodeA);
	Node *twinB = getTwinNode(nodeB);
	Category cat;

	if (simpleArcCount(nodeB) + simpleArcCount(twinB) >
	    simpleArcCount(nodeA) + simpleArcCount(twinA)) {
		origin = nodeB;
		nodeB = twinA;
		twinA = origin;
		origin = twinB;
		twinB = nodeA;
		nodeA = origin;
	}

	vaccinateCounter++;

	// Recusion data
	transferNullNodeData(nodeB, nodeA);

	// Marker management 
	for (marker = getMarker(nodeB); marker != NULL; marker = tmpMarker) {
		tmpMarker = getNextInNode(marker);
		if (getNode(getPreviousInSequence(marker)) == nodeA)
			disconnectNextPassageMarker(getPreviousInSequence
						    (marker), graph);
		else {
			extractPassageMarker(marker);
			transposePassageMarker(marker, nodeA);
		}
	}

	// Arc management
	for (arc = getArc(nodeB); arc != NULL; arc = getNextArc(arc)) {
		origin = getTwinNode(getDestination(arc));
		if (origin == nodeB)
			createAnalogousArcAndVaccinate(nodeA, twinA, arc);
		else if (origin == twinB)
			createAnalogousArcAndVaccinate(twinA, twinA, arc);
		else
			createAnalogousArcAndVaccinate(origin, twinA, arc);
	}

	for (arc = getArc(twinB); arc != NULL; arc = getNextArc(arc)) {
		origin = getTwinNode(getDestination(arc));
		if (origin == nodeA)
			continue;
		else if (origin == nodeB)
			createAnalogousArcAndVaccinate(nodeA, nodeA, arc);
		else if (origin == twinB)
			createAnalogousArcAndVaccinate(twinA, nodeA, arc);
		else
			createAnalogousArcAndVaccinate(origin, nodeA, arc);
	}

	// Update virtual coverage
	for (cat = 0; cat < CATEGORIES; cat++)
		incrementVirtualCoverage(nodeA, cat,
					 getVirtualCoverage(nodeB, cat));

	// Read starts
	transferReadStarts(nodeA, nodeB, graph);
	transferReadStarts(getTwinNode(nodeA), getTwinNode(nodeB), graph);

	// Clean up
	destroyNode(nodeB, graph);

	return true;

}

static void resolveNullHairpin(Node * node)
{
	Node *twin = getTwinNode(node);
	PassageMarker *marker, *next;
	Arc *twin2node = getArcBetweenNodes(twin, node, graph);
	Arc *node2twin = getArcBetweenNodes(node, twin, graph);

	// Eliminating good cases
	if (node2twin == NULL && twin2node == NULL)
		return;
	if (node2twin == NULL && hasSingleArc(twin))
		return;
	if (hasSingleArc(node) && twin2node == NULL)
		return;

	//puts("Simplifying null hairpin");

	for (marker = getMarker(node); marker != NULL;
	     marker = getNextInNode(marker)) {
		next = getNextInSequence(marker);
		while (next != NULL
		       && (getNode(next) == node
			   || getNode(next) == twin)) {
			if (getTwinMarker(next) == fastPath)
				fastPath = getTwinMarker(marker);
			if (getTwinMarker(next) == slowPath)
				slowPath = getTwinMarker(marker);

			disconnectNextPassageMarker(marker, graph);
			destroyPassageMarker(next);
			next = getNextInSequence(marker);
		}
	}

	for (marker = getMarker(twin); marker != NULL;
	     marker = getNextInNode(marker)) {
		next = getNextInSequence(marker);
		if (next != NULL && getNode(next) == node) {
			if (getTwinMarker(next) == fastPath)
				fastPath = getTwinMarker(marker);
			if (getTwinMarker(next) == slowPath)
				slowPath = getTwinMarker(marker);

			disconnectNextPassageMarker(marker, graph);
			destroyPassageMarker(next);
		}
	}

	destroyArc(node2twin, graph);
	destroyArc(twin2node, graph);
	foldSymmetricalNode(node);
}

static void vaccinatePath()
{
	Ticket *tkt;
	Node *nodeA, *nodeB;

	//puts("Vaccinating path");
	while (ticketQueue != NULL) {
		tkt = ticketQueue;
		ticketQueue = tkt->next;

		nodeA = getNodeInGraph(graph, tkt->id_a);

		if (nodeA == NULL || getNodeLength(nodeA) != 0) {
			deallocatePointer(ticketMemory, tkt);
			continue;
		}

		if (tkt->id_a == tkt->id_b) {
			removeNullLoop(nodeA);
			deallocatePointer(ticketMemory, tkt);
			continue;
		} else if (tkt->id_a == -tkt->id_b) {
			resolveNullHairpin(nodeA);
			deallocatePointer(ticketMemory, tkt);
			continue;
		}

		nodeB = getNodeInGraph(graph, tkt->id_b);

		if (nodeB != NULL && getNodeLength(nodeB) == 0
		    && getArcBetweenNodes(nodeA, nodeB, graph) != NULL)
			concatenateNullNodes(nodeA, nodeB);

		deallocatePointer(ticketMemory, tkt);
	}
}

// Replaces two consecutive nodes into a single equivalent node
// The extra memory is freed
static void concatenateNodesAndVaccinate(Node * nodeA, Node * nodeB,
					 Graph * graph)
{
	PassageMarker *marker, *tmpMarker;
	Node *twinA = getTwinNode(nodeA);
	Node *twinB = getTwinNode(nodeB);
	Arc *arc;
	Category cat;

	// Arc management:
	// Freeing useless arcs
	while (getArc(nodeA) != NULL)
		destroyArc(getArc(nodeA), graph);

	// Correct arcs
	for (arc = getArc(nodeB); arc != NULL; arc = getNextArc(arc)) {
		if (getDestination(arc) != twinB)
			createAnalogousArcAndVaccinate(nodeA,
						       getDestination(arc),
						       arc);
		else
			createAnalogousArcAndVaccinate(nodeA, twinA, arc);
	}

	// Passage marker management in node A:
	for (marker = getMarker(nodeA); marker != NULL;
	     marker = getNextInNode(marker))
		if (isTerminal(marker))
			incrementFinishOffset(marker,
					      getNodeLength(nodeB));

	// Swapping new born passageMarkers from B to A
	for (marker = getMarker(nodeB); marker != NULL; marker = tmpMarker) {
		tmpMarker = getNextInNode(marker);

		if (isInitial(marker)) {
			extractPassageMarker(marker);
			transposePassageMarker(marker, nodeA);
			incrementFinishOffset(getTwinMarker(marker),
					      getNodeLength(nodeA));
		} else
			disconnectNextPassageMarker(getPreviousInSequence
						    (marker), graph);
	}

	// Descriptor management 
	appendDescriptors(nodeA, nodeB);

	// Update uniqueness:
	setUniqueness(nodeA, getUniqueness(nodeA) || getUniqueness(nodeB));

	// Update virtual coverage
	for (cat = 0; cat < CATEGORIES; cat++)
		incrementVirtualCoverage(nodeA, cat,
					 getVirtualCoverage(nodeB, cat));

	// Read starts
	transferReadStarts(nodeA, nodeB, graph);
	transferReadStarts(getTwinNode(nodeA), getTwinNode(nodeB), graph);

	// Freeing gobbled node
	destroyNode(nodeB, graph);
}

static void simplifyNode(Node * node)
{
	Node *twin = getTwinNode(node);
	Node *destination, *twinDestination;

	if (!hasSingleArc(node))
		return;

	destination = getDestination(getArc(node));
	twinDestination = getTwinNode(destination);

	while (hasSingleArc(node)
	       && hasSingleArc(twinDestination)
	       && destination != twin && destination != node) {
		transferNodeData(destination, node);
		//printf("Concatenating nodes %li and %li\n", getNodeID(node), getNodeID(destination));
		concatenateNodesAndVaccinate(node, destination, graph);

		if (!hasSingleArc(node))
			return;
		destination = getDestination(getArc(node));
		twinDestination = getTwinNode(destination);
	}

}

static void concatenatePathNodes(PassageMarker * pathStart)
{
	PassageMarker *pathMarker;

	//puts("Removing null loops");
	for (pathMarker = pathStart; pathMarker != NULL;
	     pathMarker = getNextInSequence(pathMarker)) {
		simplifyNode(getNode(pathMarker));
	}
}

static void cleanUpRedundancy()
{
	PassageMarker *slowMarker = getNextInSequence(slowPath);
	PassageMarker *fastMarker = getNextInSequence(fastPath);
	Coordinate slowLength, fastLength, fastHandicap;
	Node *slowNode, *fastNode;
	Coordinate offset;

	//puts("Correcting new redundancy");
	mapSlowOntoFast();
	mapDistancesOntoPaths();

	fastHandicap = 0;
	while (slowMarker != NULL && fastMarker != NULL) {
		modifCounter++;

		slowLength = getPassageMarkerFinish(slowMarker);
		fastLength = getPassageMarkerFinish(fastMarker);
		slowNode = getNode(slowMarker);
		fastNode = getNode(fastMarker);
		//printf("Slow %li\tFast %li (%li)\n", slowLength, fastLength, fastLength + fastHandicap);

		if (slowNode == fastNode) {
			//printf("0/ Already merged together %li == %li\n", getNodeID(slowNode), getNodeID(fastNode)); 
			fastHandicap = slowLength - fastLength;
			slowMarker = getNextInSequence(slowMarker);
			fastMarker = getNextInSequence(fastMarker);
		} else if (slowNode == getTwinNode(fastNode)) {
			//printf("1/ Creme de la hairpin %li $$ %li\n", getNodeID(slowNode), getNodeID(fastNode)); 
			fastHandicap = slowLength - fastLength;
			slowMarker = getNextInSequence(slowMarker);
			fastMarker = getNextInSequence(fastMarker);
			foldSymmetricalNode(fastNode);
		} else if (markerLeadsToNode(slowMarker, fastNode)) {
			//printf("2/ Remapping empty fast arc onto slow nodes\n");
			reduceSlowNodes(slowMarker, fastNode);
			remapEmptyPathOntoMiddlePath(fastMarker,
						     slowMarker);
			while (getNode(slowMarker) != fastNode)
				slowMarker = getNextInSequence(slowMarker);
		} else if (markerLeadsToNode(fastMarker, slowNode)) {
			//printf("3/ Remapping empty slow arc onto fast nodes\n");
			remapEmptyPathOntoMiddlePath(slowMarker,
						     fastMarker);
			while (getNode(fastMarker) != slowNode)
				fastMarker = getNextInSequence(fastMarker);
		}

		else if (slowLength == fastLength + fastHandicap) {
			//printf("A/ Mapped identical nodes together %li <=> %li\n", getNodeID(slowNode), getNodeID(fastNode)); 
			remapNodeOntoNeighbour(slowNode, fastNode);
			slowMarker = getNextInSequence(slowMarker);
			fastMarker = getNextInSequence(fastMarker);
		} else if (slowLength < fastLength + fastHandicap) {
			//printf("B/ Mapped back of fast node into slow %li -> %li\n", getNodeID(fastNode), getNodeID(slowNode));         
			offset = fastLength + fastHandicap - slowLength;
			offset *= getNodeLength(fastNode);
			offset /= getPassageMarkerLength(fastMarker);

			// DEBUG
			if (offset > getNodeLength(fastNode)) {
				puts("Oversized offset, you screwed up mate!");
				abort();
			}
			// END OF DEBUG

			splitNodeDescriptor(fastNode, slowNode, offset);
			remapBackOfNodeOntoNeighbour(fastNode, slowNode,
						     getPassageMarkerLength
						     (fastMarker));

			slowMarker = getNextInSequence(slowMarker);
		} else {
			//printf("C/ Mapped back of slow node into fast %li -> %li\n", getNodeID(slowNode), getNodeID(fastNode));         
			remapBackOfNodeOntoNeighbour(slowNode, fastNode,
						     getPassageMarkerLength
						     (slowMarker));

			fastMarker = getNextInSequence(fastMarker);
		}
	}

	while (!isInitial(slowPath))
		slowPath = getPreviousInSequence(slowPath);
	while (!isInitial(fastPath))
		fastPath = getPreviousInSequence(fastPath);

	// Freeing up memory  
	if (slowMarker != NULL)
		concatenatePathNodes(slowPath);
	else
		concatenatePathNodes(fastPath);

	destroyPaths();

	// Cleaning up silly structures
	vaccinatePath();
}

static void comparePaths(Node * destination, Node * origin)
{
	IDnum slowLength, fastLength;
	Node *fastNode, *slowNode;
	IDnum i;
	PassageMarker *marker;

	btCounter++;

	//Measure lengths
	slowLength = fastLength = 0;
	fastNode = destination;
	slowNode = origin;

	while (fastNode != slowNode) {
		if (getNodeTime(fastNode) > getNodeTime(slowNode)) {
			fastLength++;
			fastNode = getNodePrevious(fastNode);
			btLengthCounter++;
		} else if (getNodeTime(fastNode) < getNodeTime(slowNode)) {
			slowLength++;
			slowNode = getNodePrevious(slowNode);
			btLengthCounter++;
		} else if (isPreviousToNode(slowNode, fastNode)) {
			while (fastNode != slowNode) {
				fastLength++;
				fastNode = getNodePrevious(fastNode);
				btLengthCounter++;
			}
		} else if (isPreviousToNode(fastNode, slowNode)) {
			while (slowNode != fastNode) {
				slowLength++;
				slowNode = getNodePrevious(slowNode);
				btLengthCounter++;
			}
		} else {
			fastLength++;
			fastNode = getNodePrevious(fastNode);
			slowLength++;
			slowNode = getNodePrevious(slowNode);
			btLengthCounter += 2;
		}

		if (slowLength > MAXNODELENGTH
		    || fastLength > MAXNODELENGTH)
			return;
	}

	if (fastLength == 0)
		return;

	//Backtracking to record actual paths
	fastPath = addUncertainPassageMarker(1, destination);
	setPassageMarkerStatus(fastPath, true);

	for (i = 0; i < fastLength; i++) {
		marker =
		    addUncertainPassageMarker(1,
					      getNodePrevious(getNode
							      (fastPath)));
		setPassageMarkerStatus(marker, true);
		connectPassageMarkers(marker, fastPath, graph);
		fastPath = marker;
	}

	slowPath = addUncertainPassageMarker(2, destination);
	setPassageMarkerStatus(slowPath, true);

	marker = addUncertainPassageMarker(2, origin);
	setPassageMarkerStatus(marker, true);
	connectPassageMarkers(marker, slowPath, graph);
	slowPath = marker;

	for (i = 0; i < slowLength; i++) {
		marker =
		    addUncertainPassageMarker(2,
					      getNodePrevious(getNode
							      (slowPath)));
		setPassageMarkerStatus(marker, true);
		connectPassageMarkers(marker, slowPath, graph);
		slowPath = marker;
	}

	//Extract sequences
	if (!extractSequence(fastPath, fastSequence)
	    || !extractSequence(slowPath, slowSequence)) {
		destroyPaths();
		return;
	}
	//Compare sequences
	if (compareSequences(fastSequence, slowSequence)) {
		cleanUpRedundancy();
		return;
	}
	//puts("\tFinished comparing paths, changes made");
	destroyPaths();
}

static void tourBusArc(Node * origin, Arc * arc, Time originTime)
{
	Node *destination = getDestination(arc);
	Time arcTime, totalTime, destinationTime;
	IDnum nodeIndex = getNodeID(destination) + nodeCount(graph);
	Node *oldPrevious = previous[nodeIndex];

	if (oldPrevious == origin || getNodeStatus(destination) == 1)
		return;

	arcCounter++;

	arcTime =
	    ((Time) getNodeLength(origin)) / ((Time) getMultiplicity(arc));
	totalTime = originTime + arcTime;

	destinationTime = times[nodeIndex];

	if (destinationTime == -1) {
		//printf("New node %li %f %li\n", getNodeID(destination),
		//totalTime, getMultiplicity(arc));

		setNodeTime(destination, totalTime);
		dheapNodes[nodeIndex] =
		    insertNodeIntoDHeap(dheap, totalTime, destination);
		previous[nodeIndex] = origin;
		return;
	} else if (destinationTime > totalTime) {
		//printf("Old node with slower path %li %li %p %f %f\n",
		//getNodeID(destination),
		//getNodeID(previous[nodeIndex]),
		//dheapNodes[nodeIndex], times[nodeIndex], totalTime);

		if (dheapNodes[nodeIndex] == NULL) {
			//puts("Already expanded though");
			return;
		}

		replaceCounter++;
		setNodeTime(destination, totalTime);
		replaceKeyInDHeap(dheap, dheapNodes[nodeIndex], totalTime);
		previous[nodeIndex] = origin;

		comparePaths(destination, oldPrevious);
		return;
	} else {
		//printf("Old node with better path %li, %li %p %f\n",
		//getNodeID(destination),
		//getNodeID(previous[nodeIndex]),
		//dheapNodes[nodeIndex], times[nodeIndex]);

		if (destinationTime == getNodeTime(origin)
		    && isPreviousToNode(destination, origin))
			return;

		comparePaths(destination, origin);
	}
}

static void initializeTodoLists()
{
	IDnum index;
	Node *node;
	Arc *arc;
	Ticket *tkt;
	IDnum nodes = nodeCount(graph);
	Ticket **currentList;
	Ticket *currentTicket, *tmp;
	Node *destination;

	puts("Initializing todo lists");

	for (index = 1; index <= nodes; index++) {
		node = getNodeInGraph(graph, index);
		currentList = &todoLists[index + nodes];
		*currentList = NULL;

		for (arc = getArc(node); arc != NULL;
		     arc = getNextArc(arc)) {
			destination = getDestination(arc);

			if (destination == node)
				continue;

			tkt = newTicket();
			tkt->id_a = getNodeID(destination);

			currentTicket = *currentList;
			if (currentTicket == NULL
			    || currentTicket->id_a > tkt->id_a) {
				tkt->next = currentTicket;
				*currentList = tkt;
				continue;
			}

			while (currentTicket->next != NULL
			       && currentTicket->next->id_a < tkt->id_a)
				currentTicket = currentTicket->next;

			tmp = currentTicket->next;
			currentTicket->next = tkt;
			tkt->next = tmp;
		}
	}

	for (index = -1; index >= -nodes; index--) {
		node = getNodeInGraph(graph, index);
		currentList = &todoLists[index + nodes];
		*currentList = NULL;

		for (arc = getArc(node); arc != NULL;
		     arc = getNextArc(arc)) {
			destination = getDestination(arc);

			if (destination == node)
				continue;

			tkt = newTicket();
			tkt->id_a = getNodeID(destination);

			currentTicket = *currentList;
			if (currentTicket == NULL
			    || currentTicket->id_a > tkt->id_a) {
				tkt->next = currentTicket;
				*currentList = tkt;
				continue;
			}

			while (currentTicket->next != NULL
			       && currentTicket->next->id_a < tkt->id_a)
				currentTicket = currentTicket->next;

			tmp = currentTicket->next;
			currentTicket->next = tkt;
			tkt->next = tmp;
		}
	}

	puts("Done with initilization");
}

static void tourBusNode(Node * node)
{
	Arc *arc;
	Node *destination;
	Time nodeTime = getNodeTime(node);
	IDnum id;

	//printf("Node %li %f %i %p\n", getNodeID(node),
	//       times[getNodeID(node) + nodeCount(graph)], simpleArcCount(node),
	//       node);

	dbgCounter++;
	if (dbgCounter % 1000 == 0) {
		printf
		    ("%li nodes visited %li %li %li %li %li %li %li %li %li %li\n",
		     dbgCounter, clock(), modifCounter, arcCounter,
		     btCounter, btLengthCounter, swCounter,
		     swLengthCounter, getDFibHeapSize(dheap),
		     replaceCounter, vaccinateCounter);
		fflush(stdout);
	}

	setSingleNodeStatus(node, 2);
	activeNode = node;
	todo = &todoLists[getNodeID(activeNode) + nodeCount(graph)];
	done = NULL;

	while ((id = nextTodoTicket()) != 0) {
		destination = getNodeInGraph(graph, id);

		// Node doesn't exist anymore
		if (destination == NULL)
			continue;

		arc = getArcBetweenNodes(activeNode, destination, graph);

		// Arc does not exist for some reason (?)
		if (arc == NULL)
			continue;

		tourBusArc(activeNode, arc, nodeTime);
	}

	freeDoneTickets();
}

static Coordinate getTipLength(Node * node)
{
	Node *current = getTwinNode(node);
	Coordinate length = 0;

	if (simpleArcCount(current) > 1)
		return getNodeLength(node);

	while (current != NULL && simpleArcCount(getTwinNode(current)) < 2
	       && simpleArcCount(current) < 2) {
		length += getNodeLength(current);
		current = getDestination(getArc(current));
	}

	return length;
}

static boolean isMinorityChoice(Node * node)
{
	Node *current = getTwinNode(node);
	Arc *arc, *activeArc;
	IDnum mult = 0;

	// Finding first tangle
	while (current != NULL && simpleArcCount(getTwinNode(current)) < 2
	       && simpleArcCount(current) < 2) {
		activeArc = getArc(current);
		current = getDestination(activeArc);
	}

	// If isolated snippet:
	if (current == NULL)
		return true;

	// Joined tips  
	if (simpleArcCount(getTwinNode(current)) < 2)
		return false;

	// If unique event
	if (getMultiplicity(activeArc) == 1)
		return true;

	// Computing max arc
	for (arc = getArc(getTwinNode(current)); arc != NULL;
	     arc = getNextArc(arc))
		if (getMultiplicity(arc) > mult)
			mult = getMultiplicity(arc);

	// Testing for minority
	return mult >= getMultiplicity(activeArc);
}

void clipTips(Graph * graph)
{
	IDnum index;
	Node *current, *twin;
	boolean modified = true;
	int Wordlength = getWordLength(graph);

	puts("Clipping short tips off graph");

	while (modified) {
		modified = false;
		for (index = 1; index <= nodeCount(graph); index++) {
			current = getNodeInGraph(graph, index);

			if (current == NULL)
				continue;

			twin = getTwinNode(current);

			if (getArc(current) == NULL
			    && getTipLength(current) < 2 * Wordlength
			    && isMinorityChoice(current)) {
				destroyNode(current, graph);
				modified = true;
			} else if (getArc(twin) == NULL
				   && getTipLength(twin) < 2 * Wordlength
				   && isMinorityChoice(twin)) {
				destroyNode(twin, graph);
				modified = true;
			}
		}
	}

	concatenateGraph(graph);
	printf("%li nodes left\n", nodeCount(graph));
}

void clipTipsHard(Graph * graph)
{
	IDnum index;
	Node *current, *twin;
	boolean modified = true;
	int Wordlength = getWordLength(graph);

	puts("Clipping short tips off graph");

	while (modified) {
		modified = false;
		for (index = 1; index <= nodeCount(graph); index++) {
			current = getNodeInGraph(graph, index);

			if (current == NULL)
				continue;

			twin = getTwinNode(current);

			if (getArc(current) == NULL
			    && getTipLength(current) < 2 * Wordlength) {
				//printf("Clipping current %li of length %li\n", getNodeID(current), getNodeLength(current));   
				destroyNode(current, graph);
				modified = true;
			} else if (getArc(twin) == NULL
				   && getTipLength(twin) <
				   2 * Wordlength) {
				//printf("Clipping twin %li of length %li\n", getNodeID(twin), getNodeLength(twin));    
				destroyNode(twin, graph);
				modified = true;
			}
		}
	}

	concatenateGraph(graph);
	printf("%li nodes left\n", nodeCount(graph));
}

static void tourBus(Node * startingPoint)
{
	Node *currentNode = startingPoint;
	IDnum nodeID = getNodeID(startingPoint) + nodeCount(graph);

	printf("Tour bus from node %li...\n", getNodeID(startingPoint));

	times[nodeID] = 0;
	previous[nodeID] = currentNode;

	while (currentNode != NULL) {
		dheapNodes[getNodeID(currentNode) + nodeCount(graph)] =
		    NULL;
		tourBusNode(currentNode);
		currentNode = removeNextNodeFromDHeap(dheap);
	}
}

void correctGraph(Graph * argGraph, double argCutoff)
{
	IDnum nodes;
	IDnum index;

	//Setting global params
	graph = argGraph;
	cutoff = argCutoff;
	WORDLENGTH = getWordLength(graph);
	dbgCounter = 0;
	modifCounter = 0;
	arcCounter = 0;
	btCounter = 0;
	btLengthCounter = 0;
	replaceCounter = 0;
	vaccinateCounter = 0;
	// Done with global params

	printf("Correcting graph with cutoff %f\n", cutoff);

	clipTips(graph);
	nodes = nodeCount(graph);

	// Allocating memory
	times = malloc((2 * nodes + 1) * sizeof(Time));
	previous = malloc((2 * nodes + 1) * sizeof(Node *));

	dheapNodes = malloc((2 * nodes + 1) * sizeof(DFibHeapNode *));

	dheap = newDFibHeap();

	fastSequence = newTightString(MAXREADLENGTH);
	slowSequence = newTightString(MAXREADLENGTH);

	eligibleStartingPoints = malloc((2 * nodes + 1) * sizeof(IDnum));

	for (index = 0; index < (2 * nodeCount(graph) + 1); index++) {
		times[index] = -1;
		dheapNodes[index] = NULL;
		previous[index] = NULL;
	}

	progressStatus = calloc((2 * nodes + 1), sizeof(boolean));
	todoLists = calloc((2 * nodes + 1), sizeof(Ticket *));
	//Done with memory 

	resetNodeStatus(graph);
	determineEligibleStartingPoints();
	initializeTodoLists();
	activateArcLookupTable(graph);

	while (true) {
		puts("Going through the cycle...");

		startingNode = nextStartingPoint();

		if (startingNode == NULL)
			break;

		tourBus(startingNode);
		updateNodeStatus(startingNode);
	}

	deactivateArcLookupTable(graph);
	concatenateGraph(graph);

	clipTipsHard(graph);

	//Deallocating globals
	free(times);
	free(previous);
	free(dheapNodes);
	destroyDHeap(dheap);

	free(fastSequence);
	free(slowSequence);

	free(progressStatus);
	free(todoLists);
	//Done deallocating
}
