// Copyright (c) 2004 The Broad Institute at MIT and Harvard

#include "util/Kronos.h"

// Kronos: a utility try to kill all your children
//
// Like its namesake, this class might occasionally fail to find all 
// children.  With processes rhea-->rock-->zeus, if rock exits then
// the ppid of zeus is reassigned to init, so kronos won't kill it.
//
// The process tree may change between our reading it and sending
// kill commands.  It is possible -- but vanishingly improbable --
// that we will accidentally kill an unrelated newborn process with
// coincidentally the same pid as a just-exited child.

// Usage:  in your signal handler, put the lines
//   ProcTree kronos;
//   kill(kronos.my_descendants(), <signal>);


// ProcTree class:
// Upon instantiation, this class uses the /proc filesystem to
// get the PID and PPID of all running processes.
// The details of /proc are platform-dependent, of course.
//
// Methods allow you to find all the children and all the descendents
// of a particular process.


// need to filter out non-PID files in /proc
// Overload versions with and without const, for linux vs alpha.
int is_pid_file(const struct dirent *entry) {
  for( const char *i=entry->d_name; *i != 0; i++ )
    if (!isdigit(*i)) return(0);
  return(1);
}
int is_pid_file(      struct dirent *entry) {
  for( const char *i=entry->d_name; *i != 0; i++ )
    if (!isdigit(*i)) return(0);
  return(1);
}


// Load_PPID_Data() :  Read the /proc filesystem and build the
// ppid_map and children_multimap structures.

void ProcTree::Load_PPID_Data( ) {
  pid_t pid, ppid;
  struct dirent **files;
  int n = scandir( "/proc", &files, &is_pid_file, NULL );
  if( n <= 0 )
    cerr << "Error reading /proc; might not kill child processes" << endl;
  while(n--) {
    pid=ppid=0;  // on other architectures, do no harm

#ifdef __alpha
    String proc_filename = "/proc/";
    proc_filename += files[n]->d_name;
    int fd=open(proc_filename.c_str(),O_WRONLY);
    struct prstatus p;
    ioctl(fd, PIOCSTATUS, &p);
    close(fd);
    pid = p.pr_pid;
    ppid = p.pr_ppid;
#endif

#ifdef __ia64__
    String proc_filename = "/proc/";
    proc_filename += files[n]->d_name;
    proc_filename += "/stat";
    ifstream proc_file(proc_filename.c_str());
    String s; char c;
    proc_file >> pid >> s >> c >> ppid;  //throw away s,c.
#endif

    ppid_map[pid] = ppid;
    children_multimap.insert(make_pair(ppid,pid));
    free(files[n]);
  }
  free(files);
}


// Returns values of children_multimap.equal_range(pid) as a vec<pid_t>
vec<pid_t> ProcTree::children(pid_t pid) {
  vec<pid_t> kids;
  for( multimap<pid_t,pid_t>::iterator i = children_multimap.lower_bound(pid); 
       i->first == pid; i++ )
    kids.push_back(i->second);
  return(kids);
}


// Finds all descendants, by breath-first search using children().
// If the children map is corrupted to contain a loop, this will run forever.
vec<pid_t> ProcTree::descendants(pid_t pid) {
  vec<pid_t> progeny = children(pid);
  for( unsigned int i = 0; i != progeny.size(); i++ ) {
    vec<pid_t> ikids = children(progeny[i]);
    progeny.insert(progeny.end(), ikids.begin(), ikids.end());
  }
  return(progeny);
}


// Overload the system call "kill" to take a vec<pid_t> of PIDs.
// Returns 0 for all success, -n for n errors, but signalling a
// nonexistent process shouldn't be an error -- maybe it exited
// cleanly after we loaded the ppid data but before the kill.

int kill(vec<pid_t> pids, int sig)
{
  int fails=0;

  for(vec<pid_t>::iterator i=pids.begin(); i != pids.end(); i++)
    if( kill(*i, sig) == -1 && errno != ESRCH ) 
      fails++;
  return(-fails);
}
