/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2007) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include <unistd.h>
#include <setjmp.h>
#include <signal.h>
#include <fstream>
#include <strstream>

#ifndef FatalErr
     #define FatalErr(message) { cout << message << endl << endl; exit(-1); }
#endif

#include <iostream>
#include <iomanip>

#include <sys/file.h> //for flock

#include "system/ParsedArgs.h"
#include "system/System.h"

const Bool         parsed_args::INVALID_BOOL = 2;
const int          parsed_args::INVALID_INT = INT_MIN+1;
const unsigned int parsed_args::INVALID_UINT = (unsigned int) -1 ;
const double       parsed_args::INVALID_DOUBLE = -1.1234;
const String       parsed_args::INVALID_STRING = "<required>";

Bool parsed_args::pretty_help;
String parsed_args::usage_format;  // usage line format
String parsed_args::param_format;  // parameter name format
String parsed_args::header_format; // header line format
String parsed_args::info_format;   // paramter information format
String parsed_args::doc_format;    // paramter documentation format

void parsed_args::PrintTheCommandPretty( ostream& out, String make_date, 
     const String& prefix )
{      String bar( "-----------------------------------------------------------"
		   "---------------------" );
       String leftmarg = "";
       out << prefix << "\n" << prefix << bar << "\n" << prefix << leftmarg 
          << Date( ) << " run";
       String md(make_date);
       if ( make_date != "" ) out << ", based on " 
            << md.After( "\"" ).Before( "\"" ) << " make";
       out << "\n";
       if ( orig_command_ != "" )
       {     String oc;
             SlashFold( orig_command_, oc );
             out << prefix << oc;
             out << "........................................................."
                  << ".......................\n";    }
       String tc;
       SlashFold( TheCommand( ), tc );
       if ( prefix != "" ) 
       {    tc.GlobalReplaceBy( "\n", prefix + "\n" );
            tc.resize( tc.size( ) - prefix.size( ) - 1 );
            tc += "\n";    }
       out << prefix << tc << prefix << bar << "\n" << prefix << endl;    }

static jmp_buf logTheCommandEnv;
static Bool inLogging = False;

static void abortLogging(int) {
  if ( inLogging )
    longjmp( logTheCommandEnv, 1 );
}

parsed_args::parsed_args( int argc, char *argv[], String make_date )
       : command_( Basename(argv[0]) ), doc_(0)
{    
     char* pretty_help_ptr = getenv( "ARACHNE_PRETTY_HELP" );
     if (pretty_help_ptr != 0 && String(pretty_help_ptr) == "Color") {
       usage_format = START_BOLD;
       param_format = START_MAGENTA;
       header_format = START_BOLD;
       info_format = START_LIGHT_BLUE;
       doc_format = "";
       pretty_help = True;
     } else if (pretty_help_ptr != 0 && String(pretty_help_ptr) == "Bold") {
       usage_format = START_BOLD;
       param_format = START_BOLD;
       header_format = START_BOLD;
       info_format = "";
       doc_format = "";
       pretty_help = True;
     } else {
       pretty_help = False;
     }
     name_.reserve( argc - 1 );
     def_.reserve( argc - 1 );
     used_.reserve( argc - 1 );
     if ( argc == 1 || ( argc > 1 && strcmp( argv[1], "-h" ) == 0 ) )
          get_help_ = True;
     else
       {    static Bool included_args;  // static because of longjmp below
       included_args = False;
          get_help_ = False;
          for ( int i = 1; i < argc; i++ )
          {    if ( String(argv[i]).Contains( "@@", 0 ) )
               {    included_args = True;
                    String source = String(argv[i]).After( "@@" );
                    if ( !IsRegularFile(source) )
                    {     PRINT2(i, argv[i]);
                          InputErr( "Failed to locate " << source << "." );     }
                    Ifstream( in, source );
                    String line, x, inargs;
                    while(in)
                    {    getline( in, line );
                         if ( !in ) break;
                         if ( line.Contains( "#", 0 ) ) continue;
                         istrstream ini( line.c_str( ) );
                         while(ini)
                         {    ini >> x;
                              if ( x == "" ) break;
                              if ( !ParseString(x) )
                              {    PRINT2(i, argv[i]);
	                           InputErr( "Could not parse argument " << x 
                                        << " from file " << source 
                                        << "." );    }    }    }    }
               else if ( ! this->ParseString( argc, argv, i) )
               {    PRINT2(i, argv[i]);
	            InputErr( "You did not invoke this program with " <<
		          "white-space-free arguments of the form PARAMETER=VALUE." 
		          << "\n\nTo see the syntax for this command, type \"" 
		          << command_ << " -h\"." );    }    }

	  if ( getenv("ARACHNE_COMMAND_LOG_DIR") ) {
	    sighandler_t prevSignal = signal( SIGALRM, abortLogging );
	    if ( setjmp( logTheCommandEnv ) == 0 ) {
	      alarm( 15 );
	      inLogging = True;
	      LogTheCommand();
	      inLogging = False;
	    }
	    alarm( 0 );
	    signal( SIGALRM, prevSignal );

	  }
          // LogTheCommand(); 

          // Print the command name, etc.
     
          if ( TheCommand( ).Contains( "Assemble ", 0 ) ) return;
          if ( TheCommand( ).Contains( "PackSource ", 0 ) ) return;
          if ( TheCommand( ).Contains( " NO_HEADER=True" ) ) return;
          if ( TheCommand( ).Contains( " NH=True" ) ) return;
          if (included_args)
          {    for ( int i = 0; i < argc; i++ )
               {    if ( i > 0 ) orig_command_ += " ";
                    orig_command_ += argv[i];    }    }
          PrintTheCommandPretty( cout, make_date );    }    }


void parsed_args::LogTheCommand( ) {
  if ( getenv("ARACHNE_COMMAND_LOG_DIR") ) 
    LogTheCommand( getenv("ARACHNE_COMMAND_LOG_DIR") );
}

void parsed_args::LogTheCommand( String  filename ) {
  // Find the year and month.
  vec<String> date;
  Tokenize(Date(), date);
  ForceAssertEq(date.size(), 5);
  filename = filename + "." + (getenv("USER") ? String(getenv("USER")) + "." : String("")) + date[1] + "." + date[4];

  int fd=open(filename.c_str(), O_WRONLY | O_APPEND | O_CREAT, 0666);
  if (-1 == fd) return;

  struct flock lock;
  lock.l_type = F_WRLCK;
  lock.l_start = 0;
  lock.l_whence = SEEK_SET;
  lock.l_len = 0;

  String log ;
  {    char nowstr[80];
     time_t nowbin;
     const struct tm *nowstruct;
     static bool locale_has_been_set = false;
     if( ! locale_has_been_set ) {
       (void)setlocale(LC_ALL, "");
       locale_has_been_set = true;
     }
     if (time(&nowbin) == (time_t) - 1) log = "(date unavailable - time failed)";
     else {
       nowstruct = localtime(&nowbin);
       if (strftime(nowstr, 80, "%m-%d %H:%M:%S", nowstruct) == (size_t) 0)
	 log = "(date unavailable - strftime failed)";
       else
       log = nowstr;
       
     }
  }
  

  log = log + "\t" + ToString(getpid()) + "\t" + ( getenv("ARACHNE_PARENT_PID") ? String(getenv("ARACHNE_PARENT_PID")) : ToString(getppid()) );
  
  log = log + "\t" + (getenv("USER") ? getenv("USER") : "unknown") + "\t" + TheCommand() + "\n";
  const char * data = log.c_str();
  
  // Attempt to advisory-lock the command-log file, waiting if necessary.  
  if (-1 != fcntl(fd, F_SETLKW, &lock)) {
    // If successful, write command and unlock. 
    write(fd, data, log.size());
    lock.l_type = F_UNLCK;
    fcntl(fd, F_SETLKW, &lock);
  }
  
  close(fd);
}

bool parsed_args::ParseString( int argc, char **argv, int & i) {
  int len = strlen(argv[i]);
  if (i == argc-1 
      || argv[i][len-1] != '=' //e.g. PARAM=blah
      || String(argv[i+1]).Contains("=") ) { 
    return ParseString(argv[i]);
  }
  else { //e.g. PARAM= blah
    String temp(argv[i]);
    temp += argv[i+1];
    ++i; //consume the next argument!
    return ParseString(temp);
  }
}

void parsed_args::FetchArgsFromFile( const String& filename )
{    if ( IsRegularFile(filename) )
     {    Ifstream( in, filename );
          String line, a;
          while(1)
          {    getline( in, line );
               if ( !in ) break;
               if ( line.Contains( "#", 0 ) ) continue;
               if ( line.size( ) == 0 ) continue;
               istrstream iline( line.c_str( ) );
               while(iline)
               {    iline >> a;
                    if ( a.empty( ) ) break;
                    if ( !this->ParseString(a) )
	                 InputErr( "\nThe file " << filename << "\ndoes not contain "
		              << "white-space-free arguments of the form "
		              << "PARAMETER=VALUE." );    }    }    }    }

bool parsed_args::ParseString( const String& string )
{
  int eq = string.Position( "=" );
  if ( eq <= 0 ) 
    return false;

  String name = string.substr( 0, eq );
  if ( !Member( name_, name ) )
  {
    def_.push_back( string.substr( eq+1, string.size( ) ) );
    name_.push_back( name );
    used_.push_back(False);
  }

  return true;
}
  

Bool parsed_args::GetHelpOnly()
{
  return get_help_;
}

void parsed_args::ConversionFailure(int i)
{    cout << "I couldn't convert the command line value \"" << def_[i] <<
          "\" for argument \"" << name_[i] << "\" to the required type.\n";
     System( command_ + " -h" );
     cout << "Aborting.\n\n";
     exit(1);    }

void parsed_args::NoValue(String n)
{    cout << "You must specify a command-line value for " << n << ".\n";
     System( command_ + " -h" );
     cout << "Aborting.\n\n";
     exit(1);    }

String parsed_args::CheckForArgAndAbbreviation( String n, String abbr )
{
  if ( abbr == "" )
    return n;

  bool gotName(False), gotAbbr(False);
  for ( unsigned int i = 0; i < name_.size( ); i++ )
     {
       if ( name_[i] == n )
	 {
	   gotName = True;
	   continue;
	 }
       if ( name_[i] == abbr )
	 {
	   gotAbbr = True;
	 }
     }

  if ( gotName && gotAbbr )
    {
      cout << "You must not specify both an argument-" << n 
	   << " and its abbreviation-" << abbr << ".\n"
	   << "\nTo see the syntax for this command, type \"" 
	   << command_ << " -h\".\n\n";
     exit(1);    
    }

  if ( gotName )
    return n;
  
  return abbr;

}   

void parsed_args::CheckEnv( const String& n, const String& abbr )
{    for ( int i = 0; i < name_.isize( ); i++ )
          if ( name_[i] == n || name_[i] == abbr ) return;
     int npasses = ( abbr == "" ? 1 : 2 );
     for ( int pass = 1; pass <= npasses; pass++ )
     {    String env_var = "ARACHNE_" + ( pass == 1 ? n : abbr );
          char* env = getenv( env_var.c_str( ) );
          if ( env != 0 )
          {    def_.push_back( String(env) );
               name_.push_back(n);
               used_.push_back(False);    }    }    }

template<>
void parsed_args::GetValue<String>
  ( String n, String & value, const String & abbr, const String & deflt ) {
    value = GetStringValue(n, abbr,deflt); 
}
    
template<>
void parsed_args::GetValue<vec<String> >
  ( String n, vec<String> & value, const String & abbr, const String & deflt ) {
    ParseStringSet(GetStringValue(n, abbr,deflt), value, false); 
}
    
template<>
void parsed_args::GetValue<vec<int> >
  ( String n, vec<int> & value, const String & abbr, const String & deflt ) {
    ParseIntSet(GetStringValue(n, abbr,deflt), value, false); 
}
    
template<>
void parsed_args::GetValue<vec<double> >
  ( String n, vec<double> & value, const String & abbr, const String & deflt ) {
    ParseDoubleSet(GetStringValue(n, abbr,deflt), value, false); 
}

template<>
void parsed_args::GetValue<unsigned int>
( String n, unsigned int & value, const String & abbr, const String & deflt ) {
  value = deflt.IsInt() 
    ? GetUnsignedIntValue(n, abbr, deflt.Int())
    : GetUnsignedIntValue(n, abbr);
  return;
}

template<>
void parsed_args::GetValue<int>
( String n, int & value, const String & abbr, const String & deflt ) {
  value = deflt.IsInt() 
    ? GetIntValue(n, abbr, deflt.Int())
    : GetIntValue(n, abbr);
  return;
}

template<>
void parsed_args::GetValue<double>
( String n, double & value, const String & abbr, const String & deflt ) {
  value = deflt.IsDouble() 
    ? GetDoubleValue(n, abbr, deflt.Double())
    : GetDoubleValue(n, abbr);
  return;
}

String parsed_args::GetStringValue( String n, String abbr, String deflt  ) { 
  CheckEnv( n, abbr );
  String toFind = CheckForArgAndAbbreviation( n, abbr );
  for ( unsigned int i = 0; i < name_.size( ); i++ )
    if ( name_[i] == toFind ) 
      {    used_[i] = True;
      if ( def_[i].size( ) > 0 && def_[i][0] == '"' 
	   && def_[i][ def_.size( ) - 1 ] == '"' )
	{    for ( unsigned int j = 1; j < def_.size( ) - 1; j++ )
	  def_[j-1] = def_[j];
	def_.resize( def_.size( ) - 2 );    }
      return def_[i];    }
  if ( deflt != INVALID_STRING ) return deflt;
  NoValue(n);
  return GARBAGE;    
}

 


Bool parsed_args::GetBoolValue( String n, String abbr, Bool deflt )
{ 
  CheckEnv( n, abbr );
  String toFind = CheckForArgAndAbbreviation( n, abbr );
  for ( unsigned int i = 0; i < name_.size( ); i++ )
          if ( name_[i] == toFind ) 
          {    used_[i] = True;
               if ( def_[i] == "True" ) return True;
               else if ( def_[i] == "False" ) return False;
               else FatalErr( "The value of " << n 
                    << " must be either True or False." );    }
     if ( deflt != INVALID_BOOL ) return deflt;
     NoValue(n);
     return GARBAGE;    
}

template<>
void parsed_args::GetValue<Bool>
  ( String n, Bool & value, const String & abbr, const String & deflt ) {
    value = (!deflt.IsBool()) ? GetBoolValue(n, abbr) 
    : ("True" == deflt) 
      ?  GetBoolValue(n, abbr, True)
      :  GetBoolValue(n, abbr, False);
    return;
  }


int parsed_args::GetUnsignedIntValue( String n, String abbr, unsigned int deflt )
{    CheckEnv( n, abbr );
  String toFind = CheckForArgAndAbbreviation( n, abbr );
  for ( int i = 0; i < name_.isize( ); i++ )
  {    if ( name_[i] == toFind ) 
    {    if ( def_[i].size( ) > 0 && def_[i].IsInt( ) )
      {    longlong answer = def_[i].Int( );
	static longlong answer_bound(
	    (longlong) 4 * (longlong) (1024 * 1024 * 1024) );
	if ( answer < answer_bound )
                    {    used_[i] = True;
                         return (unsigned int) answer;    }    }
               ConversionFailure(i);    }    }
     if ( deflt != INVALID_UINT ) return deflt;
     NoValue(n);
     return 0;    }



int parsed_args::GetIntValue( String n, String abbr, int deflt )
{    CheckEnv( n, abbr );
     String toFind = CheckForArgAndAbbreviation( n, abbr );
     for ( int i = 0; i < name_.isize( ); i++ )
     {    if ( name_[i] == toFind ) 
          {    if ( def_[i].size( ) > 0 && def_[i].IsInt( ) )
               {    longlong answer = def_[i].Int( );
                    static longlong answer_top(INT_MAX);
		    static longlong answer_bottom(INT_MIN);
                    if ( answer <= answer_top && answer >= answer_bottom )
                    {    used_[i] = True;
                         return (int) answer;    }    }
               ConversionFailure(i);    }    }
     if ( deflt != INVALID_INT ) return deflt;
     NoValue(n);
     return 0;    }

double parsed_args::GetDoubleValue( String n, String abbr, double deflt )
{  
  CheckEnv( n, abbr );
  String toFind = CheckForArgAndAbbreviation( n, abbr );
  for ( unsigned int i = 0; i < name_.size( ); i++ )
          if ( name_[i] == toFind ) 
          {    if ( def_[i].size( ) > 0 && def_[i].IsDouble( ) )
               {    double answer = def_[i].Double( );
                    used_[i] = True;
                    return answer;    }
               ConversionFailure(i);    }

     if ( deflt != INVALID_DOUBLE ) return deflt;
     NoValue(n);
     return GARBAGE;    }

extern Bool use_gdb_for_tracebacks;

void parsed_args::CheckForExtraArgs( )
{    Bool junk_found = False;
     for ( unsigned int i = 0; i < used_.size( ); i++ )
          if ( used_[i] == False )
          {    if ( name_[i] == "GDB" )
               {    if ( def_[i] == "True" ) use_gdb_for_tracebacks = True;
                    else if ( def_[i] != "False" )
                    {    cout << "illegal value for parameter GDB" << endl;
                         exit(1);    }
                    continue;    }
               if ( name_[i] == "NO_HEADER" )
               {    if ( def_[i] != "True" && def_[i] != "False" )
                    {    cout << "illegal value for parameter NO_HEADER" << endl;
                         exit(1);    }
                    continue;    }
               if ( name_[i] == "NH" )
               {    if ( def_[i] != "True" && def_[i] != "False" )
                    {    cout << "illegal value for parameter NH" << endl;
                         exit(1);    }
                    continue;    }
               cout << "You should not have specified a value for " <<
                    name_[i] << ".\n";
               junk_found = True;    }
     if (junk_found) 
       {
	 cout << "\nTo see the syntax for this command, type \"" << command_
	      << " -h\". \n" << endl;
	 exit(-1);    
       }
}

void parsed_args::AddArgHelp( const String& arg, 
			      const String& type, 
			      const String& abbreviation,
			      const String& default_value,
			      const String& documentation)
{
  arg_help_.push_back(parsed_arg_help
		      (arg, type, abbreviation, default_value, documentation));
}

void parsed_args::PrintArgHelp()
{
  if (parsed_args::pretty_help) cout << usage_format;
  cout << "\nUsage: " << command_ << " arg1=value1 arg2=value2 ..." << endl;
  if (parsed_args::pretty_help) cout << END_ESCAPE;
  if (doc_) {
    WhitespaceFold(String(doc_), cout);
    cout << "\n";
  }
  vec<parsed_arg_help>::iterator firstOpt =
    stable_partition(arg_help_.begin(), arg_help_.end(),
		     mem_fun_ref(&parsed_arg_help::isRequired));
  if (firstOpt != arg_help_.begin()) {
    if (parsed_args::pretty_help) cout << header_format;
    cout << "\nRequired arguments:\n\n";
    if (parsed_args::pretty_help) cout << END_ESCAPE;
    copy( arg_help_.begin(), firstOpt, 
	  ostream_iterator<parsed_arg_help>(cout, "\n") );
  }
  if (parsed_args::pretty_help) cout << header_format;
  cout << "\nOptional arguments:\n\n";
  if (parsed_args::pretty_help) cout << END_ESCAPE;
  copy( firstOpt, arg_help_.end(), 
	ostream_iterator<parsed_arg_help>(cout, "\n") );
  cout << parsed_arg_help( "GDB", "Bool", "", "False", "Whether to use GDB for tracebacks." )
       << "\n"
       << parsed_arg_help( "NO_HEADER", "Bool", "NH", "False",
			   "Whether to suppress the normal command-line header block." )
       << "\n";
  cout << endl;
}

ostream& operator<< (ostream& out, const parsed_arg_help& an_arg)
{
  // Output first line describing the argument briefly.
  if (parsed_args::pretty_help) out << parsed_args::param_format;
  out << an_arg.name_;
  if (parsed_args::pretty_help) out << END_ESCAPE;
  if ( !an_arg.abbreviation_.empty() )
  {    out << ", or ";
       if (parsed_args::pretty_help) out <<  parsed_args::param_format;
       out << an_arg.abbreviation_;
       if (parsed_args::pretty_help) out << END_ESCAPE;    }
  if (parsed_args::pretty_help) out <<  parsed_args::info_format;
  out << " (" << an_arg.type_ << ") ";
  if ( ! (an_arg.isRequired() || an_arg.default_.empty() ) )
    out << "default: " << an_arg.default_;
  if (parsed_args::pretty_help) out << END_ESCAPE;
  
  // Now fold and display the documentation line, if any.
  if (!an_arg.doc_.empty()) {
    if (parsed_args::pretty_help) out <<  parsed_args::doc_format;
    WhitespaceFold(an_arg.doc_, out, 70, "  ");
    if (parsed_args::pretty_help) out << END_ESCAPE;
  }
  return out;
}

String parsed_args::TheCommand( )
{    String com = command_;
     for ( unsigned int i = 0; i < name_.size( ); i++ )
     {    Bool need_quotes = False;
          const String& d = def_[i];
          for ( unsigned int j = 0; j < d.size( ); j++ )
          {    if ( isspace( d[j] ) ) need_quotes = True;
               if ( d[j] == '{' || d[j] == '}' ) need_quotes = True;
               if ( d[j] == '[' || d[j] == ']' ) need_quotes = True;
               if ( d[j] == '*' ) need_quotes = True;    }
          if ( !need_quotes ) com += " " + name_[i] + "=" + d;
          else com += " " + name_[i] + "=\"" + d + "\"";    }
     return com;    }

bool parsed_args::RemoveArg(String n, String abbr)
{  
  String toFind = CheckForArgAndAbbreviation( n, abbr );
  for ( int i = 0; i < name_.isize( ); i++ )
    if ( name_[i] == toFind ) {
      name_.erase(name_.begin() + i, name_.begin() + i+1);
      def_.erase(def_.begin() + i,  def_.begin() + i+1);
      used_.erase(used_.begin() + i,  used_.begin() + i+1);
      return true;
    }
  return false;
}
