/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef	ASSERT_H
#define ASSERT_H

#include <iostream>
#include "system/Types.h" //for longlong.
#include "system/StaticAssert.h"


/// The traceback produced by this function (which is also used by
/// Assert() and ForceAssert()) uses the GNU binutil addr2line to get
/// the function name from each stack frame.  The name of the
/// executable to be used for this purpose can be overridden by setting
/// the environment variable BACKTRACER to the name or path of the
/// executable you want to use.  This was intended to simplify the use
/// of backtraces on various platforms, as the calling convention
/// (i.e., the arguments and their order) is always that of addr2line.

void TracebackThisProcess( ostream& out = cout, Bool exit_when_done = True,
     Bool minimal = False );

///\file Assert.h
/// There are four versions of the Assert suite:
///
///      NDEBUG: treat Asserts as if they weren't there
///      SIMPLE_ASSERT: use the standard assert.
///      FAST_ASSERT: print assert message, back trace, 
///                   allows inlining of functions that contain asserts,
///                   slightly longer compile time.
///      the DEFAULT: print assert message, back trace,
///                   static variable prevents inlining of functions that contain 
///                   asserts.
///
/// The DEFAULT and FAST_ASSERT versions are only available (at present) under 
/// GNU C++.
///
/// Note: ForceAssert is a version of Assert which always asserts, even if NDEBUG
/// is turned on.

#ifndef __GNUC__
     #define SIMPLE_ASSERT
#endif

#include <assert.h>

#ifdef SIMPLE_ASSERT


// We need to do ForceAsserts the hard way if NDEBUG is on, as assert() won't work.

#ifdef NDEBUG
     extern "C" {
       extern void __assert( char*, char*, int );
     }

     #define ForceAssert(expr)  (((int) (expr)) ? (void)0 : __assert(#expr, __FILE__, __LINE__))
     #define ForceAssertEq(expr1, expr2) ForceAssert(expr1 == expr2)
     #define ForceAssertNe(expr1, expr2) ForceAssert(!(expr1 == expr2))
     #define ForceAssertLt(expr1, expr2) ForceAssert(expr1 <  expr2)
     #define ForceAssertGt(expr1, expr2) ForceAssert(expr1 >  expr2)
     #define ForceAssertLe(expr1, expr2) ForceAssert(expr1 <= expr2)
     #define ForceAssertGe(expr1, expr2) ForceAssert(expr1 >= expr2)
#else
     #define ForceAssert                 assert
     #define ForceAssertEq(expr1, expr2) assert(expr1 == expr2);
     #define ForceAssertNe(expr1, expr2) assert(!(expr1 == expr2));
     #define ForceAssertLt(expr1, expr2) assert(expr1 < expr2);
     #define ForceAssertGt(expr1, expr2) assert(expr1 > expr2);
     #define ForceAssertLe(expr1, expr2) assert(expr1 <= expr2);
     #define ForceAssertGe(expr1, expr2) assert(expr1 >= expr2);
#endif

     #define Assert                 assert
     #define AssertEq(expr1, expr2) assert(expr1 == expr2);
     #define AssertNe(expr1, expr2) assert(!(expr1 == expr2));
     #define AssertLt(expr1, expr2) assert(expr1 < expr2);
     #define AssertGt(expr1, expr2) assert(expr1 > expr2);
     #define AssertLe(expr1, expr2) assert(expr1 <= expr2);
     #define AssertGe(expr1, expr2) assert(expr1 >= expr2);

#else

#include <string>
using namespace std;

#ifdef FAST_ASSERT

void AssertBody( const char* expr_string, const char* function_name, 
		 const int line_num, const char* file_name );
void AssertEqBody( longlong, longlong, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name );
void AssertEqBody( const string&, const string&, 
		   const char* expr1_string, const char* expr2_string,
		   const char* function_name, const int line_num, 
                   const char* file_name );
void AssertNeBody( longlong, longlong, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name );
void AssertNeBody( const string&, const string&, 
		   const char* expr1_string, const char* expr2_string,
		   const char* function_name, const int line_num, 
                   const char* file_name );
void AssertLtBody( longlong, longlong, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name );
void AssertGtBody( longlong, longlong, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name );
void AssertLeBody( longlong, longlong, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name );
void AssertGeBody( longlong, longlong, const char* expr1_string, 
                   const char* expr2_string, const char* function_name, 
                   const int line_num, const char* file_name );

#define Assert(expr)                                                         \
{    if ( !(expr) ) AssertBody( #expr, __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }

#define AssertEq(expr1, expr2)                                               \
{    if ( !(expr1 == expr2) ) AssertEqBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }

#define AssertNe(expr1, expr2)                                               \
{    if ( expr1 == expr2 ) AssertNeBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }

#define AssertLt(expr1, expr2)                                               \
{    if ( !(expr1 < expr2) ) AssertLtBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }

#define AssertGt(expr1, expr2)                                               \
{    if ( expr1 <= expr2 ) AssertGtBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }

#define AssertLe(expr1, expr2)                                               \
{    if ( expr1 > expr2 )  AssertLeBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }
					 
#define AssertGe(expr1, expr2)                                               \
{    if ( expr1 < expr2 )  AssertGeBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }
					 
#define ForceAssert(expr)                                                    \
{    if ( !(expr) ) AssertBody( #expr, __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }

#define ForceAssertEq(expr1, expr2)                                          \
{    if ( !(expr1 == expr2) ) AssertEqBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }

#define ForceAssertNe(expr1, expr2)                                          \
{    if ( expr1 == expr2 ) AssertNeBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }

#define ForceAssertLt(expr1, expr2)                                          \
{    if ( !(expr1 < expr2) ) AssertLtBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }

#define ForceAssertGt(expr1, expr2)                                          \
{    if ( expr1 <= expr2 ) AssertGtBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }

#define ForceAssertLe(expr1, expr2)                                          \
{    if ( expr1 > expr2 )  AssertLeBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }
					 
#define ForceAssertGe(expr1, expr2)                                          \
{    if ( expr1 < expr2 )  AssertGeBody( expr1, expr2, #expr1, #expr2,       \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );    }
					 
#else // DEFAULT

#ifndef DEFAULT_ASSERT
#define DEFAULT_ASSERT
#endif

#include "system/AssertStructures.h"

void AssertBody( const assert_structure* );
void AssertEqBody( longlong, longlong, const assert_structure2* );
void AssertEqBody( const string&, const string&, const assert_structure2* );
void AssertNeBody( longlong, longlong, const assert_structure2* );
void AssertNeBody( const string&, const string&, const assert_structure2* );
void AssertLtBody( longlong, longlong, const assert_structure2* );
void AssertGtBody( longlong, longlong, const assert_structure2* );
void AssertLeBody( longlong, longlong, const assert_structure2* );
void AssertGeBody( longlong, longlong, const assert_structure2* );

     #define Assert(expr)                                                         \
          {    static assert_structure this_assert( #expr, __PRETTY_FUNCTION__,   \
                    __LINE__, __FILE__ );                                         \
               if ( !(expr) ) AssertBody( &this_assert );    }

     #define AssertEq(expr1, expr2)                                               \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( !(expr1 == expr2) ) AssertEqBody( expr1, expr2, &this_assert );    }

     #define AssertNe(expr1, expr2)                                               \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( expr1 == expr2 ) AssertNeBody( expr1, expr2, &this_assert );    }

     #define AssertLt(expr1, expr2)                                               \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( !(expr1 < expr2) ) AssertLtBody( expr1, expr2, &this_assert );    }

     #define AssertGt(expr1, expr2)                                               \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( expr1 <= expr2 ) AssertGtBody( expr1, expr2, &this_assert );    }

     #define AssertLe(expr1, expr2)                                               \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( expr1 > expr2 ) AssertLeBody( expr1, expr2, &this_assert );    }

     #define AssertGe(expr1, expr2)                                               \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( expr1 < expr2 ) AssertGeBody( expr1, expr2, &this_assert );    }


     #define ForceAssert(expr)                                                    \
          {    static assert_structure this_assert( #expr, __PRETTY_FUNCTION__,   \
                    __LINE__, __FILE__ );                                         \
               if ( !(expr) ) AssertBody( &this_assert );    }

     #define ForceAssertEq(expr1, expr2)                                          \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( !(expr1 == expr2) ) AssertEqBody( expr1, expr2, &this_assert );    }

     #define ForceAssertNe(expr1, expr2)                                          \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( expr1 == expr2 ) AssertNeBody( expr1, expr2, &this_assert );    }

     #define ForceAssertLt(expr1, expr2)                                          \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( !(expr1 < expr2) ) AssertLtBody( expr1, expr2, &this_assert );    }

     #define ForceAssertGt(expr1, expr2)                                          \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( expr1 <= expr2 ) AssertGtBody( expr1, expr2, &this_assert );    }

     #define ForceAssertLe(expr1, expr2)                                          \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( expr1 > expr2 ) AssertLeBody( expr1, expr2, &this_assert );    }

     #define ForceAssertGe(expr1, expr2)                                          \
          {    static assert_structure2 this_assert( #expr1, #expr2,              \
                    __PRETTY_FUNCTION__, __LINE__, __FILE__ );                    \
               if ( expr1 < expr2 ) AssertGeBody( expr1, expr2, &this_assert );    }
#endif //FAST_ASSERT

// If the NDEBUG flag is set, redefine all the non-forced asserts to be null functions.

// These redefines mess up SIMPLE_ASSERT, so we do that separately.

#ifdef NDEBUG

     #undef Assert
     #undef AssertEq
     #undef AssertNe
     #undef AssertLt
     #undef AssertGt
     #undef AssertLe
     #undef AssertGe

     #define Assert(expr)           ((void) 0)
     #define AssertEq(expr1, expr2) ((void) 0)
     #define AssertNe(expr1, expr2) ((void) 0)
     #define AssertLt(expr1, expr2) ((void) 0)
     #define AssertGt(expr1, expr2) ((void) 0)
     #define AssertLe(expr1, expr2) ((void) 0)
     #define AssertGe(expr1, expr2) ((void) 0)

#endif //NDEBUG

#endif //SIMPLE_ASSERT

#endif // ASSERT_H
